/*
 * Copyright 2004 by Kappich+Kniß Systemberatung, Aachen
 * Copyright 2007-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.sys.funclib.configObjectAcquisition.
 *
 * de.bsvrz.sys.funclib.configObjectAcquisition is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.sys.funclib.configObjectAcquisition is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.sys.funclib.configObjectAcquisition; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.sys.funclib.configObjectAcquisition;

import de.bsvrz.dav.daf.main.config.ConfigurationArea;
import de.bsvrz.dav.daf.main.config.ConfigurationAuthority;
import de.bsvrz.dav.daf.main.config.ConfigurationObject;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.ObjectSet;
import de.bsvrz.dav.daf.main.config.ObjectTimeSpecification;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectCollection;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.sys.funclib.debug.Debug;
import java.util.LinkedList;
import java.util.List;

/**
 * Hilfsfunktion(en) für den Zugriff auf die Konfiguration
 *
 * @author Kappich Systemberatung
*/
public final class ConfigurationHelper {

    /** Der Debug-Logger der Klasse */
    private static final Debug debug = Debug.getLogger();

    /** Erzeugt ein neues Objekt der Klasse {@code ConfigurationHelper} */
    public ConfigurationHelper() {
    }

    /**
     * Holt die durch den {@code objectSpecString} spezifizierten Objekte.
     *
     * @param objectSpecString String der die Objekte beschreibt. Eine Objektspezifikation besteht aus einer oder mehreren mit Komma getrennten
     *                         Teilspezifikationen. Eine Teilspezifikation kann nach einem der folgenden Muster aufgebaut sein:
     *                         <p>
     *                         - `PID` wählt das Objekt mit der angegebenen PID aus - `Typ-PID:*` wählt alle Objekte des angegebenen Typs aus -
     *                         `PID:Mengenname` wählt alle Objekte der angebenen Menge unterhalb des angegebenen Objekts aus - `KB-PID:Typ-Pid` wählt
     *                         alle Objekte des angegebenen Typs im angegebenen Konfigurationsbereich aus - `KV-PID:Typ-Pid` wählt alle Objekte des
     *                         angegebenen Typs in allen Konfigurationsbereichen aus, für die der angegebenen KV verantwortlich ist
     *                         <p>
     *                         Diese Muster können teilweise auch kombiniert werden. Beispielsweise wählt `kb.testObjekte:typ
     *                         .messQuerschnitt:FahrStreifen`
     *                         alle Elemente der Menge `Fahrstreifen` aus, die an Objekten des Typs `typ.messQuerschnitt` im Bereich kb.testObjekte
     *                         enthalten sind.
     * @param dataModel        Objekt zum Zugriff auf die Konfiguration.
     *
     * @return Spezifizierte Objekte.
     *
     * @throws IllegalArgumentException Wenn die Objektspezifikation nicht interpretiert werden kann.
     */
    public static List<SystemObject> getObjects(String objectSpecString, DataModel dataModel) throws IllegalArgumentException {
        return getObjects(objectSpecString, dataModel, ObjectTimeSpecification.valid());
    }

    /**
     * Holt die durch den {@code objectSpecString} spezifizierten Objekte.
     *
     * @param objectSpecString  String der die Objekte beschreibt. Eine Objektspezifikation besteht aus einer oder mehreren mit Komma getrennten
     *                          Teilspezifikationen. Eine Teilspezifikation kann nach einem der folgenden Muster aufgebaut sein:
     *                          <p>
     *                          - `PID` wählt das Objekt mit der angegebenen PID aus - `Typ-PID:*` wählt alle Objekte des angegebenen Typs aus -
     *                          `PID:Mengenname` wählt alle Objekte der angebenen Menge unterhalb des angegebenen Objekts aus - `KB-PID:Typ-Pid` wählt
     *                          alle Objekte des angegebenen Typs im angegebenen Konfigurationsbereich aus - `KV-PID:Typ-Pid` wählt alle Objekte des
     *                          angegebenen Typs in allen Konfigurationsbereichen aus, für die der angegebenen KV verantwortlich ist
     *                          <p>
     *                          Diese Muster können teilweise auch kombiniert werden. Beispielsweise wählt `kb.testObjekte:typ
     *                          .messQuerschnitt:FahrStreifen`
     *                          alle Elemente der Menge `Fahrstreifen` aus, die an Objekten des Typs `typ.messQuerschnitt` im Bereich kb.testObjekte
     *                          enthalten sind.
     * @param dataModel         Objekt zum Zugriff auf die Konfiguration.
     * @param timeSpecification Zeitbereich, in dem Objekte geholt werden sollen. Damit können auch bereits gelöschte dynamische Objekte o.ä.
     *                          berücksichtigt werden.
     *
     * @return Spezifizierte Objekte.
     *
     * @throws IllegalArgumentException Wenn die Objektspezifikation nicht interpretiert werden kann.
     * @since 3.15
     */
    public static List<SystemObject> getObjects(String objectSpecString, DataModel dataModel, ObjectTimeSpecification timeSpecification)
        throws IllegalArgumentException {
        List<SystemObject> allObjects = new LinkedList<>();
        String[] objectSpecs = objectSpecString.split(",");
        for (final String objectSpec : objectSpecs) {
            List<SystemObject> objects = new LinkedList<>();

            String[] objectSpecParts = objectSpec.split(":");
            try {
                long objectId = 0;
                objectId = Long.parseLong(objectSpecParts[0]);
                objects.add(dataModel.getObject(objectId));
            } catch (Exception e) {
                SystemObject theNewObject = dataModel.getObject(objectSpecParts[0]);
                if (theNewObject != null) {
                    objects.add(theNewObject);
                } else {
                    debug.error("Objekt \"" + objectSpecParts[0] + "\" existiert nicht.");
                    throw new IllegalArgumentException("Objekt \"" + objectSpecParts[0] + "\" existiert nicht", e);
                }
            }
            for (int partIndex = 1; partIndex < objectSpecParts.length; ++partIndex) {
                String objectSetName = objectSpecParts[partIndex];
                List<SystemObject> newObjects = new LinkedList<>();
                for (final Object object : objects) {
                    if (object instanceof ConfigurationObject) {
                        if (objectSetName.equals("*")) {
	                        if (object instanceof SystemObjectType typeObject) {
                                List<SystemObject> elements = getElementsInTime(timeSpecification, typeObject);
                                newObjects.addAll(elements);
                            }
                        } else if (object instanceof ConfigurationArea || object instanceof ConfigurationAuthority) {
                            // Interpretation von bereichPid:typPid durch alle Objekte des Typs in dem Bereich
                            // Interpretation von kvPid:typPid durch alle Objekte des Typs in dee Bereichen des KV
                            LinkedList<ConfigurationArea> areaList = new LinkedList<>();
                            if (object instanceof ConfigurationArea) {
                                areaList.add((ConfigurationArea) object);
                            } else {
                                ConfigurationAuthority kv = (ConfigurationAuthority) object;
                                List<SystemObject> allAreas = getElementsInTime(timeSpecification, dataModel.getType("typ.konfigurationsBereich"));
                                for (SystemObject areaObject : allAreas) {
                                    if (((ConfigurationArea) areaObject).getConfigurationAuthority().equals(kv)) {
                                        areaList.add((ConfigurationArea) areaObject);
                                    }
                                }
                            }
                            String typePid = objectSetName;
                            SystemObject typeObject = dataModel.getObject(typePid);
	                        if (typeObject instanceof SystemObjectType systemObjectType) {
                                LinkedList<SystemObjectType> typeList = new LinkedList<>();
                                typeList.add(systemObjectType);
                                newObjects.addAll(dataModel.getObjects(areaList, typeList, timeSpecification));
                            } else {
                                throw new IllegalArgumentException(
                                    "Objektspezifikation mit bereich:typ bzw. kv:typ nicht gültig, weil " + "Typ '" + typePid +
                                    "' nicht aufgelöst werden konnte. " + "(Spezifikation: " + objectSpec + ")");
                            }
                        } else {
                            ObjectSet set = ((ConfigurationObject) object).getObjectSet(objectSetName);
                            if (set != null) {
                                newObjects.addAll(getElementsInTime(timeSpecification, set));
                            }
                        }
                    }
                }
                objects = newObjects;
            }
            allObjects.addAll(objects);
        }
        return allObjects;
    }

    public static List<SystemObject> getElementsInTime(final ObjectTimeSpecification timeSpecification, final SystemObjectCollection typeObject) {
	    List<SystemObject> elements = switch (timeSpecification.getType()) {
		    case VALID -> typeObject.getElements();
		    case VALID_AT_TIME -> typeObject.getElements(timeSpecification.getTime());
		    case VALID_IN_PERIOD ->
				    typeObject.getElementsInPeriod(timeSpecification.getStartTime(), timeSpecification.getEndTime());
		    case VALID_DURING_PERIOD ->
				    typeObject.getElementsDuringPeriod(timeSpecification.getStartTime(), timeSpecification.getEndTime());
		    default -> throw new IllegalArgumentException("Falscher Typ:" + timeSpecification.getType());
	    };
        return elements;
    }
}
