/*
 * Copyright 2004 by Kappich+Kniß Systemberatung, Aachen
 * Copyright 2006-2020 by Kappich Systemberatung, Aachen
 * Copyright 2021 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.sys.funclib.commandLineArgs.
 *
 * de.bsvrz.sys.funclib.commandLineArgs is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.sys.funclib.commandLineArgs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.sys.funclib.commandLineArgs; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.sys.funclib.commandLineArgs;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.math.BigDecimal;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Scanner;
import java.util.TimeZone;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * Klasse zum Zugriff auf die Aufrufargumente einer Applikation. Eine Applikation bekommt die Aufrufargumente von der Laufzeitumgebung in einem
 * String-Array als Parameter der {@code main}-Funktion übergeben. Dieses String-Array wird dem Konstruktor dieser Klasse übergeben. Mit der Methode
 * {@link #fetchArgument} kann auf die einzelnen Argumente zugegriffen werden. Beim Zugriff auf ein Argument wird der entsprechende Eintrag im
 * String-Array auf {@code null} gesetzt. Nach dem Zugriff auf alle von einer Applikation unterstützten Argumente kann mit der Methode {@link
 * #ensureAllArgumentsUsed} sichergestellt werden, daß alle angegebenen Argumente verwendet wurden.
 *
 * @author Kappich Systemberatung
*/
public class ArgumentList {
    /**
     * Speichert eine Kopie der ursprünglichen Aufrufargumente der Applikation, die dem Konstruktor übergeben wurden.
     */
    private final String[] _initialArgumentStrings;

    /**
     * Speichert eine Referenz auf die Aufrufargumente der Applikation, die dem Konstruktor übergeben wurden. Nachdem ein Argument interpretiert
     * wurde, wird es im übergebenen Array auf {@code null} gesetzt.
     */
    private final String[] _argumentStrings;

    /**
     * Erzeugt eine neue Argumentliste und initialisiert diese mit den übergebenen Aufrufargumenten der Applikation. Einzelne Argumente, die von der
     * Applikation bereits interpretiert wurden, sollten vorher auf den Wert {@code null} gesetzt werden, damit sie nicht nochmal interpretiert
     * werden.
     *
     * @param argumentStrings String-Array, das die Aufrufargumente enthält, die der Applikation beim Aufruf der main-Funktion übergeben werden.
     */
    public ArgumentList(String[] argumentStrings) {
        _argumentStrings = argumentStrings;
        _initialArgumentStrings = new String[argumentStrings.length];
        System.arraycopy(_argumentStrings, 0, _initialArgumentStrings, 0, _argumentStrings.length);
    }

    /**
     * Hauptfunktion zum Test einzelner Methoden
     */
    public static void main(String[] args) {
        try {
            DateFormat outputFormat = new SimpleDateFormat("dd.MM.yyyy HH:mm:ss,SSS");
            ArgumentList arguments = new ArgumentList(args);
            long fromTime = arguments.fetchArgument("-von=24:00").asTime();
            long duration = arguments.fetchArgument("-dauer=011t#011m0s").asRelativeTime();

            long now = System.currentTimeMillis();
            System.out.println("now:" + now + " ms");
            System.out.println("now:" + outputFormat.format(new Date(now)));
            System.out.println("fromTime:" + fromTime + " ms");
            System.out.println("fromTime:" + outputFormat.format(new Date(fromTime)));
            System.out.println("duration:" + duration + " ms");
            System.out.println("duration:" + outputFormat.format(new Date(duration)));
            arguments.ensureAllArgumentsUsed();
        } catch (Exception e) {
            System.out.println("Fehler: " + e);
            System.exit(1);
            return;
        }
    }

    /**
     * Liefert ein bestimmtes Argument zurück und setzt es im String-Array, das beim Konstruktor übergeben wurde, auf {@code null}. Im übergebenen
     * Parameter der Funktion wird der Name des Arguments und bei Bedarf (durch ein Gleich-Zeichen getrennt) ein Default-Wert angegeben. Die
     * Aufrufargumente müssen dem Format "argumentName=argumentWert" entsprechen: Beim Zugriff auf ein Argument muss der Argument-Name angegeben
     * werden. Ergebnis des Zugriffs ist ein Objekt der Klasse {@link Argument} über das der Wert des Arguments abgefragt werden kann. Wenn das
     * gewünschte Argument in der Argumentliste gefunden wurde, wird der entsprechende Eintrag im String-Array, das dem {@link #ArgumentList
     * Konstruktor} übergeben wurde, auf den Wert {@code null} gesetzt. Wenn auf ein Argument zugegriffen wird, das in der Argumentlist nicht mehr
     * vorhanden ist, weil es bereits vorher interpretiert wurde, dann wird der entsprechende Wert aus der initialen Argumentliste erneut
     * zurückgegeben. Wenn auf ein Argument zugegriffen wird, das in der initialen Argumentliste nicht vorhanden ist, wird als Wert der Default-Wert
     * benutzt, wenn dieser im Parameter der Methode (durch ein Gleichzeichen vom Argumentnamen getrennt) angegeben wurde. Wenn das gewünschte
     * Argument nicht in der Argumentliste enthalten ist und kein Default-Wert angegeben wurde, wird eine Ausnahme generiert.
     *
     * @param nameAndOptionalDefault Name des gewünschten Arguments und optional durch ein Gleichzeichen getrennt der Default-Wert des Arguments.
     *
     * @return Ein Argument-Objekt über das mit verschiedenen Methoden auf den Wert des Arguments zugegriffen werden kann.
     *
     * @throws IllegalArgumentException Wenn kein Wert für das gewünschte Argument ermittelt werden konnte.
     */
    public Argument fetchArgument(String nameAndOptionalDefault) {
        int defaultSeparatorPosition = nameAndOptionalDefault.indexOf('=');
        String name = nameAndOptionalDefault;
        if (defaultSeparatorPosition >= 0) {
            name = nameAndOptionalDefault.substring(0, defaultSeparatorPosition);
        }
        for (int i = 0; i < _argumentStrings.length; ++i) {
            String argumentString = _argumentStrings[i];
            if (argumentString != null && (argumentString.equals(name) || argumentString.startsWith(name + "="))) {
                _argumentStrings[i] = null;
                return new Argument(argumentString);
            }
        }
        for (String argumentString : _initialArgumentStrings) {
            if (argumentString != null && (argumentString.equals(name) || argumentString.startsWith(name + "="))) {
                return new Argument(argumentString);
            }
        }
        if (defaultSeparatorPosition >= 0) {
            return new Argument(nameAndOptionalDefault);
        }
        throw new IllegalArgumentException("fehlendes Argument: " + name);
    }

    /**
     * Prüft, ob ein bestimmtes Argument vorhanden ist und noch nicht interpretiert wurde.
     *
     * @param name Name des gesuchten Arguments.
     *
     * @return {@code true} Wenn das gesuchte Argument in der Argumentliste enthalten ist und noch nicht interpretiert wurde, sonst {@code false}.
     */
    public boolean hasArgument(String name) {
        for (String argumentString : _argumentStrings) {
            if (argumentString != null && (argumentString.equals(name) || argumentString.startsWith(name + "="))) {
                return true;
            }
        }
        return false;
    }

    /**
     * Diese Methode stellt sicher, daß alle Argumente interpretiert wurden.
     *
     * @throws IllegalStateException Wenn in der Argumentliste noch nicht ausgewertete Argumente enthalten sind.
     */
    public void ensureAllArgumentsUsed() {
        if (hasUnusedArguments()) {
            StringBuilder message = new StringBuilder("Unbenutzte Argumente: {");
            Argument[] unusedArguments = fetchUnusedArguments();
            for (final Argument unusedArgument : unusedArguments) {
                message.append(unusedArgument).append(", ");
            }
            message.append("}");
            throw new IllegalStateException(message.toString());
        }
    }

    /**
     * Bestimmt, ob in der Argumentliste noch Argumente existieren, die noch nicht ausgewertet wurden.
     *
     * @return {@code true}, falls weitere Argumente existieren; {@code false}, falls alle Argumente ausgewertet wurden.
     */
    public boolean hasUnusedArguments() {
        for (final String _argumentString : _argumentStrings) {
            if (_argumentString != null) {
                return true;
            }
        }
        return false;
    }

    /**
     * Liefert den Namen des nächsten noch nicht interpretierten Arguments zurück.
     *
     * @return Name des nächsten noch nicht interpretierten Arguments.
     *
     * @throws IllegalStateException Wenn bereits alle Argumente interpretiert wurden.
     */
    public String getNextArgumentName() {
        for (String argumentString : _argumentStrings) {
            if (argumentString != null) {
                return new Argument(argumentString).getName();
            }
        }
        throw new IllegalStateException("Zu wenig Argumente");
    }

    /**
     * Liefert das erste noch nicht interpretierte Argument zurück und setzt es im String-Array, das beim Konstruktor übergeben wurde, auf {@code
     * null}. Die Aufrufargumente müssen dem Format "argumentName=argumentWert" entsprechen: Beim Zugriff auf ein Argument muss der Argument-Name
     * angegeben werden. Ergebnis des Zugriffs ist ein Objekt der Klasse {@link Argument} über das der Wert des Arguments abgefragt werden kann. Für
     * Argumente die kein Gleichzeichen mit folgendem argumentWert enthalten, wird als Wert der Text "wahr" angenommen. Der entsprechende Eintrag im
     * String-Array, das dem {@link #ArgumentList Konstruktor} übergeben wurde, wird auf den Wert {@code null} gesetzt. Wenn in der Argumentliste kein
     * Argument mehr enthalten war, wird eine Ausnahme generiert.
     *
     * @return Ein Argument-Objekt über das mit verschiedenen Methoden auf den Wert des Arguments zugegriffen werden kann.
     *
     * @throws IllegalStateException Wenn bereits alle Argumente interpretiert wurden.
     */
    public Argument fetchNextArgument() {
        for (int i = 0; i < _argumentStrings.length; ++i) {
            String argumentString = _argumentStrings[i];
            if (argumentString != null) {
                _argumentStrings[i] = null;
                return new Argument(argumentString);
            }
        }
        throw new IllegalStateException("Zu wenig Argumente");
    }

    /**
     * Bestimmt die Anzahl der in der Argumentliste noch vorhandenen und nicht ausgewerteten Argumente.
     *
     * @return Anzahl noch nicht ausgewerteten Argumente in der Argumentliste.
     */
    public int getUnusedArgumentCount() {
        int unusedCount = 0;
        for (final String _argumentString : _argumentStrings) {
            if (_argumentString != null) {
                ++unusedCount;
            }
        }
        return unusedCount;
    }

    /**
     * Liefert ein Feld mit den noch nicht ausgewerteten Argumenten der Aufrufliste zurück und setzt die entsprechenden Einträge im String-Array, das
     * beim Konstruktor übergeben wurde, auf {@code null}.
     *
     * @return Feld mit Argument-Objekten der noch nicht ausgewerteten Argumente.
     */
    public Argument[] fetchUnusedArguments() {
        int unusedCount = getUnusedArgumentCount();
        if (unusedCount == 0) {
            return null;
        }
        Argument[] unusedArguments = new Argument[unusedCount];
        unusedCount = 0;
        for (int i = 0; i < _argumentStrings.length; ++i) {
            if (_argumentStrings[i] != null) {
                unusedArguments[unusedCount++] = new Argument(_argumentStrings[i]);
                _argumentStrings[i] = null;
            }
        }
        return unusedArguments;
    }

    /**
     * Liefert das String-Array mit den noch nicht interpretierten Aufrufargumenten der Applikation zurück. Das zurückgegebene Objekt ist das selbe
     * Objekt mit dem der Konstruktor aufgerufen wurde, allerdings ist zu beachten, dass die Elemente des Arrays, die bereits mit den Methoden {@link
     * #fetchArgument} bzw. {@link #fetchNextArgument} interpretiert wurden, im Array auf {@code null} gesetzt wurden.
     *
     * @return Das String-Array mit den noch nicht interpretierten Aufrufargumenten der Applikation.
     */
    public String[] getArgumentStrings() {
        return _argumentStrings;
    }

    /**
     * Liefert das String-Array mit den initialen Aufrufargumenten der Applikation zurück. Das zurückgegebene Array enthält die selben Aufrufargumente
     * die dem Konstruktor übergeben wurden.
     *
     * @return Das String-Array mit den initialen Aufrufargumenten der Applikation.
     */
    public String[] getInitialArgumentStrings() {
        return _initialArgumentStrings;
    }

    /**
     * Liefert eine textuelle Beschreibung dieser Argumentliste mit den initialen Argumenten zurück. Das genaue Format ist nicht festgelegt und kann
     * sich ändern.
     *
     * @return Beschreibung dieser Argumentliste.
     */
    public String toString() {
        return "ArgumentList" + Arrays.asList(_initialArgumentStrings).toString();
    }

    /**
     * Klasse zum Zugriff auf Name und Wert eines Aufrufarguments.
     */
    public static class Argument {

        public static final Pattern DURATION_PATTERN = Pattern.compile(
            "" + "(?:(-?[0-9]+(?:,[0-9]{0,3})?)\\s*(?:d|t|tage?)[, ]*)?" + "(?:(-?[0-9]+(?:,[0-9]{0,3})?)\\s*(?:h|stunden?)[, ]*)?" +
            "(?:(-?[0-9]+(?:,[0-9]{0,3})?)\\s*(?:m(?:inuten?)?)[, ]*)?" + "(?:(-?[0-9]+(?:,[0-9]{0,3})?)\\s*(?:s(?:ekunden?)?)[, ]*)?" +
            "(?:(-?[0-9]+)\\s*(?:ms))?", Pattern.CASE_INSENSITIVE);
        public static final Pattern DURATION_PATTERN_2 = Pattern
            .compile("" + "(?:([0-9]+)\\s*(?:d|t|tage)[, ]*)?" + "([0-9]+):" + "([0-9]{1,2})" + "(?::([0-9]{1,2})" + "(?:,([0-9]{1,3}))?)?",
                     Pattern.CASE_INSENSITIVE);
        public static final Pattern BYTE_UNIT_PATTERN = Pattern.compile("\\s*((?:[KMGTPEZ]i?)?)B?", Pattern.CASE_INSENSITIVE);
        public static final Pattern BYTE_DELIM_PATTERN = Pattern.compile("\\s*(?=((?:[KMGTPEZ]i?)?)B?$)", Pattern.CASE_INSENSITIVE);
        /**
         * Pattern für SI-Einheiten mit Prefix.
         */
        private static final Pattern SI_UNIT_PATTERN = Pattern.compile("[kMGTPEZcmuµnpf].+");
        private static final DateFormat[] _parseDateFormats = {new SimpleDateFormat("HH:mm:ss,SSS dd.MM.yy"),
                                                               new SimpleDateFormat("HH:mm:ss dd.MM.yy"),
                                                               new SimpleDateFormat("HH:mm dd.MM.yy"),
                                                               new SimpleDateFormat("dd.MM.yy HH:mm:ss,SSS"),
                                                               new SimpleDateFormat("dd.MM.yy HH:mm:ss"),
                                                               new SimpleDateFormat("dd.MM.yy HH:mm"),
                                                               new SimpleDateFormat("dd.MM.yy"),};
        private static final DateFormat[] _parseTimeFormats =
            {new SimpleDateFormat("HH:mm:ss,SSS"), new SimpleDateFormat("HH:mm:ss"), new SimpleDateFormat("HH:mm"),};

//		public Argument(String name, String value) {
//			_name= name;
//			_value= value;
//		}
        private static final long _startDay;
        private static final String _relNumberPattern = "-?(?:(?:0[0-7]{1,22}+)|(?:[1-9][0-9]{0,18}+)|(?:(?:#|0x|0X)[0-9a-fA-F]{0,16}+)|(?:0))";
        private static final String _relNamePattern = "[tThHsSmM][a-zA-Z]{0,15}+";
        private static final String _relNumberNamePattern = "(?<=" + _relNumberPattern + ")\\s*(?=" + _relNamePattern + ")";
        private static final String _relNameNumberPattern = "(?<=" + _relNamePattern + ")\\s*(?=" + _relNumberPattern + ")";
        private static final String _relPattern = "(?:" + _relNumberNamePattern + ")|(?:" + _relNameNumberPattern + ")";

        static {
            Calendar startDay = new GregorianCalendar();
            //startDay.setTimeInMillis(System.currentTimeMillis());
            startDay.set(Calendar.HOUR_OF_DAY, 0);
            startDay.set(Calendar.MINUTE, 0);
            startDay.set(Calendar.SECOND, 0);
            startDay.set(Calendar.MILLISECOND, 0);
            _startDay = startDay.getTimeInMillis();
            for (final DateFormat _parseTimeFormat : _parseTimeFormats) {
                _parseTimeFormat.setTimeZone(TimeZone.getTimeZone("GMT"));
            }
        }

        /**
         * Speichert den Namen eines Arguments.
         */
        private final String _name;
        /**
         * Speichert den Wert eines Arguments.
         */
        private final String _value;

        /**
         * Erzeugt ein neues Argument-Objekt aus dem übergebenen Argument-Text. Der Argument-Text muss folgenden Aufbau haben:
         * "argumentName=argumentWert". Name und Wert des Argument-Objekts werden entsprechend gesetzt. Wenn der Argument-Text kein Gleichzeichen
         * enthält, dann wird der ganze Argument-Text als Name des Arguments interpretiert und es wird vermerkt, daß das Argument keinen Wert hat.
         */
        Argument(String argumentString) {
            int separatorPosition = argumentString.indexOf('=');
            if (separatorPosition >= 0) {
                _name = argumentString.substring(0, separatorPosition);
                _value = argumentString.substring(separatorPosition + 1);
            } else {
                _name = argumentString;
                _value = null; // Default-Wert, wenn kein Wert angegeben wurde
            }
        }

        /**
         * Erzeugt ein neues Argument-Objekt aus den übergebenen Parametern
         *
         * @param name  Name des neuen Arguments
         * @param value Wert des neuen Arguments
         */
        private Argument(String name, String value) {
            _name = name;
            _value = value;
        }

        private static long addDuration(final String text, final long factor) {
            if (text == null) {
                return 0;
            }
            String[] split = text.split(",");
            long intValue = split[0].isEmpty() ? 0 : Long.parseLong(split[0]);
            if (split.length == 1) {
                return intValue * factor;
            }
            String fractionText = split[1];
            long fraction = Long.parseLong(fractionText);
            for (int i = fractionText.length(); i < 3; i++) {
                fraction *= 10;
            }
            if (text.startsWith("-")) {
                fraction *= -1;
            }
            return intValue * factor + fraction * factor / 1000;
        }

        private static <T> T tryParse(final Function<DateTimeFormatter, T> parseFunc, final DateTimeFormatter... params) {
            for (DateTimeFormatter param : params) {
                try {
                    return parseFunc.apply(param);
                } catch (Exception ignored) {
                }
            }
            LocalDateTime now = LocalDateTime.now();
            throw new IllegalArgumentException("Ungültiges Format. Eines der folgenden Formate ist erlaubt: \n" +
                                               Arrays.stream(params).map(it -> it.format(now)).collect(Collectors.joining("\n")));
        }

        /**
         * Erzeugt ein neues Argument dessen Wert aus dem Namen dieses Arguments übernommen wird.
         *
         * @return Neues Argument-Objekt
         */
        public Argument toArgumentWithNameAsValue() {
            return new Argument(getName(), getName());
        }

        /**
         * Bestimmt den Namen des Arguments.
         *
         * @return Name des Arguments.
         */
        public String getName() {
            return _name;
        }

        /**
         * Überprüft, ob das Argument einen Wert hat. Wenn das Argument keinen Wert hat, kann nur noch mit der Methode #{@link #booleanValue} auf den
         * Wert zugegriffen werden, ohne daß eine Ausnahme generiert wird.
         *
         * @return {@code true}, falls das Argument einen Wert hat; sonst {@code false}.
         */
        public boolean hasValue() {
            return _value != null;
        }

        /**
         * Bestimmt den Wert des Arguments.
         *
         * @return Wert des Arguments.
         *
         * @throws IllegalStateException Wenn das Argument keinen Wert hat.
         */
        public String getValue() {
            if (hasValue()) {
                return _value;
            }
            throw new IllegalArgumentException("Argument " + getName() + " hat keinen Wert");
        }

        /**
         * Gibt den Wert des Arguments als {@code boolean} Wert zurück. Die Argumentwerte "wahr", "ja", "1" werden zum Boolschen Wert {@code true}
         * konvertiert; die Argumentwerte "falsch", "nein", "0" werden zum Boolschen Wert {@code false} konvertiert. Die Groß-/Kleinschreibung des
         * Argumentwerts hat beim Vergleich keine Relevanz. Wenn das Argument keinen Wert hat, dann wird als Ergebnis der Konvertierung {@code true}
         * zurückgegeben.
         *
         * @return Der konvertierte Argumentwert.
         *
         * @throws IllegalArgumentException Wenn der Argumentwert nicht konvertiert werden konnte.
         */
        public boolean booleanValue() throws IllegalArgumentException {
            if (!hasValue()) {
                return true;
            }
            String value = _value.toLowerCase();
	        return switch (value) {
		        case "wahr", "ja", "1" -> true;
		        case "falsch", "nein", "0" -> false;

		        //englische Varianten werden auch unterstützt:
		        case "true", "yes" -> true;
		        case "false", "no" -> false;
		        default ->
				        throw new IllegalArgumentException("Argument " + getName() + " hat keinen boolschen Wert: " + getValue());
	        };
        }

        /**
         * Gibt den Wert des Arguments als {@code byte} Wert zurück. Der Argumentwert wird in einen Zahlwert vom Typ {@code byte} konvertiert.
         *
         * @return Der konvertierte Argumentwert.
         *
         * @throws NumberFormatException Wenn der Argumentwert nicht konvertiert werden konnte.
         * @throws IllegalStateException Wenn das Argument keinen Wert hat.
         */
        public byte byteValue() throws NumberFormatException {
            try {
                return Byte.parseByte(getValue());
            } catch (Exception e) {
                throw new NumberFormatException("Argument " + getName() + " hat keinen Byte-Wert: " + getValue());
            }
        }

        /**
         * Gibt den Wert des Arguments als {@code short} Wert zurück. Der Argumentwert wird in einen Zahlwert vom Typ {@code short} konvertiert.
         *
         * @return Der konvertierte Argumentwert.
         *
         * @throws NumberFormatException Wenn der Argumentwert nicht konvertiert werden konnte.
         * @throws IllegalStateException Wenn das Argument keinen Wert hat.
         */
        public short shortValue() throws NumberFormatException {
            try {
                return Short.parseShort(getValue());
            } catch (Exception e) {
                throw new NumberFormatException("Argument " + getName() + " hat keinen Short-Wert: " + getValue());
            }
        }

        /**
         * Gibt den Wert des Arguments als {@code int} Wert zurück. Der Argumentwert wird in einen Zahlwert vom Typ {@code int} konvertiert.
         *
         * @return Der konvertierte Argumentwert.
         *
         * @throws NumberFormatException Wenn der Argumentwert nicht konvertiert werden konnte.
         * @throws IllegalStateException Wenn das Argument keinen Wert hat.
         */
        public int intValue() throws NumberFormatException {
            try {
                return Integer.parseInt(getValue());
            } catch (Exception e) {
                throw new NumberFormatException("Argument " + getName() + " hat keinen Integer-Wert: " + getValue());
            }
        }

        /**
         * Gibt den Wert des Arguments als {@code long} Wert zurück. Der Argumentwert wird in einen Zahlwert vom Typ {@code long} konvertiert.
         *
         * @return Der konvertierte Argumentwert.
         *
         * @throws NumberFormatException Wenn der Argumentwert nicht konvertiert werden konnte.
         * @throws IllegalStateException Wenn das Argument keinen Wert hat.
         */
        public long longValue() throws NumberFormatException {
            try {
                return Long.parseLong(getValue());
            } catch (Exception e) {
                throw new NumberFormatException("Argument " + getName() + " hat keinen Long-Wert: " + getValue());
            }
        }

        /**
         * Gibt den Wert des Arguments als {@code float} Wert zurück. Der Argumentwert wird in einen Zahlwert vom Typ {@code float} konvertiert.
         *
         * @return Der konvertierte Argumentwert.
         *
         * @throws NumberFormatException Wenn der Argumentwert nicht konvertiert werden konnte.
         * @throws IllegalStateException Wenn das Argument keinen Wert hat.
         */
        public float floatValue() throws NumberFormatException {
            try {
                return Float.parseFloat(getValue());
            } catch (Exception e) {
                throw new NumberFormatException("Argument " + getName() + " hat keinen Float-Wert: " + getValue());
            }
        }

        /**
         * Gibt den Wert des Arguments als {@code double} Wert zurück. Der Argumentwert wird in einen Zahlwert vom Typ {@code double} konvertiert.
         *
         * @return Der konvertierte Argumentwert.
         *
         * @throws NumberFormatException Wenn der Argumentwert nicht konvertiert werden konnte.
         * @throws IllegalStateException Wenn das Argument keinen Wert hat.
         */
        public double doubleValue() throws NumberFormatException {
            try {
                return Double.parseDouble(getValue());
            } catch (Exception e) {
                throw new NumberFormatException("Argument " + getName() + " hat keinen Double-Wert: " + getValue());
            }
        }

        /**
         * Gibt den Wert des Arguments als {@code byte} Wert zurück. Der Argumentwert wird in einen Zahlwert vom Typ {@code byte} konvertiert und
         * überprüft, ob der Wert nicht ausserhalb der angegebenen Grenzen liegt.
         *
         * @param minimum Kleinster erlaubter Wert.
         * @param maximum Größter erlaubter Wert.
         *
         * @return Der konvertierte Argumentwert.
         *
         * @throws IllegalArgumentException Wenn der Wert kleiner als das Minimum oder größer als das Maximum ist.
         * @throws NumberFormatException    Wenn der Argumentwert nicht konvertiert werden konnte.
         * @throws IllegalStateException    Wenn das Argument keinen Wert hat.
         */
        public byte byteValueBetween(byte minimum, byte maximum) throws NumberFormatException {
            byte value;
            try {
                value = Byte.parseByte(getValue());
            } catch (Exception e) {
                throw new NumberFormatException("Argument " + getName() + " hat keinen Byte-Wert: " + getValue());
            }
            if (value >= minimum && value <= maximum) {
                return value;
            }
            throw new IllegalArgumentException(
                "Argumentwert " + getValue() + "von Argument " + getName() + " liegt nicht zwischen " + minimum + " und " + maximum);
        }

        /**
         * Gibt den Wert des Arguments als {@code short} Wert zurück. Der Argumentwert wird in einen Zahlwert vom Typ {@code short} konvertiert und
         * überprüft, ob der Wert nicht ausserhalb der angegebenen Grenzen liegt.
         *
         * @param minimum Kleinster erlaubter Wert.
         * @param maximum Größter erlaubter Wert.
         *
         * @return Der konvertierte Argumentwert.
         *
         * @throws IllegalArgumentException Wenn der Wert kleiner als das Minimum oder größer als das Maximum ist.
         * @throws NumberFormatException    Wenn der Argumentwert nicht konvertiert werden konnte.
         * @throws IllegalStateException    Wenn das Argument keinen Wert hat.
         */
        public short shortValueBetween(short minimum, short maximum) throws NumberFormatException {
            short value;
            try {
                value = Short.parseShort(getValue());
            } catch (Exception e) {
                throw new NumberFormatException("Argument " + getName() + " hat keinen Short-Wert: " + getValue());
            }
            if (value >= minimum && value <= maximum) {
                return value;
            }
            throw new IllegalArgumentException(
                "Argumentwert " + getValue() + "von Argument " + getName() + " liegt nicht zwischen " + minimum + " und " + maximum);
        }

        /**
         * Gibt den Wert des Arguments als {@code int} Wert zurück. Der Argumentwert wird in einen Zahlwert vom Typ {@code int} konvertiert und
         * überprüft, ob der Wert nicht ausserhalb der angegebenen Grenzen liegt.
         *
         * @param minimum Kleinster erlaubter Wert.
         * @param maximum Größter erlaubter Wert.
         *
         * @return Der konvertierte Argumentwert.
         *
         * @throws IllegalArgumentException Wenn der Wert kleiner als das Minimum oder größer als das Maximum ist.
         * @throws NumberFormatException    Wenn der Argumentwert nicht konvertiert werden konnte.
         * @throws IllegalStateException    Wenn das Argument keinen Wert hat.
         */
        public int intValueBetween(int minimum, int maximum) throws NumberFormatException {
            int value;
            try {
                value = Integer.parseInt(getValue());
            } catch (Exception e) {
                throw new NumberFormatException("Argument " + getName() + " hat keinen Integer-Wert: " + getValue());
            }
            if (value >= minimum && value <= maximum) {
                return value;
            }
            throw new IllegalArgumentException(
                "Argumentwert " + getValue() + " von Argument " + getName() + " liegt nicht zwischen " + minimum + " und " + maximum);
        }

        /**
         * Gibt den Wert des Arguments als {@code long} Wert zurück. Der Argumentwert wird in einen Zahlwert vom Typ {@code long} konvertiert und
         * überprüft, ob der Wert nicht ausserhalb der angegebenen Grenzen liegt.
         *
         * @param minimum Kleinster erlaubter Wert.
         * @param maximum Größter erlaubter Wert.
         *
         * @return Der konvertierte Argumentwert.
         *
         * @throws IllegalArgumentException Wenn der Wert kleiner als das Minimum oder größer als das Maximum ist.
         * @throws NumberFormatException    Wenn der Argumentwert nicht konvertiert werden konnte.
         * @throws IllegalStateException    Wenn das Argument keinen Wert hat.
         */
        public long longValueBetween(long minimum, long maximum) throws NumberFormatException {
            long value;
            try {
                value = Long.parseLong(getValue());
            } catch (Exception e) {
                throw new NumberFormatException("Argument " + getName() + " hat keinen Long-Wert: " + getValue());
            }
            if (value >= minimum && value <= maximum) {
                return value;
            }
            throw new IllegalArgumentException(
                "Argumentwert " + getValue() + "von Argument " + getName() + " liegt nicht zwischen " + minimum + " und " + maximum);
        }

        /**
         * Gibt den Wert des Arguments als {@code float} Wert zurück. Der Argumentwert wird in einen Zahlwert vom Typ {@code float} konvertiert und
         * überprüft, ob der Wert nicht ausserhalb der angegebenen Grenzen liegt.
         *
         * @param minimum Kleinster erlaubter Wert.
         * @param maximum Größter erlaubter Wert.
         *
         * @return Der konvertierte Argumentwert.
         *
         * @throws IllegalArgumentException Wenn der Wert kleiner als das Minimum oder größer als das Maximum ist.
         * @throws NumberFormatException    Wenn der Argumentwert nicht konvertiert werden konnte.
         * @throws IllegalStateException    Wenn das Argument keinen Wert hat.
         */
        public float floatValueBetween(float minimum, float maximum) throws NumberFormatException {
            float value;
            try {
                value = Float.parseFloat(getValue());
            } catch (Exception e) {
                throw new NumberFormatException("Argument " + getName() + " hat keinen Float-Wert: " + getValue());
            }
            if (value >= minimum && value <= maximum) {
                return value;
            }
            throw new IllegalArgumentException(
                "Argumentwert " + getValue() + "von Argument " + getName() + " liegt nicht zwischen " + minimum + " und " + maximum);
        }

        /**
         * Gibt den Wert des Arguments als {@code double} Wert zurück. Der Argumentwert wird in einen Zahlwert vom Typ {@code double} konvertiert und
         * überprüft, ob der Wert nicht ausserhalb der angegebenen Grenzen liegt.
         *
         * @param minimum Kleinster erlaubter Wert.
         * @param maximum Größter erlaubter Wert.
         *
         * @return Der konvertierte Argumentwert.
         *
         * @throws IllegalArgumentException Wenn der Wert kleiner als das Minimum oder größer als das Maximum ist.
         * @throws NumberFormatException    Wenn der Argumentwert nicht konvertiert werden konnte.
         * @throws IllegalStateException    Wenn das Argument keinen Wert hat.
         */
        public double doubleValueBetween(double minimum, double maximum) throws NumberFormatException {
            double value;
            try {
                value = Double.parseDouble(getValue());
            } catch (Exception e) {
                throw new NumberFormatException("Argument " + getName() + " hat keinen Double-Wert: " + getValue());
            }
            if (value >= minimum && value <= maximum) {
                return value;
            }
            throw new IllegalArgumentException(
                "Argumentwert " + getValue() + "von Argument " + getName() + " liegt nicht zwischen " + minimum + " und " + maximum);
        }

        /**
         * Gibt den Wert des Arguments als Datei-Identifikation zurück. Der Argumentwert wird als Dateiname der Datei interpretiert.
         *
         * @return Dateiobjekt zum Zugriff auf die durch den Argumentwert identifizierten Datei.
         *
         * @throws IllegalStateException Wenn das Argument keinen Wert hat.
         */
        public File asFile() {
            return new File(getValue());
        }

        /**
         * Gibt den Wert des Arguments als Datei-Identifikation zurück. Der Argumentwert wird als Dateiname der Datei interpretiert.
         *
         * @return Dateiobjekt zum Zugriff auf die durch den Argumentwert identifizierten Datei.
         *
         * @throws IllegalStateException Wenn das Argument keinen Wert hat.
         */
        public Path asPath() {
            return Paths.get(_value);
        }

        /**
         * Gibt den Wert des Arguments als Datei-Identifikation einer existierenden Datei zurück. Der Argumentwert wird als Dateiname der Datei
         * interpretiert. Es wird geprüft, ob die Datei existiert.
         *
         * @return Dateiobjekt zum Zugriff auf die durch den Argumentwert identifizierten Datei.
         *
         * @throws IllegalArgumentException Wenn die identifizierte Datei nicht existiert.
         * @throws IllegalStateException    Wenn das Argument keinen Wert hat.
         */
        public File asExistingFile() {
            File file = asFile();
            if (!file.exists()) {
                throw new IllegalArgumentException(
                    "Argument " + getName() + ": Datei existiert nicht: " + getValue() + ", Pfad: " + file.getAbsolutePath());
            }
            return file;
        }

        /**
         * Gibt den Wert des Arguments als Datei-Identifikation einer lesbaren Datei zurück. Der Argumentwert wird als Dateiname der Datei
         * interpretiert. Es wird geprüft, ob die Datei existiert und ob ein lesender Zugriff erlaubt ist.
         *
         * @return Dateiobjekt zum Zugriff auf die durch den Argumentwert identifizierten Datei.
         *
         * @throws IllegalArgumentException Wenn die identifizierte Datei nicht existiert oder nicht lesbar ist.
         * @throws IllegalStateException    Wenn das Argument keinen Wert hat.
         */
        public File asReadableFile() {
            File file = asExistingFile();
            if (!file.canRead()) {
                throw new IllegalArgumentException("Argument " + getName() + ": Datei ist nicht lesbar: " + getValue());
            }
            return file;
        }

        /**
         * Gibt den Wert des Arguments als Datei-Identifikation einer beschreibaren Datei zurück. Der Argumentwert wird als Dateiname der Datei
         * interpretiert. Es wird geprüft, ob die Datei existiert und ob ein schreibender Zugriff erlaubt ist.
         *
         * @return Dateiobjekt zum Zugriff auf die durch den Argumentwert identifizierten Datei.
         *
         * @throws IllegalArgumentException Wenn die identifizierte Datei nicht existiert oder nicht beschreibar ist.
         * @throws IllegalStateException    Wenn das Argument keinen Wert hat.
         */
        public File asWritableFile() {
            File file = asExistingFile();
            if (!file.canWrite()) {
                throw new IllegalArgumentException("Argument " + getName() + ": Datei ist nicht beschreibar: " + getValue());
            }
            return file;
        }

        /**
         * Gibt den Wert des Arguments als Datei-Identifikation einer beschreibaren Datei zurück. Der Argumentwert wird als Dateiname der Datei
         * interpretiert. Es wird geprüft, ob die Datei existiert und ob ein schreibender Zugriff erlaubt ist.
         *
         * @param createIfNotExistent Wenn die spezifizierte Datei nicht existiert und dieser Parameter den Wert {@code true} hat, wird eine neue
         *                            Datei erzeugt.
         *
         * @return Dateiobjekt zum Zugriff auf die durch den Argumentwert identifizierten Datei.
         *
         * @throws IllegalArgumentException Wenn die identifizierte Datei nicht existiert und nicht angelegt werden sollte oder nicht beschreibar
         *                                  ist.
         * @throws IOException              Wenn die Datei nicht angelegt werden konnte.
         * @throws IllegalStateException    Wenn das Argument keinen Wert hat.
         */
        public File asWritableFile(boolean createIfNotExistent) throws IOException {
            File file = asFile();
            if (createIfNotExistent && !file.exists()) {
                file.createNewFile();
            }
            return asWritableFile();
        }

        /**
         * Gibt den Wert des Arguments als Datei-Identifikation einer änderbaren Datei zurück. Der Argumentwert wird als Dateiname der Datei
         * interpretiert. Es wird geprüft, ob die Datei existiert und ob ein lesender und schreibender Zugriff erlaubt ist.
         *
         * @return Dateiobjekt zum Zugriff auf die durch den Argumentwert identifizierten Datei.
         *
         * @throws IllegalArgumentException Wenn die identifizierte Datei nicht existiert oder nicht nicht lesbar oder nicht beschreibar ist.
         * @throws IllegalStateException    Wenn das Argument keinen Wert hat.
         */
        public File asChangeableFile() {
            File file = asReadableFile();
            if (!file.canWrite()) {
                throw new IllegalArgumentException("Argument " + getName() + ": Datei ist nicht beschreibar: " + getValue());
            }
            return file;
        }

        /**
         * Gibt den Wert des Arguments als Datei-Identifikation einer änderbaren Datei zurück. Der Argumentwert wird als Dateiname der Datei
         * interpretiert. Es wird geprüft, ob die Datei existiert und ob ein lesender und schreibender Zugriff erlaubt ist.
         *
         * @param createIfNotExistent Wenn die spezifizierte Datei nicht existiert und dieser Parameter den Wert {@code true} hat, wird eine neue
         *                            Datei erzeugt.
         *
         * @return Dateiobjekt zum Zugriff auf die durch den Argumentwert identifizierten Datei.
         *
         * @throws IllegalArgumentException Wenn die identifizierte Datei nicht existiert und nicht angelegt werden sollte oder nicht lesbar oder
         *                                  nicht beschreibar ist.
         * @throws IOException              Wenn die Datei nicht angelegt werden konnte.
         * @throws IllegalStateException    Wenn das Argument keinen Wert hat.
         */
        public File asChangeableFile(boolean createIfNotExistent) throws IOException {
            File file = asFile();
            if (createIfNotExistent && !file.exists()) {
                file.createNewFile();
            }
            return asReadableFile();
        }

        /**
         * Gibt den Wert des Arguments als Datei-Identifikation eines Dateiverzeichnisses zurück. Der Argumentwert wird als Dateiname des
         * Dateiverzeichnisses interpretiert. Es wird geprüft, ob die spezifizierte Datei existiert und ein Verzeichnis ist.
         *
         * @return Dateiobjekt zum Zugriff auf das durch den Argumentwert identifizierte Dateiverzeichnis.
         *
         * @throws IllegalArgumentException Wenn die identifizierte Datei nicht existiert oder kein Verzeichnis ist.
         * @throws IllegalStateException    Wenn das Argument keinen Wert hat.
         */
        public File asDirectory() {
            File file = asExistingFile();
            if (file.isDirectory()) {
                return file;
            }
            throw new IllegalArgumentException("Argument " + getName() + ": Kein Dateiverzeichnis: " + getValue());
        }

        /**
         * Bestimmt den Wert des Arguments als Zeichenkette.
         *
         * @return Wert des Arguments.
         *
         * @throws IllegalStateException Wenn das Argument keinen Wert hat.
         */
        public String asString() {
            return getValue();
        }

        /**
         * Bestimmt den Wert des Arguments als nicht leere Zeichenkette.
         *
         * @return Wert des Arguments.
         *
         * @throws IllegalArgumentException Wenn der Argumentwert leer ist.
         * @throws IllegalStateException    Wenn das Argument keinen Wert hat.
         */
        public String asNonEmptyString() {
            String value = getValue();
            if (value.isEmpty()) {
                throw new IllegalArgumentException("Argument " + getName() + " darf nicht leer sein");
            }
            return value;
        }

        public ValueCase asValueCase(ValueSelection validValues) {
            String value = getValue();
            ValueCase valueCase = validValues.get(value);
            if (valueCase != null) {
                return valueCase;
            }
            throw new IllegalArgumentException(
                "Argument " + getName() + " hat einen ungültigen Wert: \"" + value + "\", erlaubt sind folgende Werte:" + validValues.getInfo());
        }

        /**
         * Bestimmt den Wert des Arguments als Enum-Konstante.
         *
         * @param typeClass Klasse von dem der Enum-Wert eingelesen werden soll. Unterstützt native Enum-Klassen und Enum-ähnliche Klassen, mit festen
         *                  öffentlichen Konstanten. Groß- und Kleinschreibung wird ignoriert.
         *
         * @return Wert des Arguments.
         *
         * @throws IllegalArgumentException Wenn der Argumentwert leer oder ungültig ist.
         */
        public <E> E asEnum(final Class<E> typeClass) {
            final List<String> validValues = new ArrayList<>();
            String value = getValue();
            E[] enumConstants = typeClass.getEnumConstants();
            if (enumConstants != null) {
                for (E e : enumConstants) {
                    if (((Enum<?>) e).name().equalsIgnoreCase(value)) {
                        return e;
                    }
                    if (e.toString().equalsIgnoreCase(value)) {
                        return e;
                    }
                    validValues.add(e.toString());
                }
            } else {
                try {
                    Field[] fields = typeClass.getFields();
                    for (Field field : fields) {
                        if (Modifier.isStatic(field.getModifiers()) && Modifier.isFinal(field.getModifiers()) && field.getType().equals(typeClass)) {
                            Object fieldValue = field.get(null);
                            if (field.getName().equalsIgnoreCase(value)) {
                                return (E) fieldValue;
                            }
                            if (fieldValue.toString().equalsIgnoreCase(value)) {
                                return (E) fieldValue;
                            }
                            validValues.add(fieldValue.toString());
                        }
                    }
                } catch (IllegalAccessException e) {
                    e.printStackTrace();
                }
            }
            throw new IllegalArgumentException(
                "Argument " + getName() + " hat einen ungültigen Wert: \"" + value + "\", erlaubt sind folgende Werte: " + validValues);
        }

        /**
         * Interpretiert den Wert des Arguments als Zeitangabe. Erkannt werden absolute Zeitangabe wie in der Methode {@link #asAbsoluteTime} und
         * relative Zeitangaben wie in der Methode @{link #asRelativeTime}. Wenn eine relative Zeitangabe angegeben wurde, wird der angegebene Wert
         * vom aktuellen Zeitpunkt abgezogen d.h. das Ergebnis liegt bei positiven Angaben in der Vergangenheit liegt.
         *
         * @return Anzahl Millisekunden seit 1970.
         */
        public long asTime() {
            try {
                return asAbsoluteTime();
            } catch (Exception e) {
                return -asRelativeTime() + System.currentTimeMillis();
            }
        }

        /**
         * Interpretiert den Wert des Arguments als absolute Zeitangabe. Das Argument muss aus einer Zeitangabe im Format HH:MM[:SS[,mmm]] und/oder
         * aus einer Datumsangabe im Format TT.MM.[YY]YY bestehen. Die Reihenfolge von Datum und Zeit ist egal. Wenn nur eine Zeitangabe im Argument
         * enthalten ist, wird als Datum der Tag benutzt, an dem das Programm gestartet wurde. Wenn nur eine Datumsangabe im Argument enthalten ist,
         * dann wird als Zeitangabe 0:00 Uhr Lokalzeit benutzt. Bei Datumsangaben mit zweistelliger Jahreszahl wird ein Jahr gewählt, das im Bereich
         * von 80 Jahren vor und 20 Jahren nach dem aktuellen Jahr liegt. Als spezieller Wert wird der Text "jetzt" erkannt und durch die beim Aufruf
         * der Methode aktuelle Zeit interpretiert.
         *
         * @return Anzahl Millisekunden seit 1970.
         */
        public long asAbsoluteTime() {
            DateFormat format;
            Date date;
            String value = getValue();
            if (value.toLowerCase().equals("jetzt")) {
                return System.currentTimeMillis();
            }
            value = value.replace('-', ' ');
            for (final DateFormat _parseDateFormat : _parseDateFormats) {
                format = _parseDateFormat;
                try {
                    synchronized (format) {
                        date = format.parse(value);
                    }
                    return date.getTime();
                } catch (ParseException e) {
                    //continue with next Format
                }
            }
            for (final DateFormat _parseTimeFormat : _parseTimeFormats) {
                format = _parseTimeFormat;
                try {
                    synchronized (format) {
                        date = format.parse(getValue());
                    }
                    //long today= (System.currentTimeMillis() / (1000 * 60 * 60 * 24) ) * (1000 * 60 * 60 * 24);
                    return date.getTime() + _startDay;
                } catch (ParseException e) {
                    //continue with next Format
                }
            }
            throw new IllegalArgumentException("keine absolute Zeitangabe");
        }

        /**
         * Interpretiert den Wert des Arguments als relative Zeitangabe. Das Argument muss aus einer Liste von Zahlen und Einheiten bestehen. Als
         * Einheiten sind "t" und "Tag[e]" für Tage, "h" und "Stunde[n]" für Stunden, "m" und "Minute[n]" für Minuten, "s" und "Sekunde[n]" für
         * Sekunden sowie "ms" und "Millisekunden[e]" für Millisekunden erkannt. Die Einzelnen Werte werden in Millisekunden umgerechnet und
         * aufsummiert. Als spezieller Wert wird der Text "jetzt" erkannt und als "0 Sekunden" interpretiert.
         *
         * @return Relative Zeitangabe in Millisekunden.
         */
        public long asRelativeTime() {
            String value = getValue();
            if (value.toLowerCase().equals("jetzt")) {
                return 0;
            }
            String[] splitted = value.trim().split(_relPattern);
            long number = 0;
            long millis = 0;
            for (int i = 0; i < splitted.length; ++i) {
                String word = splitted[i];
                number = Long.decode(word);
                if (++i < splitted.length) {
                    word = splitted[i].toLowerCase();
                    if (word.equals("t") || word.startsWith("tag")) {
                        millis += (1000 * 60 * 60 * 24) * number;
                    } else if (word.equals("h") || word.startsWith("stunde")) {
                        millis += (1000 * 60 * 60) * number;
                    } else if (word.equals("m") || word.startsWith("minute")) {
                        millis += (1000 * 60) * number;
                    } else if (word.equals("s") || word.startsWith("sekunde")) {
                        millis += 1000 * number;
                    } else if (word.equals("ms") || word.startsWith("milli")) {
                        millis += number;
                    } else {
                        throw new IllegalArgumentException("Ungültige relative Zeitangabe: " + splitted[i]);
                    }
                } else if (number != 0) {
                    throw new IllegalArgumentException("Einheit bei relativer Zeitangabe fehlt");
                }
            }
            return millis;
        }

        /**
         * Interpretiert das Aufrufargument als eine Datenmenge, Beispielsweise würde "-test=10M" 10 Megabytes entsprechen.
         * <p>Diese Methode ist Fehlertolerant, folgende Werte sind gleichwertig: "10000k", "10000K", "10m", "10M", "10MB"
         * <p>
         * Diese Methode entspricht grob {@link #asSiUnit(String) asSiUnit("B")}, unterscheidet sich aber in folgenden Punkten:
         * <ul>
         *     <li>Es sind nur ganzzahlige Rückgabewerte erlaubt</li>
         *     <li>Groß- und Kleinschreibung wird ignoriert, d.h. m = Mega, nicht Milli</li>
         *     <li>Unterstützung für SI-konforme Binärpotenzen (z. B. 1KiB = 1024 bytes, 1KB = 1000 bytes)</li>
         * </ul>
         *
         * @return Anzahl Bytes
         */
        public long asByteSize() {

            BigDecimal value;
            String unit;
            try (Scanner scanner = new Scanner(_value).useLocale(Locale.GERMANY)) {
                scanner.useDelimiter(BYTE_DELIM_PATTERN);
                if (!scanner.hasNextBigDecimal()) {
                    throw new IllegalArgumentException("Ungültige Zahl: " + scanner.next());
                }
                value = scanner.nextBigDecimal();
                if (scanner.hasNextLine()) {
                    unit = scanner.nextLine();
                } else {
                    unit = "";
                }
            }

            Matcher matcher = BYTE_UNIT_PATTERN.matcher(unit);
            if (matcher.matches()) {
                switch (matcher.group(1).toLowerCase()) {
                    case "":
                        break;
                    case "k":
                        value = value.multiply(BigDecimal.valueOf(1000L));
                        break;
                    case "m":
                        value = value.multiply(BigDecimal.valueOf(1000_000L));
                        break;
                    case "g":
                        value = value.multiply(BigDecimal.valueOf(1000_000_000L));
                        break;
                    case "t":
                        value = value.multiply(BigDecimal.valueOf(1000_000_000_000L));
                        break;
                    case "p":
                        value = value.multiply(BigDecimal.valueOf(1000_000_000_000_000L));
                        break;
                    case "ki":
                        value = value.multiply(BigDecimal.valueOf(1024L));
                        break;
                    case "mi":
                        value = value.multiply(BigDecimal.valueOf(1024L * 1024L));
                        break;
                    case "gi":
                        value = value.multiply(BigDecimal.valueOf(1024L * 1024L * 1024L));
                        break;
                    case "ti":
                        value = value.multiply(BigDecimal.valueOf(1024L * 1024L * 1024L * 1024L));
                        break;
                    case "pi":
                        value = value.multiply(BigDecimal.valueOf(1024L * 1024L * 1024L * 1024L * 1024L));
                        break;
                    default:
                        throw new IllegalArgumentException("Ungültige Einheit: " + matcher.group(1));
                }
                try {
                    return value.toBigInteger().longValueExact();
                } catch (ArithmeticException e) {
                    throw new IllegalArgumentException("Wert außerhalb des erlaubten Bereichs: " + value, e);
                }
            } else {
                throw new IllegalArgumentException("Ungültige Byte-Angabe: " + _value);
            }
        }

        /**
         * Gibt den Wert dieses Arguments als Wert einer SI-Einheit zurück. Wird zum Beispiel ein Argument mit {@code asSiUnit("m")} angefragt, dann
         * führen folgende Eingaben zu den entsprechenden Ergebnissen:
         * <ul>
         *     <li>{@code "1m" → 1.0}</li>
         *     <li>{@code "1km" → 1000.0}</li>
         *     <li>{@code "1mm" → 0.001}</li>
         *     <li>{@code "1cm" → 0.01}</li>
         * </ul>
         * Die Basis-Einheit kann vom Benutzer weggelassen werden, sofern dadurch keine Unklarheit entsteht:
         * <ul>
         *     <li>{@code "1" → 1.0}</li>
         *     <li>{@code "1m" → 1.0} (!!!)</li>
         *     <li>{@code "1mm" → 0.001}</li>
         *     <li>{@code "1k" → 1000.0}</li>
         *     <li>{@code "1c" → 0.01}</li>
         * </ul>
         * Der Programmierer kann den Wert auch in einer bereits skalierten Einheit abfragen. Wird zum Beispiel ein Argument mit {@code asSiUnit
         * ("cm")} angefragt, dann führen folgende Eingaben zu den entsprechenden Ergebnissen:
         * <ul>
         *     <li>{@code "1m" → 100.0}</li>
         *     <li>{@code "1km" → 100000.0}</li>
         *     <li>{@code "1mm" → 0.1}</li>
         *     <li>{@code "1cm" → 1.0}</li>
         * </ul>
         * Groß-und Kleinschreibung der Einheiten ist in beiden Fällen wesentlich (Beispiel: m = Milli, M = Mega)
         *
         * @param siUnit Von Programm geforderte Einheit
         *
         * @return Wert, der in die gewünschte Einheit umgerechnet wurde. Bietet dank {@link BigDecimal} hier unbegrenzte Genauigkeit.
         */
        public BigDecimal asSiUnit(String siUnit) {
            int shift = 0;
            if (SI_UNIT_PATTERN.matcher(siUnit).matches()) {
	            shift = switch (siUnit.substring(0, 1)) {
		            case "k" -> 3;
		            case "M" -> 6;
		            case "G" -> 9;
		            case "T" -> 12;
		            case "P" -> 15;
		            case "c" -> -2;
		            case "m" -> -3;
		            case "u", "µ" -> -6;
		            case "n" -> -9;
		            case "p" -> -12;
		            case "f" -> -15;
		            default -> throw new IllegalArgumentException("Ungültige Einheit: " + siUnit);
	            };
                siUnit = siUnit.substring(1);
            }
            Pattern unitPattern = Pattern.compile("\\s*([a-zA-Zµ]{0,2}?)(?:" + Pattern.quote(siUnit) + ")?");
            Pattern delimPattern = Pattern.compile("\\s*(?=[a-zA-Zµ]{0,2}?(?:" + Pattern.quote(siUnit) + ")?$)");

            BigDecimal value;
            String unit;
            try (Scanner scanner = new Scanner(_value).useLocale(Locale.GERMANY)) {
                scanner.useDelimiter(delimPattern);
                if (!scanner.hasNextBigDecimal()) {
                    throw new IllegalArgumentException("Ungültige Zahl: " + scanner.next());
                }
                value = scanner.nextBigDecimal();
                if (scanner.hasNextLine()) {
                    unit = scanner.nextLine();
                } else {
                    unit = "";
                }
            }
            Matcher matcher = unitPattern.matcher(unit);
            if (matcher.matches()) {
                switch (matcher.group(1)) {
                    case "":
                        break;
                    case "k":
                        value = value.scaleByPowerOfTen(3);
                        break;
                    case "M":
                        value = value.scaleByPowerOfTen(6);
                        break;
                    case "G":
                        value = value.scaleByPowerOfTen(9);
                        break;
                    case "T":
                        value = value.scaleByPowerOfTen(12);
                        break;
                    case "P":
                        value = value.scaleByPowerOfTen(15);
                        break;
                    case "c":
                        value = value.scaleByPowerOfTen(-2);
                        break;
                    case "m":
                        value = value.scaleByPowerOfTen(-3);
                        break;
                    case "u":
                    case "µ":
                        value = value.scaleByPowerOfTen(-6);
                        break;
                    case "n":
                        value = value.scaleByPowerOfTen(-9);
                        break;
                    case "p":
                        value = value.scaleByPowerOfTen(-12);
                        break;
                    case "f":
                        value = value.scaleByPowerOfTen(-15);
                        break;
                    default:
                        throw new IllegalArgumentException("Ungültige Einheit: " + matcher.group(1) + " erwartet: " + siUnit);
                }
                return value.scaleByPowerOfTen(-shift);
            } else {
                throw new IllegalArgumentException("Ungültige Einheit: " + unit);
            }
        }

        /**
         * Interpretiert den Wert des Arguments als Zeitdauer (Klasse {@link Duration}. Diese Methode liefert im Gegensatz zu {@link
         * #asRelativeTime()} eine {@link Duration} zurück und interpretiert allgemein mehr Formate. Beispielsweise werden folgende Werte
         * unterstützt:
         * <ul>
         *     <li>{@code 12:55:12,231}</li>
         *     <li>{@code 12:55}</li>
         *     <li>{@code 3 Tage 5 Minuten 2 Sekunden}</li>
         *     <li>{@code 3 Tage, 12:55:12,231}</li>
         * </ul>
         * <p>
         * Der Wert "jetzt" wird nicht unterstützt.
         *
         * @return Zeitdauer
         */
        public Duration asDuration() {
            String trim = _value.trim();

            Matcher matcher = DURATION_PATTERN.matcher(trim);
            if (!matcher.matches()) {
                matcher = DURATION_PATTERN_2.matcher(trim);
                if (!matcher.matches()) {
                    throw new IllegalArgumentException("Unbekanntes Format für eine Dauer: " + _value);
                }
            }
            long millis = 0;
            millis += addDuration(matcher.group(1), 86400000);
            millis += addDuration(matcher.group(2), 3600000);
            millis += addDuration(matcher.group(3), 60000);
            millis += addDuration(matcher.group(4), 1000);
            if (matcher.group(5) != null) {
                millis += addDuration("," + matcher.group(5), 1000);
            }
            return Duration.ofMillis(millis);
        }

        /**
         * Gibt das angegebene Datum zurück, im Format dd.MM.yyyy.
         * <p>
         * Das Jahr kann statt 4-stellig auch zweistellig angegeben werden, was aber nicht empfohlen wird, da Mehrdeutigkeiten entstehen können. Die
         * verwendete Bibliothek interpretiert 87 bspw. als 2087, nicht als 1987.
         *
         * @return Datum
         */
        public LocalDate asLocalDate() {
            return tryParse(it -> LocalDate.parse(_value, it), DateTimeFormatter.ofPattern("dd.MM.yyyy", Locale.GERMAN),
                            DateTimeFormatter.ofPattern("dd.MM.yy", Locale.GERMAN));
        }

        /**
         * Gibt die angegebene Zeit zurück, im Format HH:mm:ss,SSS. Die Millisekunden und Sekunden können optional weggelassen werden.
         *
         * @return Zeit
         */
        public LocalTime asLocalTime() {
            return tryParse(it -> LocalTime.parse(_value, it), DateTimeFormatter.ofPattern("HH:mm:ss,SSS", Locale.GERMAN),
                            DateTimeFormatter.ofPattern("HH:mm:ss", Locale.GERMAN), DateTimeFormatter.ofPattern("HH:mm", Locale.GERMAN));
        }

        /**
         * Ermittelt ein Datum als {@link LocalDateTime}. Diese Klasse unterstützt folgende Formate:
         * <ul>
         *     <li>{@code dd.MM.yyyy, HH:mm:ss,SSS}</li>
         *     <li>{@code dd.MM.yyyy, HH:mm:ss}</li>
         *     <li>{@code dd.MM.yyyy, HH:mm}</li>
         *     <li>{@code HH:mm:ss,SSS, dd.MM.yyyy}</li>
         *     <li>{@code HH:mm:ss, dd.MM.yyyy}</li>
         *     <li>{@code HH:mm, dd.MM.yyyy}</li>
         * </ul>
         * <p>
         * Das Komma zwischen Datum und Zeit ist dabei optional, das Leerzeichen ist aber erforderlich.
         * <p>
         * Das Jahr kann statt 4-stellig auch zweistellig angegeben werden, was aber nicht empfohlen wird, da Mehrdeutigkeiten entstehen können. 
         * Die verwendete Bibliothek interpretiert 87 bspw. als 2087, nicht als 1987.
         *
         * @return Datum und Zeit
         */
        public LocalDateTime asLocalDateTime() {
            try {
                LocalTime time = asLocalTime();
                return time.atDate(LocalDate.now());
            } catch (Exception ignored) {
            }
            try {
                LocalDate date = asLocalDate();
                return date.atStartOfDay();
            } catch (Exception ignored) {
            }
            return tryParse(it -> LocalDateTime.parse(_value, it), DateTimeFormatter.ofPattern("dd.MM.yyyy[,] HH:mm:ss,SSS", Locale.GERMAN),
                            DateTimeFormatter.ofPattern("dd.MM.yyyy[,] HH:mm:ss", Locale.GERMAN),
                            DateTimeFormatter.ofPattern("dd.MM.yyyy[,] HH:mm", Locale.GERMAN),
                            DateTimeFormatter.ofPattern("HH:mm:ss,SSS[,] dd.MM.yyyy", Locale.GERMAN),
                            DateTimeFormatter.ofPattern("HH:mm:ss[,] dd.MM.yyyy", Locale.GERMAN),
                            DateTimeFormatter.ofPattern("HH:mm[,] dd.MM.yyyy", Locale.GERMAN),
                            DateTimeFormatter.ofPattern("dd.MM.yy[,] HH:mm:ss,SSS", Locale.GERMAN),
                            DateTimeFormatter.ofPattern("dd.MM.yy[,] HH:mm:ss", Locale.GERMAN),
                            DateTimeFormatter.ofPattern("dd.MM.yy[,] HH:mm", Locale.GERMAN),
                            DateTimeFormatter.ofPattern("HH:mm:ss,SSS[,] dd.MM.yy", Locale.GERMAN),
                            DateTimeFormatter.ofPattern("HH:mm:ss[,] dd.MM.yy", Locale.GERMAN),
                            DateTimeFormatter.ofPattern("HH:mm[,] dd.MM.yy", Locale.GERMAN));
        }

        /**
         * Gibt {@linkplain #asLocalDateTime() die angegebene Datum+Uhrzeit} als {@link Instant} zurück (in der lokalen Standardzeitzone).
         *
         * @return Angegebener Zeitpunkt
         */
        public Instant asInstant() {
            return asLocalDateTime().atZone(ZoneId.systemDefault()).toInstant();
        }

        /**
         * Erzeugt eine Zeichenkette, die den Namen und den Wert des Arguments enthält.
         *
         * @return Zeichenkette mit Name und Wert des Arguments.
         *
         * @throws IllegalStateException Wenn das Argument keinen Wert hat.
         */
        public String toString() {
            return "Argument " + _name + (hasValue() ? "=" + _value : "");
        }

    }

    public static class ValueSelection {
        List<ValueCase> _valueCases = new LinkedList<>();

        public ValueSelection() {
        }

        public ValueCase add(String caseName) {
            ValueCase valueCase = new ValueCase(caseName);
            _valueCases.add(valueCase);
            return valueCase;
        }

        ValueCase get(String caseName) {
            for (final ValueCase valueCase : _valueCases) {
                if (valueCase.matches(caseName)) {
                    return valueCase;
                }
            }
            return null;
        }

        public String toString() {
            StringBuilder result = new StringBuilder();
            result.append("ValueSelection{");
            Iterator<ValueCase> caseIterator = _valueCases.iterator();
            while (caseIterator.hasNext()) {
                result.append(caseIterator.next().toString());
                if (caseIterator.hasNext()) {
                    result.append(", ");
                }
            }
            result.append("}");
            return result.toString();
        }

        public String getInfo() {
            StringBuilder result = new StringBuilder();
            for (final ValueCase _valueCase : _valueCases) {
                result.append(_valueCase.getInfo());
            }
            return result.toString();
        }
    }

    public static class ValueCase {
        List<String> _caseNames = new LinkedList<>();
        boolean _ignoreCase;
        Object _conversion;
        String _description;

        ValueCase(String caseName) {
            alias(caseName);
        }

        public ValueCase alias(String aliasName) {
            _caseNames.add(aliasName);
            return this;
        }

        public ValueCase ignoreCase() {
            _ignoreCase = true;
            return this;
        }

        public ValueCase checkCase() {
            _ignoreCase = false;
            return this;
        }

        public ValueCase convertTo(Object object) {
            _conversion = object;
            return this;
        }

        public ValueCase convertTo(int conversionValue) {
            _conversion = conversionValue;
            return this;
        }

        public ValueCase convertTo(boolean conversionValue) {
            _conversion = conversionValue ? Boolean.TRUE : Boolean.FALSE;
            return this;
        }

        public ValueCase purpose(String description) {
            _description = description;
            return this;
        }

        public Object convert() {
            return _conversion;
        }

        public int intValue() {
            return ((Number) _conversion).intValue();
        }

        public boolean booleanValue() {
            return (Boolean) _conversion;
        }

        public boolean matches(String name) {
            if (_ignoreCase) {
                name = name.toLowerCase();
            }
            for (String caseName : _caseNames) {
                if (_ignoreCase) {
                    caseName = caseName.toLowerCase();
                }
                if (name.equals(caseName)) {
                    return true;
                }
            }
            return false;
        }

        public String toString() {
            StringBuilder result = new StringBuilder("ValueCase{");
            if (_description != null) {
                result.append('"');
                result.append(_description);
                result.append('"');
                result.append(",");
            }
            result.append(_ignoreCase ? "ignoreCase" : "checkCase");
            result.append(",match{");
            Iterator<String> nameIterator = _caseNames.iterator();
            while (nameIterator.hasNext()) {
                result.append('"');
                result.append(nameIterator.next());
                result.append('"');
                if (nameIterator.hasNext()) {
                    result.append(",");
                }
            }
            result.append("}");
            if (_conversion != null) {
                result.append(",conversion:");
                result.append(_conversion);
            }
            result.append("}");
            return result.toString();
        }

        public String getInfo() {
            StringBuilder result = new StringBuilder();
            result.append(System.getProperty("line.separator")).append("   ");
            for (final String _caseName : _caseNames) {
                result.append('"').append(_caseName).append('"').append(", ");
            }
            result.append("(").append(_ignoreCase ? "ohne" : "mit").append(" Prüfung der Groß-/Kleinschreibung)");
            if (_description != null) {
                result.append(System.getProperty("line.separator")).append("      Zweck: ").append(_description);
            }
            return result.toString();
        }
    }
}
