/*
 * Copyright 2005 by Kappich+Kniß Systemberatung Aachen (K2S)
 * Copyright 2007-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.sys.funclib.consoleProcessFrame.
 *
 * de.bsvrz.sys.funclib.consoleProcessFrame is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.sys.funclib.consoleProcessFrame is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.sys.funclib.consoleProcessFrame; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.sys.funclib.consoleProcessFrame;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Insets;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.lang.reflect.InvocationTargetException;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.function.Supplier;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextPane;
import javax.swing.SwingUtilities;
import javax.swing.text.AttributeSet;
import javax.swing.text.BadLocationException;
import javax.swing.text.DefaultStyledDocument;
import javax.swing.text.MutableAttributeSet;
import javax.swing.text.SimpleAttributeSet;
import javax.swing.text.StyleConstants;

/**
 * Implementiert ein JPanel, das mit einem externen Prozess verbunden ist. Der Prozess kann gestartet und beendet werden. Die Textausgaben des
 * Prozesses werden im JPanel dargestellt.
 *
 * @author Kappich Systemberatung
*/
public class ConsoleProcessPanel extends JPanel {

    private static final int MAX_TEXT_LENGTH = 1000000;

    private static final List<Process> _runningProcesses;

    static {
        _runningProcesses = new LinkedList<>();
        Runtime.getRuntime().addShutdownHook(new Thread(new Runnable() {
            public void run() {
                synchronized (_runningProcesses) {
                    for (Process process : _runningProcesses) {
                        process.destroy();
                    }
                }
            }
        }));
    }

    private final Process _process;

    private final JTextPane _textPane;

    private final Thread _processOutputHatch;

    private final Thread _processErrorHatch;
    private final MutableAttributeSet _stdIn = new SimpleAttributeSet();
    private final MutableAttributeSet _system = new SimpleAttributeSet();
    private Supplier<char[]> _passwordSupplier;
    private String _passwordPrompt;
    private boolean _exitPrinted;

    protected ConsoleProcessPanel(String[] commandArray, String[] environment, File workingDirectory) throws IOException {
        super(new BorderLayout());
        StyleConstants.setForeground(_stdIn, Color.blue);
        StyleConstants.setForeground(_system, Color.gray);

        _process = Runtime.getRuntime().exec(commandArray, environment, workingDirectory);
        synchronized (_runningProcesses) {
            _runningProcesses.add(_process);
        }
        _textPane = new JTextPane(new DefaultStyledDocument());
        _textPane.setEditable(false);
        _processErrorHatch = new Thread(new TextPaneHatch(_process.getErrorStream(), Color.red));
        _processOutputHatch = new Thread(new TextPaneHatch(_process.getInputStream(), _textPane.getForeground()));

        _textPane.addKeyListener(new KeyAdapter() {
            @Override
            public void keyTyped(final KeyEvent e) {
                try {
                    _textPane.setCaretPosition(_textPane.getDocument().getLength());
                    OutputStream outputStream = _process.getOutputStream();
                    outputStream.write(e.getKeyChar());
                    outputStream.flush();
                    append(String.valueOf(e.getKeyChar()), _stdIn);
                } catch (IOException ignored) {
                } finally {
                    e.consume();
                }
            }
        });
        Font font = new Font("Monospaced", Font.PLAIN, 11);
        _textPane.setFont(font);
        _textPane.setMargin(new Insets(5, 5, 5, 5));
        _textPane.setText("");
        for (String s : commandArray) {
            append(s, _system);
            append("\n", _system);
        }
        append("\n", _system);
        JScrollPane scrollpane = new JScrollPane(_textPane);
        scrollpane.setPreferredSize(new Dimension(650, 300));
        add(scrollpane, BorderLayout.CENTER);
    }

    public static ConsoleProcessPanel createProcessPanel(String[] commandArray, String[] environment, File workingDirectory) throws IOException {
        return new ConsoleProcessPanel(commandArray, environment, workingDirectory);
    }

    public static ConsoleProcessPanel createJavaProcessPanel(String className, String[] arguments, String[] environment, File workingDirectory)
        throws IOException {
        String fileSeparator = System.getProperty("file.separator");
        String javaHome = System.getProperty("java.home");
        String classPath = System.getProperty("java.class.path");

        List<String> commandList = new LinkedList<>();
        commandList.add(javaHome + fileSeparator + "bin" + fileSeparator + "java");
        commandList.add("-Dfile.encoding=UTF-8");
        commandList.add("-cp");
        commandList.add(classPath);
        commandList.add(className);
        if (arguments != null) {
            commandList.addAll(Arrays.asList(arguments));
        }

        final String[] commandArray = commandList.toArray(new String[0]);

        return createProcessPanel(commandArray, environment, workingDirectory);
    }

    public void killProcess() {
        synchronized (_runningProcesses) {
            _runningProcesses.remove(_process);
        }
        _process.destroy();
    }

    public void start() {
        _processOutputHatch.start();
        _processErrorHatch.start();
    }

    /**
     * Setzt Daten für eine Passwort-Aktualisierung der Applikation. Das Passwort wird UTF-16BE-kodiert an STDIn der Applikation übertragen. UTF-16BE
     * wurde gewählt, da alle mögliche Zeichen übertragen werden können und dadurch das Passwort nicht trivial lesbar ist wird, sollte irgendwer die
     * Übertragung beobachten können.
     *
     * @param passwordSupplier Quelle für Passwort
     * @param prompt           Markierung ab der passwort gesendet wird.
     *
     * @since 3.14
     */
    public void setPasswordSupplier(final Supplier<char[]> passwordSupplier, final String prompt) {
        _passwordSupplier = passwordSupplier;
        _passwordPrompt = prompt;
    }

    private void sendPassword() throws IOException {
        OutputStream outputStream = _process.getOutputStream();
        char[] chars = _passwordSupplier.get();

        if (chars == null) {
            killProcess();
            return;
        }
        ByteBuffer buf = StandardCharsets.UTF_16BE.encode(CharBuffer.wrap(chars));
        byte[] b = new byte[buf.remaining()];
        buf.get(b, 0, buf.remaining());
        outputStream.write(b);
        outputStream.write("\n".getBytes(StandardCharsets.UTF_16BE));
        outputStream.flush();
    }

    private void append(final String text, final AttributeSet attributeSet) {
        invokeAndWait(() -> {
            try {
                int textLength = _textPane.getDocument().getLength();
                boolean atEnd = _textPane.getSelectionStart() == textLength;
                if (textLength > MAX_TEXT_LENGTH) {
                    // Zuviel Text => obere Hälfte löschen
                    final int deleteLength = Math.max(MAX_TEXT_LENGTH / 2, textLength - MAX_TEXT_LENGTH);
                    _textPane.getStyledDocument().remove(0, deleteLength);
                    _textPane.getStyledDocument().insertString(0, "(Anfang gelöscht)...\n", _system);
                    textLength = _textPane.getDocument().getLength();
                }
                _textPane.getStyledDocument().insertString(textLength, text, attributeSet);
                if (atEnd) {
                    _textPane.setCaretPosition(_textPane.getDocument().getLength());
                }
            } catch (BadLocationException e) {
                throw new AssertionError(e);
            }
        });
    }

    private void invokeAndWait(Runnable runnable) {
        if (SwingUtilities.isEventDispatchThread()) {
            runnable.run();
        } else {
            try {
                SwingUtilities.invokeAndWait(runnable);
            } catch (InvocationTargetException e) {
                e.printStackTrace();
                throw new RuntimeException(e);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }
    }

    private class TextPaneHatch implements Runnable {

        private final BufferedReader _inputReader;
        private final MutableAttributeSet _attributeSet = new SimpleAttributeSet();

        public TextPaneHatch(InputStream inputStream, final Color color) {
            _inputReader = new BufferedReader(new InputStreamReader(inputStream, StandardCharsets.UTF_8));
            StyleConstants.setForeground(_attributeSet, color);
        }

        public void run() {
            try {
                char[] buffer = new char[1000];
                while (true) {
                    int got = _inputReader.read(buffer);
                    if (got < 0) {
                        invokeAndWait(() -> {
                            if (!_exitPrinted) {
                                _exitPrinted = true;
                                append("Prozess beendet.", _system);
                            }
                        });
                        break;
                    }
                    String buf = new String(buffer, 0, got);
                    if (_passwordSupplier != null && buf.endsWith(_passwordPrompt)) {
                        sendPassword();
                        continue;
                    }
                    append(buf, _attributeSet);
                }
            } catch (IOException e) {
//                e.printStackTrace();
//                System.err.println("I/O Error");
            }
        }
    }
}
