/*
 * Copyright 2006 by Kappich+Kniß Systemberatung Aachen (K2S)
 * Copyright 2006-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.puk.config.
 *
 * de.bsvrz.puk.config is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.puk.config is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.puk.config.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.puk.config.configFile.datamodel;

import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.ConfigurationArea;
import de.bsvrz.dav.daf.main.config.ConfigurationChangeException;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectInfo;
import de.bsvrz.dav.daf.main.config.SystemObjectType;

/**
 * Diese abstrakte Klasse implementiert die Methoden des {@link SystemObject}-Interfaces, welche für die verschiedenen Implementierungen eines
 * SystemObjekts immer gleich bleiben.
 *
 * @author Stephan Homeyer (sth), Kappich Systemberatung
 * @version $Revision$ / $Date$ / ($Author$)
 */
public abstract class AbstractConfigSystemObject implements SystemObject {
    /**
     * Der Konfigurationsbereich dieses SystemObjekts.
     */
    private final ConfigConfigurationArea _configurationArea;

    /**
     * Standardkonstruktor. Speichert den Konfigurationsbereich, zu dem dieses System-Objekt gehört.
     *
     * @param configurationArea der Konfigurationsbereich dieses System-Objekts
     */
    public AbstractConfigSystemObject(ConfigurationArea configurationArea) {
        _configurationArea = (ConfigConfigurationArea) configurationArea;
    }

    @Override
    public String getNameOrPidOrId() {
        String result = getName();
        if (result.isEmpty()) {
            result = getPid();
        }
        if (result.isEmpty()) {
            result = Long.toString(getId());
        }
        return result;
    }

    @Override
    public String getPidOrNameOrId() {
        String result = getPid();
        if (result.isEmpty()) {
            result = getName();
        }
        if (result.isEmpty()) {
            result = Long.toString(getId());
        }
        return result;
    }

    @Override
    public String getPidOrId() {
        String result = getPid();
        if (result.isEmpty()) {
            result = Long.toString(getId());
        }
        return result;
    }

    @Override
    public ConfigConfigurationArea getConfigurationArea() {
        return _configurationArea;
    }

    @Override
    public ConfigDataModel getDataModel() {
        return _configurationArea.getDataModel();
    }

    @Override
    public SystemObjectInfo getInfo() {
        final Data data = getConfigurationData(getDataModel().getAttributeGroup("atg.info"));
        if (data != null) {
            return new SystemObjectInfo(data.getTextValue("kurzinfo").getText(), data.getTextValue("beschreibung").getText());
        } else {
            return SystemObjectInfo.UNDEFINED;
        }
    }

    @Override
    public Data getConfigurationData(AttributeGroup atg) {
        return getConfigurationData(atg, getDataModel().getAspect("asp.eigenschaften"));
    }

    @Override
    public void setConfigurationData(AttributeGroup atg, Data data) throws ConfigurationChangeException {
        setConfigurationData(atg, getDataModel().getAspect("asp.eigenschaften"), data);
    }

    @Override
    public boolean isOfType(String typePid) {
        return isOfType(getDataModel().getType(typePid));
    }

    @Override
    public boolean isOfType(SystemObjectType type) {
        if (type.equals(getType())) {
            return true;
        }
        return getType().inheritsFrom(type);
    }

    /**
     * Der Vergleich zweier SystemObjekte (o1, o) erfolgt durch deren ID.
     *
     * @param o zu vergleichendes SystemObjekt
     *
     * @return {@code -1}, falls {@code o1.getId() < o.getId()}<br> {@code 1}, falls {@code o1.getId() > o.getId()}<br> {@code 0}, falls {@code
     *     o1.getId() == o.getId()}
     */
    @Override
    public int compareTo(Object o) {
        SystemObject otherObject = (SystemObject) o;
        return Long.compare(getId(), otherObject.getId());
    }

    /**
     * Gibt die String-Repräsentation dieses SystemObjekts zurück.
     *
     * @return die String-Repräsentation dieses SystemObjekts
     */
    public String toString() {
        final StringBuilder text = new StringBuilder();
        final String name = getName();
        final String pid = getPid();
        if (name != null && !name.isEmpty()) {
            text.append(name).append(" ");
        }
        if (pid != null && !pid.isEmpty()) {
            text.append("(").append(pid).append(")");
        } else {
            text.append("[").append(getId()).append("]");
        }
        return text.toString();
    }
}
