/*
 * Copyright 2006-2020 by Kappich Systemberatung, Aachen
 * Copyright 2021 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.puk.config.
 *
 * de.bsvrz.puk.config is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.puk.config is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.puk.config.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.puk.config.main.authentication;

import de.bsvrz.dav.daf.communication.protocol.UserLogin;
import de.bsvrz.dav.daf.communication.srpAuthentication.SrpClientAuthentication;
import de.bsvrz.dav.daf.communication.srpAuthentication.SrpCryptoParameter;
import de.bsvrz.dav.daf.communication.srpAuthentication.SrpUtilities;
import de.bsvrz.dav.daf.communication.srpAuthentication.SrpVerifierAndUser;
import de.bsvrz.dav.daf.communication.srpAuthentication.SrpVerifierData;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.DataAndATGUsageInformation;
import de.bsvrz.dav.daf.main.authentication.ClientCredentials;
import de.bsvrz.dav.daf.main.config.AttributeGroupUsage;
import de.bsvrz.dav.daf.main.config.ConfigurationArea;
import de.bsvrz.dav.daf.main.config.ConfigurationChangeException;
import de.bsvrz.dav.daf.main.config.ConfigurationTaskException;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.DynamicObjectType;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.dav.daf.main.impl.config.request.RequestException;
import de.bsvrz.sys.funclib.crypt.EncryptDecryptProcedure;
import de.bsvrz.sys.funclib.crypt.decrypt.DecryptFactory;
import de.bsvrz.sys.funclib.crypt.encrypt.EncryptFactory;
import de.bsvrz.sys.funclib.dataSerializer.Deserializer;
import de.bsvrz.sys.funclib.dataSerializer.NoSuchVersionException;
import de.bsvrz.sys.funclib.dataSerializer.SerializingFactory;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.filelock.FileLock;
import de.bsvrz.sys.funclib.xmlSupport.CountingErrorHandler;
import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.math.BigInteger;
import java.net.URI;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Random;
import java.util.Set;
import java.util.stream.Collectors;
import javax.crypto.Mac;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import org.w3c.dom.Document;
import org.w3c.dom.DocumentType;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * Diese Klasse stellt alle Methoden zur Verfügung, um die Benutzer eines Datenverteilers eindeutig zu identifizieren. Es werden weitere Methoden zur
 * Verfügung gestellt, um die Benutzer zu verwalten (anlegen neuer Benutzer, Passwörter ändern, usw.).
 * <p>
 * Die Klasse verwaltet selbstständig die Datei, in der die Benutzer mit ihrem Passwörtern (normales Passwort und Einmal-Passwörter) und ihren Rechten
 * gespeichert sind.
 * <p>
 * Der Klasse werden nur verschlüsselte Aufträge übergeben und sie entschlüsselt diese automatisch und führt die Aufträge aus, falls der Benutzer die
 * nötigen Rechte besitzt.
 *
 * @author Achim Wullenkord (AW), Kappich Systemberatung
 * @version $Revision:5077 $ / $Date:2007-09-02 14:48:31 +0200 (So, 02 Sep 2007) $ / ($Author:rs $)
 */
public class ConfigAuthentication implements Authentication {

    /** DebugLogger für Debug-Ausgaben */
    private static final Debug _debug = Debug.getLogger();
    /**
     * Geheimer Zufallstext, der für die Erzeugung von Fake-Verifiern benutzt wird
     */
    private static final String _secretToken = new BigInteger(64, new SecureRandom()).toString(16);
    /** Als Schlüssel dient der Benutzername (String) als Value werden alle Informationen, die zu einem Benutzer gespeichert wurden, zurückgegeben. */
    private final Map<String, UserAccount> _userAccounts = new HashMap<>();
    /** XML-Datei, wird zum anlagen einer Sicherheitskopie gebraucht */
    private final File _xmlFile;
    /** Repräsentiert die vollständige XML-Datei. */
    private final Document _xmlDocument;
    /** Speichert die Basis der Verzeichnisse für die Konfigurationsbereiche. */
    private final URI _uriBase;
    /**
     * Diese Liste speichert alle Texte, die mit {@link #getText} erzeugt wurden. Die Texte werden immer an das Ender der Liste eingefügt. Wird ein
     * Text empfangen, wird dieser aus der Liste gelöscht.
     * <p>
     * Erreicht die eine bestimmte Größe, wird das erste Element gelöscht, da das erste Element am längsten in der Liste vorhanden ist.
     * <p>
     * Die Liste ist nicht synchronisiert.
     */
    private final LinkedList<String> _randomText = new LinkedList<>();
    /** Wird benötigt um bei den entsprechenden Konfigurationsbereichen neue Benutzer anzulegen */
    private final DataModel _dataModel;
    private final FileLock _lockAuthenticationFile;

    /**
     * Lädt alle Informationen aus der angegebenen Datei. Ist die Datei nicht vorhanden, wird eine Datei mit allen Grundeinstellungen erzeugt.
     *
     * @param userFile XML-Datei, in der alle Benutzer gespeichert sind.
     */
    public ConfigAuthentication(File userFile, DataModel dataModel) throws ParserConfigurationException {

        // Die Datei gegen doppelten Zugriff sichern
        _lockAuthenticationFile = new FileLock(userFile);
        try {
            _lockAuthenticationFile.lock();
        } catch (IOException e) {
            final String errorMessage =
                "IOException beim Versuch die lock-Datei zu schreiben. Datei, die gesichert werden sollte " + userFile.getAbsolutePath();
            e.printStackTrace();
            _debug.error(errorMessage, e);
            throw new RuntimeException(errorMessage);
        }

        try {
            _xmlFile = userFile.getCanonicalFile();
        } catch (IOException e) {
            throw new IllegalArgumentException(e);
        }

        _dataModel = dataModel;

        // Es gibt die Datei, also Daten auslesen
        final CountingErrorHandler errorHandler = new CountingErrorHandler();
        final DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        factory.setNamespaceAware(true);
        factory.setIgnoringElementContentWhitespace(true);

        // die Validierung der XML-Datei anhand der DTD durchführen
        factory.setValidating(true);
        factory.setAttribute("http://xml.org/sax/features/validation", Boolean.TRUE);
        DocumentBuilder builder = factory.newDocumentBuilder();

        _debug.config("Datei wird eingelesen", _xmlFile);
        try {
            builder.setErrorHandler(errorHandler);
            builder.setEntityResolver(new ConfigAuthenticationEntityResolver());
            _xmlDocument = builder.parse(_xmlFile);    // evtl. mittels BufferedInputStream cachen
            errorHandler.printSummary();
            if (errorHandler.getErrorCount() > 0) {
                throw new RuntimeException(errorHandler.getErrorCount() + " Fehler beim Parsen der XML-Datei " + _xmlFile.toString());
            }
        } catch (Exception ex) {
            final String errorMessage = "Die Benutzerdaten der Konfiguration konnten nicht eingelesen werden: " + _xmlFile.toString();
            _debug.error(errorMessage, ex);
            throw new RuntimeException(errorMessage, ex);
        }
        _uriBase = _xmlFile.getParentFile().toURI();
        _debug.config("Verzeichnisbasis für die Benutzer der Konfiguration", _uriBase.toString());
        // Daten aus der XML-Datei einlesen
        readUserAccounts();
        _debug.config("Benutzerdaten der Konfiguration wurden vollständig eingelesen.");
    }

    /**
     * Wählt einen zufälligen Benutzer und erzeugt eine Debug-Meldung
     */
    private static SystemObject chooseRandomUser(final List<SystemObject> userList, final StringBuilder stringBuilder) {
        stringBuilder.append("\nEs sollte unter dem lokalen AOE genau ein Benutzerobjekt pro Benutzernamen geben.\nDer Benutzer \"")
            .append(userList.get(0).getPidOrId()).append("\" wurde willkürlich ausgewählt.");
        _debug.warning(stringBuilder.toString());
        return userList.get(0);
    }

    private static SrpVerifierData fakeVerifier(final String userName, final byte[] salt, final ClientCredentials clientCredentials) {
        return SrpClientAuthentication.createVerifier(SrpCryptoParameter.getDefaultInstance(), userName, clientCredentials, salt);
    }

    /**
     * Ließt alle Benutzer aus der XML-Datei ein und erzeugt entsprechende Java-Objekte. Diese werden dann in der in der Hashtable gespeichert. Die
     * Methode ist private, weil diese Funktionalität nur an dieser Stelle zur Verfügung gestellt werden soll.
     */
    private void readUserAccounts() {
        synchronized (_xmlDocument) {
            Element xmlRoot = _xmlDocument.getDocumentElement();

            NodeList entryList = xmlRoot.getElementsByTagName("benutzeridentifikation");
            for (int i = 0; i < entryList.getLength(); i++) {
                final Element element = (Element) entryList.item(i);

                final String userName = element.getAttribute("name");
                // Passwort, aus der XML-Datei. Das ist nicht in Klarschrift
                final String xmlPassword = element.getAttribute("passwort");

                // Hat der Benutzer Admin-Rechte
                final boolean admin;

                admin = "ja".equals(element.getAttribute("admin").toLowerCase());

                // Alle Einmal-Passwörter des Accounts (auch die schon benutzen)
                final List<SingleServingPassword> allSingleServingPasswords = new ArrayList<>();

                // Einmal-Passwort Liste
                final NodeList xmlSingleServingPasswordList = element.getElementsByTagName("autorisierungspasswort");

                for (int nr = 0; nr < xmlSingleServingPasswordList.getLength(); nr++) {
                    // Einmal-Passwort als XML-Objekt
                    final Element xmlSingleServingPassword = (Element) xmlSingleServingPasswordList.item(nr);

                    // Einmal-Passwort, das aus der XML Datei eingelesen wurde, keine Klarschrift
                    final String xmlSingleServingPasswort = xmlSingleServingPassword.getAttribute("passwort");
                    // Index des Passworts (Integer)
                    final int index = Integer.parseInt(xmlSingleServingPassword.getAttribute("passwortindex"));
                    // Ist das Passwort noch gültig (ja oder nein)
                    final boolean valid;
                    valid = "ja".equals(xmlSingleServingPassword.getAttribute("gueltig").toLowerCase());
                    allSingleServingPasswords.add(new SingleServingPassword(xmlSingleServingPasswort, index, valid, xmlSingleServingPassword));
                } // Alle Einmal-Passwörter

                // Alle Einmal-Passwörter wurden eingelesen

                // Alle Infos stehen zur Verfügung, das Objekt kann in die Map eingetargen werden
                final UserAccount userAccount = new UserAccount(userName, xmlPassword, admin, allSingleServingPasswords, element);

                if (_userAccounts.containsKey(userAccount.getUsername())) {
                    // Einfach das erste vorkommen überschreiben. Dieser Fall kann nur vorkommen, wenn die XML-Datei von Hand erzeugt
                    // wurde.
                    _debug.warning("Der Benutzername " + userAccount.getUsername() + " ist bereits in der Benutzerdatei vorhanden");
                }
                _userAccounts.put(userAccount.getUsername(), userAccount);
            } // Alle Accounts durchgehen
        }
    }

    @Override
    @Deprecated
    public void isValidUser(final String username, final byte[] encryptedPassword, final String authentificationText,
                            final String authentificationProcessName) throws Exception {

        // Es wird eine IllegalArgumException geworfen, wenn das Verfahren unbekannt ist oder das Verfahren nicht benutzt werden darf
        final EncryptDecryptProcedure usedDecryptProcedure = isEncryptDecryptProcedureAllowed(authentificationProcessName);

        if (_userAccounts.containsKey(username)) {
            final byte[] originalEncryptedPassword = EncryptFactory.getEncryptInstance(usedDecryptProcedure)
                .encrypt(new String(_userAccounts.get(username).getClientCredentials().getPassword()), authentificationText);

            //Prüfen, ob das Passwort übereinstimmt
            if (!Arrays.equals(encryptedPassword, originalEncryptedPassword)) {
                // Da es nicht übereinstimmt versuchen ein Einmalpasswort zu benutzen. Wenn es ein Passwort gibt, wird es benutzt und gesperrt 
                // (XML-Datei wird
                // aktualisiert). Gibt es kein Passwort, wird eine Exception geworfen.
                _userAccounts.get(username).useSingleServingPassword(encryptedPassword, authentificationText, authentificationProcessName);
            }
        } else {
            // Es gibt zu dem Benutzernamen keine Informationen, also gibt es diesen Benutzer nicht
            _debug.warning("Zu dem Benutzer '" + username + "' existiert in der benutzerverwaltung.xml keine Benutzeridentifikation");
            throw new IllegalArgumentException("Benutzername/Passwort ist falsch");
        }
    }

    @Override
    @Deprecated
    public byte[] getText() {
        final Random rand = new Random();
        final Long randomLong = rand.nextLong();

        synchronized (_randomText) {
            // Der Wert 100 wurde willkürlich gewählt
            if (_randomText.size() == 100) {
                // Die Liste wird zu gross, siehe Kommentar der Liste.
                _randomText.removeFirst();
            }
            _randomText.addLast(randomLong.toString());
            return randomLong.toString().getBytes();
        } // synch
    }

    @Override
    public void close() {
        try {
            try {
                saveXMLFile();
            } catch (TransformerException | FileNotFoundException e) {
                final String errorText = "Fehler beim Speichern der Benutzerdateien, es wird weiter versucht weitere Daten zu sichern";
                e.printStackTrace();
                _debug.error(errorText, e);
            }
        } finally {
            _lockAuthenticationFile.unlock();
        }
    }

    /**
     * Prüft, ob der übergebene Text in der Liste der zufällig erzeugten Texte {@code _randomText} vorhanden ist. Kann der Text nicht gefunden werden,
     * wird eine Exception geworfen. Konnte der Text gefunden werden, wird der Text aus der Liste entfernt.
     *
     * @param randomText Text, der in der Liste der verschickten Texte zu finden sein muss
     *
     * @throws ConfigurationTaskException Der übergebene Text konnte in der Liste der verschickten Texte nicht gefunden werden
     */
    private void checkRandomText(byte[] randomText) throws ConfigurationTaskException {
        final String randomTextString = new String(randomText);
        synchronized (_randomText) {

            if (!_randomText.remove(randomTextString)) {
                // Der übergebene Text befindet sich nicht in den verschickten Texten.
                // Dies ist ein Fehler.
                throw new ConfigurationTaskException("Annahme verweigert");
            }
        }
    }

    @Override
    public String toString() {
        return _xmlFile.toString();
    }

    /**
     * Führt einen Auftrag der Benutzerverwaltung aus und entschlüsselt dabei das übergebene Byte-Array
     *
     * @param usernameCustomer            Benutzer, der den Auftrag erteilt
     * @param encryptedMessage            verschlüsselte Aufgabe, die ausgeführt werden soll
     * @param authentificationProcessName Entschlüsselungsverfahren
     *
     * @return Die Rückgabe des ausgeführten Tasks (beispielsweise die Anzahl der verbleibenden Einmalpasswörter, falls danach gefragt wurde. {@link
     *     UserAccount#NO_RESULT} (-1) falls die Aufgabe keine Rückgabe liefert.
     *
     * @throws RequestException           Fehler in der Anfrage
     * @throws ConfigurationTaskException Fehler beim Ausführen der Anweisung
     */
    @Override
    @Deprecated
    public int processTask(String usernameCustomer, byte[] encryptedMessage, String authentificationProcessName)
        throws RequestException, ConfigurationTaskException {
        if (_userAccounts.containsKey(usernameCustomer)) {

            // Verschlüsselten Text entschlüsseln

            // Fängt alle Exceptions des Deserialisierers und wandelt sie in RequestExceptions um. Request und ConfigurationsTaskException werden 
            // durchgelassen
            try {
                // Verfahren, das benutzt werden kann. Ist das geforderte Verfahren nicht bekannt, wird eine Exception geworfen
                final EncryptDecryptProcedure encryptDecryptProcedure = isEncryptDecryptProcedureAllowed(authentificationProcessName);

                byte[] decryptedMessage;
                try {
                    String passwordString = new String(_userAccounts.get(usernameCustomer).getClientCredentials().getPassword());
                    decryptedMessage = DecryptFactory.getDecryptInstance(encryptDecryptProcedure).decrypt(encryptedMessage, passwordString);
                } catch (Exception e) {
                    // Die Nachricht konnte nicht entschlüsselt werden, z.b. weil das Passwort falsch ist
                    _debug.fine("Fehler beim Entschlüsseln der Nachricht", e);

                    throw new ConfigurationTaskException("Die Nachricht konnte nicht entschlüsselt werden (Passwort, Benutzername falsch?)");
                }

                // Serializerversion auslesen, dies steht in den ersten 4 Bytes
                final int serializerVersion = getSerializerVersion(decryptedMessage);
                decryptedMessage = removeFirst4Bytes(decryptedMessage);

                final InputStream in = new ByteArrayInputStream(decryptedMessage);
                final Deserializer deserializer = SerializingFactory.createDeserializer(serializerVersion, in);

                // In den ersten 4 Bytes steht der Nachrichtentyp
                final int messageType = deserializer.readInt();

                // In den nächsten Bytes steht ein Zufallstext, der vorher zu Applikation geschickt wurde.
                // Dieser Text muss der Konfiguration bekannt sein. Ist der Text unbekannt
                // wird eine Exception geworfen und die Verarbeitung des Pakets abgelehnt.

                // Größe der Byte-Arrays
                final int sizeOfRandomText = deserializer.readInt();
                checkRandomText(deserializer.readBytes(sizeOfRandomText));

                // Was für ein Auftrag muss ausgeführt werden
	            // Einmal-Passwort erzeugen
	            // Neuer Benutzer
	            // Passwort ändern
	            // Benutzerrechte ändern
	            // Benutzer löschen
	            // Einmalpasswörter löschen
	            //  Neuer Benutzer inklusive konfigurierender Datensätze
	            // Abfrage nach Existenz
	            // unbekannter Auftrag
	            return switch (messageType) {
		            case 1 -> {
			            createSingleServingPassword(usernameCustomer, deserializer.readString(), deserializer.readString());
			            yield UserAccount.NO_RESULT;
		            }
		            case 2 -> {
                        createNewUser(usernameCustomer, deserializer.readString(), deserializer.readString(), deserializer.readString(),
		                        deserializer.readBoolean(), deserializer.readString(), null);
			            yield UserAccount.NO_RESULT;
                    }
		            case 3 -> {
                        changeUserPassword(usernameCustomer, deserializer.readString(), deserializer.readString());
			            yield UserAccount.NO_RESULT;
                    }
		            case 4 -> {
                        changeUserRights(usernameCustomer, deserializer.readString(), deserializer.readBoolean());
			            yield UserAccount.NO_RESULT;
                    }
		            case 5 -> {
                        deleteUser(usernameCustomer, deserializer.readString());
			            yield UserAccount.NO_RESULT;
                    }
		            case 6 -> {
                        clearSingleServingPasswords(usernameCustomer, deserializer.readString());
			            yield UserAccount.NO_RESULT;
                    }
		            case 7 ->
                        // Abfrage nach Adminstatus
				            isUserAdmin(usernameCustomer, deserializer.readString()) ? 1 : 0;
		            case 8 ->
                        // Abfrage nach Anzahl der verbleibenden Einmalpasswörtern
				            countRemainingSingleServingPasswords(usernameCustomer, deserializer.readString());
		            case 9 -> {
                        createNewUser(usernameCustomer, deserializer);
			            yield UserAccount.NO_RESULT;
                    }
		            case 10 -> {
                        final String userToCheck = deserializer.readString();
			            yield isUser (userToCheck) ? 1 : 0;
                    }
		            default -> throw new ConfigurationTaskException("Unbekannte Anweisung");
	            };
            } catch (IOException e) {
                _debug.error("Fehler im Deserialisierer", e);
                throw new RequestException(e);
            } catch (NoSuchVersionException e) {
                _debug.error("Unbekannte Version", e);
                throw new RequestException(e);
            }
        } else {
            // Der Benutzer ist unbekannt
            throw new ConfigurationTaskException("Benutzer/Passwortkombination ist falsch");
        }
    }

    @Override
    public void createNewUser(final String usernameCustomer, final Deserializer deserializer)
        throws ConfigurationTaskException, RequestException, IOException {
        createNewUser(usernameCustomer, deserializer.readString(), deserializer.readString(), deserializer.readString(), deserializer.readBoolean(),
                      deserializer.readString(), readDataAndATGUsageInformation(deserializer));
    }

    @Override
    public boolean isUser(final String userToCheck) {
        return _userAccounts.containsKey(userToCheck) && userHasObject(userToCheck, "");
    }

    /**
     * Löscht für einen angegebenen Benutzer alle Einmalpasswörter bzw. markiert diese als ungültig. Nur ein Admin und der Benutzer selbst darf diese
     * Aktion ausführen.
     *
     * @param orderer  Der Auftraggeber der Aktion
     * @param username Der Benutzer, dessen Einmalpasswörter gelöscht werden sollen
     *
     * @throws FileNotFoundException
     * @throws ConfigurationTaskException
     */
    @Override
    public void clearSingleServingPasswords(final String orderer, final String username) throws FileNotFoundException, ConfigurationTaskException {
        // prüfen, ob der Benutzer diese Aktion durchführen darf
        if (isAdmin(orderer) || orderer.equals(username)) {

            if (_userAccounts.containsKey(username)) {
                try {
                    _userAccounts.get(username).clearSingleServingPasswords();
                } catch (TransformerException e) {
                    throw new ConfigurationChangeException("Konnte Einmalpasswörter nicht löschen", e);
                }
            } else {
                throw new ConfigurationTaskException("Unbekannter Benutzer");
            }
        } else {
            throw new ConfigurationTaskException("Benutzer verfügt nicht über die benötigten Rechte");
        }
    }

    /**
     * Zählt die verbleibenden Einmalpasswörter für einen angegeben Benutzer. Nur ein Admin und der Benutzer selbst darf diese Aktion ausführen.
     *
     * @param orderer  Der Auftraggeber der Aktion
     * @param username Der Benutzer, dessen Einmalpasswörter gezählt werden sollen
     *
     * @return Die Anzahl der verbliebenen Einmalpasswörter
     *
     * @throws FileNotFoundException
     * @throws ConfigurationTaskException
     */
    @Override
    public int countRemainingSingleServingPasswords(final String orderer, final String username)
        throws FileNotFoundException, ConfigurationTaskException {
        // prüfen, ob der Benutzer diese Aktion durchführen darf
        if (isAdmin(orderer) || orderer.equals(username)) {

            if (_userAccounts.containsKey(username)) {
                return _userAccounts.get(username).countSingleServingPasswords();
            } else {
                throw new ConfigurationTaskException("Unbekannter Benutzer");
            }
        } else {
            throw new ConfigurationTaskException("Benutzer verfügt nicht über die benötigten Rechte");
        }
    }

    /**
     * Gibt die verbleibenden gültigen Einmalpasswort-IDs für einen angegeben Benutzer zurück. Nur ein Admin und der Benutzer selbst darf diese Aktion
     * ausführen.
     *
     * @param orderer  Der Auftraggeber der Aktion
     * @param username Der Benutzer, dessen Einmalpasswörter gezählt werden sollen
     *
     * @return Die IDs der verbliebenen Einmalpasswörter
     *
     * @throws FileNotFoundException
     * @throws ConfigurationTaskException
     */
    @Override
    public int[] getRemainingSingleServingPasswordIDs(final String orderer, final String username)
        throws FileNotFoundException, ConfigurationTaskException {
        // prüfen, ob der Benutzer diese Aktion durchführen darf
        if (isAdmin(orderer) || orderer.equals(username)) {

            if (_userAccounts.containsKey(username)) {
                return _userAccounts.get(username).getUsableIDs();
            } else {
                throw new ConfigurationTaskException("Unbekannter Benutzer");
            }
        } else {
            throw new ConfigurationTaskException("Benutzer verfügt nicht über die benötigten Rechte");
        }
    }

    /**
     * Prüft ob ein Benutzer Adminrechte hat. Jeder Benutzer darf diese Aktion ausführen.
     *
     * @param orderer     Der Auftraggeber der Aktion. Wird in dieser Funktion derzeit nicht berücksichtigt, da jeder diese Abfrage ausführen darf
     * @param userToCheck Der Benutzer, dessen Rechte geprüft werden sollen.
     *
     * @return True falls der Benutzer ein Admin ist
     *
     * @throws ConfigurationTaskException Der Auftrag kann nicht ausgeführt werden, weil der Benutzer nicht existiert
     */
    @Override
    public boolean isUserAdmin(final String orderer, final String userToCheck) throws ConfigurationTaskException {
        if (_userAccounts.containsKey(userToCheck)) {
            return _userAccounts.get(userToCheck).isAdmin();
        }
        throw new ConfigurationTaskException("Unbekannter Benutzer");
    }

    /**
     * Hilfsmethode die eine {@code Collection<DataAndATGUsageInformation>} aus einem Deserializer deserialisiert.
     *
     * @param deserializer Quelle der Daten
     *
     * @return Eine {@code Collection<DataAndATGUsageInformation>} mit den Daten aus dem Deserializer
     *
     * @throws IOException
     * @see ConfigurationArea#createDynamicObject(DynamicObjectType, String, String, Collection)
     */
    private Collection<DataAndATGUsageInformation> readDataAndATGUsageInformation(final Deserializer deserializer) throws IOException {
        final int numberOfPackets = deserializer.readInt();
        final ArrayList<DataAndATGUsageInformation> result = new ArrayList<>(numberOfPackets);

        for (int i = 0; i < numberOfPackets; i++) {
            final AttributeGroupUsage attributeGroupUsage = (AttributeGroupUsage) deserializer.readObjectReference(_dataModel);
            Data data = deserializer.readData(attributeGroupUsage.getAttributeGroup());
            result.add(new DataAndATGUsageInformation(attributeGroupUsage, data));
        }

        return result;
    }

    /**
     * @param username                      Benutzer, der den Auftrag angestossen hat
     * @param usernameSingleServingPasswort Benutzer für den das Einmal-Passwort gedacht ist
     * @param passwortSingleServingPasswort Einmal-Passwort
     *
     * @throws RequestException           Technischer Fehler, der Auftrag konnte nicht bearbeitet werden.
     * @throws ConfigurationTaskException Die Konfiguration weigert sich den Auftrag auszuführen weil z.b. das Passwort falsch war, der Benutzer nicht
     *                                    die nötigen Rechte besitzt usw..
     */
    @Override
    public void createSingleServingPassword(String username, String usernameSingleServingPasswort, String passwortSingleServingPasswort)
        throws RequestException, ConfigurationTaskException {
        // prüfen, ob der Benutzer überhaupt ein Einmal-Passwort erzeugen darf
        if (isAdmin(username)) {
            // Der Benutzer darf ein Einmal-Passwort anlegen, also Nachricht entschlüsseln

            // Einmal-Passwort erzeugen
            if (_userAccounts.containsKey(usernameSingleServingPasswort)) {
                _userAccounts.get(usernameSingleServingPasswort).createNewSingleServingPassword(passwortSingleServingPasswort);
            } else {
                // Der Benutzer, für den das Passwort angelegt werden soll, existiert nicht
                throw new ConfigurationTaskException("Unbekannter Benutzer");
            }
        } else {
            throw new ConfigurationTaskException("Benutzer verfügt nicht über die benötigten Rechte");
        }
    }

    /**
     * Prüft, ob das Verfahren, das zum ver/entschlüsseln benutzt wurde, zugelassen ist. Ist das Verfahren nicht zugelassen oder es kann nicht
     * zugeordnet werden, wird eine ConfigurationTaskException geworfen.
     *
     * @param usedEncryptDecryptProcedure Benutztes Verfahren als String
     *
     * @return Verfahren
     */
    private EncryptDecryptProcedure isEncryptDecryptProcedureAllowed(String usedEncryptDecryptProcedure) throws ConfigurationTaskException {
        // Es wird eine IllegalArgumException geworfen, wenn das Verfahren unbekannt ist
        final EncryptDecryptProcedure usedProcedure = EncryptDecryptProcedure.valueOf(usedEncryptDecryptProcedure);

        // Sollen weitere Verschlüsslungsverfahren benutzt werden, muss die Factory erweitert werden und die If-Abfrage um
        // die entsprechenden zugelassenen Verfahren erweitert werden
        if ((usedProcedure != EncryptDecryptProcedure.HmacMD5) && (usedProcedure != EncryptDecryptProcedure.PBEWithMD5AndDES)) {
            // Das gewählte Verschlüssungsverfahren wird nicht unterstützt
            throw new ConfigurationTaskException("Das Verfahren wird nicht unterstützt: " + usedEncryptDecryptProcedure);
        }

        return usedProcedure;
    }

    /**
     * Prüft ob der Benutzer Admin-Rechte besitzt.
     *
     * @param username Benutzername, der geprüft werden soll ob Admin-Rechte vorhanden sind
     *
     * @return true = Der Benutzer darf die Eigenschaften anderer Benutzer ändern und Einmal-Passwörter anlegen; false = Der Benutzer darf nur sein
     *     eigenes Passwort ändern
     */
    private boolean isAdmin(String username) {
        if (_userAccounts.containsKey(username)) {
            return _userAccounts.get(username).isAdmin();
        } else {
            return false;
        }
    }

    /**
     * Legt einen neuen Benutzer mit den übergebenen Parametern an.
     *
     * @param username          Benutzer, der den Auftrag erteilt
     * @param newUserName       Name des neuen Benutzers
     * @param newUserPassword   Passwort des neuen Benutzers
     * @param admin             Rechte des neuen Benutzers (true = Adminrechte; false = normaler Benutzerrechte)
     * @param newUserPid        Pid, die der neue Benutzer erhalten soll. Wird ein Leerstring ("") übergeben, so bekommt der Benutzer keine explizite
     *                          Pid
     * @param configurationArea Pid des Konfigurationsbereichs, in dem der neue Benutzer angelegt werden soll
     * @param data              Konfigurierende Datensätze, die angelegt werden sollen (falls leere Liste oder {@code null} werden keine Daten
     *                          angelegt)
     *
     * @throws ConfigurationTaskException Der neue Benutzer durfte nicht anglegt werden (Keine Rechte, Benutzer bereits vorhanden)
     * @throws RequestException           technischer Fehler beim Zugriff auf die XML-Datei
     * @see ConfigurationArea#createDynamicObject(DynamicObjectType, String, String, Collection)
     */
    @Override
    public void createNewUser(String username, String newUserName, String newUserPid, String newUserPassword, boolean admin, String configurationArea,
                              Collection<DataAndATGUsageInformation> data) throws ConfigurationTaskException, RequestException {
        if (isAdmin(username)) {

            // Es werden 4 Fälle betrachtet
            // Fall 1: Es gibt weder ein Objekt, das den Benutzer in der Konfiguration darstellt, noch einen Eintrag in der XML-Datei (Objekt 
            // erzeugen und XML-Eintrag erzeugen (Normalfall))
            // Fall 2: Es gibt einen Eintrag in der XML-Datei aber kein Objekt das den Benutzer in der Konfiguration darstellt (Objekt erzeugen und
            // gegebenfalls XML-Datei anpassen)
            // Fall 3: Es gibt ein Objekt, aber keinen Eintrag in der XML-Datei (Eintrag in die XML-Datei, Objekt nicht ändern)
            // Fall 4: Es gibt ein Objekt und einen Eintrag in der XML-Datei (Fehlerfall)

            // Speichert, ob es zu einem Benutzer ein gültiges Objekt gibt
            final boolean userHasObject = userHasObject(newUserName, newUserPid);

            if (!userHasObject && !_userAccounts.containsKey(newUserName)) {
                try {
                    // Fall 1: Eintrag XML und Objekt erzeugen

                    // Es wird erst das Objekt angelegt, da es passieren kann, dass der Benutzer keine Rechte dafür besitzt.
                    // Dann würde eine Exception geworfen und es muss auch kein Eintrag in die XML-Datei gemacht werden.
                    // Darf der Benutzer Objekt anlegen und es kommt beim schreiben der XML-datei zu einem Fehler, so kann
                    // die Methode erneut aufgerufen werden und es wird automatisch Fall 3 abgearbeitet.
                    createUserObject(configurationArea, newUserName, newUserPid, data);
                    createUserXML(newUserName, newUserPassword, admin);
                } catch (Exception e) {
                    _debug.error("Neuen Benutzer anlegen, XML und Objekt", e);
                    throw new RequestException(e);
                }
            } else if (!userHasObject) {
                // Fall 2, das Objekt fehlt
                createUserObject(configurationArea, newUserName, newUserPid, data);
            } else if (!_userAccounts.containsKey(newUserName)) {
                try {
                    // Fall 3, der Eintrag in der XML-Datei fehlt
                    createUserXML(newUserName, newUserPassword, admin);
                } catch (Exception e) {
                    _debug.error("Neuen Benutzer anlegen, XML", e);
                    throw new RequestException(e);
                }
            } else {
                // Fall 4, es ist alles vorhanden. Ein bestehender Benutzer soll überschrieben werden. Das ist ein
                // Fehler.
                throw new ConfigurationTaskException("Der Benutzername ist bereits vergeben");
            }
        } else {
            throw new ConfigurationTaskException("Der Benutzer hat nicht die nötigen Rechte");
        }
    }

    /**
     * Erzeugt ein Objekt vom Typ "typ.Benutzer".
     *
     * @param pidConfigurationArea Pid des Konfiguratinsbereichs, in dem der neue Benutzer angelegt werden soll
     * @param username             Name des Objekts
     * @param pid                  Pid des Objekts
     * @param data                 Konfigurierende Datensätze, die angelegt werden sollen, oder {@code null} falls keine angelgt werden sollen
     *
     * @throws ConfigurationChangeException Fehler beim Erzeugen des neuen Benutzers
     */
    private void createUserObject(String pidConfigurationArea, String username, String pid, Collection<DataAndATGUsageInformation> data)
        throws ConfigurationChangeException {
        final ConfigurationArea configurationArea = _dataModel.getConfigurationArea(pidConfigurationArea);
        if (configurationArea == null) {
            final String message =
                "Das Erzeugen eines neuen Benutzerobjekts ist fehlgeschlagen, weil der angegebene Konfigurationsbereich mit der PID " +
                pidConfigurationArea + " nicht gefunden wurde.";
            _debug.error(message);
            throw new ConfigurationChangeException(message);
        }
        final SystemObjectType systemObjectType = _dataModel.getType("typ.benutzer");
	    if (systemObjectType instanceof DynamicObjectType type) {
            configurationArea.createDynamicObject(type, pid, username, data);
        } else {
            final String message =
                "Das Erzeugen eines neuen Benutzerobjekts ist fehlgeschlagen, weil der typ.benutzer nicht gefunden wurde oder kein dynamischer Typ " +
                "ist";
            _debug.error(message);
            throw new ConfigurationChangeException(message);
        }
    }

    /**
     * Erzeugt einen neuen Benutzer im speicher und speichert diesen in einer XML-Datei.
     *
     * @param newUserName     Benutzername
     * @param newUserPassword Passwort
     * @param admin           Adminrechte ja/nein
     *
     * @throws FileNotFoundException
     * @throws TransformerException
     */
    private void createUserXML(String newUserName, String newUserPassword, boolean admin) throws FileNotFoundException, TransformerException {
        // Für XML-Datei
        final String newUserRightsString;
        if (admin) {
            newUserRightsString = "ja";
        } else {
            newUserRightsString = "nein";
        }
        final Element xmlObject = createXMLUserAccount(newUserName, newUserPassword, newUserRightsString);

        final UserAccount newUser = new UserAccount(newUserName, newUserPassword, admin, new ArrayList<>(), xmlObject);

        synchronized (_xmlDocument) {
            // Das neue Objekt in die Liste der bestehenden einfügen
            _xmlDocument.getDocumentElement().appendChild(xmlObject);
        }

        // Speichern
        saveXMLFile();
        _userAccounts.put(newUser.getUsername(), newUser);
    }

    /**
     * Prüft, ob es zu der Kombination Benutzername und Pid ein gültiges Objekt gibt. Ein gültiges Objekt bedeutet, dass Pid und Benutzername
     * übereinstimmen.
     *
     * @param username Benutzername
     * @param pid      Pid des Benutzers
     *
     * @return true = Es gibt ein aktuell gültiges Objekt; false = Es gibt kein aktuell gültiges Objekt
     *
     * @throws IllegalStateException Es gibt ein Objekt mit der angegebenen Pid, aber der Name des Objekts ist anders, als der übergebene Name
     */
    private boolean userHasObject(final String username, final String pid) {

        if (pid != null && !pid.isEmpty()) {
            // Es wurde eine Pid übergeben, gibt es zu der Pid ein Objekt
            final SystemObject user = _dataModel.getObject(pid);
            if (user != null) {
                // Es gibt ein Objekt zur angegebenen Pid
                if (username.equals(user.getName())) {
                    // Pid und Benutzername stimmen mit dem gefundenen Objekt überein. Das Objekt wurde gefunden
                    return true;
                } else {
                    // Es gibt zwar ein Objekt mit der Pid, aber der Name des Objekts ist anders
                    throw new IllegalStateException("Es darf zu einer Pid nur einen Benutzernamen geben");
                }
            } else {
                // Zur angegebenen Pid konnte kein Objekt gefunden werden
                return false;
            }
        } else {
            return getUserObject(username) != null;
        }
    }

    /**
     * Ließt aus einem Byte-Array die ersten 4 Bytes aus und erzeugt daraus die benutztes Serializerversion
     *
     * @param message Nachricht, die ersten 4 Bytes werden ausgelesen
     *
     * @return Integer, der aus den ersten 4 Bytes der Nachricht ausgelesen wird
     */
    private final int getSerializerVersion(final byte[] message) {
        return ((message[0] << 24) & 0xff000000) | ((message[1] << 16) & 0x00ff0000) | ((message[2] << 8) & 0x0000ff00) | (message[3] & 0x000000ff);
    }

    /**
     * Entfernt die ersten 4 Bytes eines Byte-Arrays und gibt ein neues Array zurück, bei dem genau die ersten 4 Bytes fehlen.
     *
     * @param byteArray Array, aus dem die ersten 4 Bytes entfernt werden
     *
     * @return Array, bei dem die ersten 4 Bytes des Ursprungs-Arrays fehlen
     */
    private final byte[] removeFirst4Bytes(byte[] byteArray) {
        final byte[] shortArray = new byte[byteArray.length - 4];
        System.arraycopy(byteArray, 4, shortArray, 0, shortArray.length);
        return shortArray;
    }

    /**
     * Setzt bei einem Benutzer das Passwort neu. Dies kann entweder ein Admin bei einem anderen Benutzerkonto oder ein Benutzer bei seinem eigenen
     * Benutzerkonto.
     * <p>
     * Ist für einen Benutzer nur das Objekt des Benutzers in der Konfiguration vorhanden, aber das Benutzerkonto fehlt, wird das Benutzerkonto mit
     * {@link #createNewUser} angelegt. Das neue Benutzerkonto besitzt dabei keine Adminrechte. Das neue Benutzerkonto wird dabei das Passwort
     * erhalten, das neu gesetzt werden sollte.
     * <p>
     * Gibt es zwar ein Benutzerkonto, aber kein Objekt in der Konfiguration, wird ein Fehler ausgegeben.
     * <p>
     * Sind weder Objekt noch Benutzerkonto vorhanden wird ein Fehler ausgegeben.
     *
     * @param username                  Benutzer, der den Auftrag zum ändern des Passworts erteilt hat
     * @param userNameForPasswordChange Benutzer, dessen Passwort geändert werden soll
     * @param newPassword               neues Passwort
     *
     * @throws ConfigurationTaskException Der Benutzer ist unbekannt oder es gibt zu dem Benutzer kein entsprechendes Objekt oder der Benutzer darf
     *                                    das Passwort nicht ändern (kein Admin oder der Besitzer des Passwords).
     * @throws RequestException           Fehler beim Zugriff auf die XML-Datei
     */
    @Override
    public void changeUserPassword(String username, String userNameForPasswordChange, String newPassword)
        throws ConfigurationTaskException, RequestException {

        // Die Pid des Benutzers ist unbekannt, darum ""
        final boolean hasUserObject = userHasObject(userNameForPasswordChange, "");

        synchronized (_userAccounts) {
            if (hasUserObject && _userAccounts.containsKey(userNameForPasswordChange)) {
                // Das Objekt ist vorhanden und es gibt Benutzerdaten zu dem Benutzer, der geändert werden soll (Das ist der Normalfall)

                // Der Benutzername steht zur Verfügung, nun kann geprüft werden wer versucht das Passwort zu ändern.
                // Ist es ein Admin
                // oder
                // versucht der Besitzer des Accounts das Passwort zu ändern

                if (isAdmin(username) || username.equals(userNameForPasswordChange)) {
                    try {
                        _userAccounts.get(userNameForPasswordChange).setXmlVerifierText(newPassword);
                    } catch (Exception e) {
                        _debug.error("Passwort ändern", e);
                        throw new RequestException(e);
                    }
                } else {
                    // Der Benutzer hat nicht das Recht das Passwort zu ändern
                    throw new ConfigurationTaskException("Passwortänderung verworfen");
                }
            } else if (hasUserObject && !_userAccounts.containsKey(userNameForPasswordChange)) {
                // Es gibt ein Objekt, aber kein Benutzerkonto. Falls der Benutzer ein Admin ist
                // wird ein neues Benutzerkonto erzeugt
                if (isAdmin(username)) {
                    try {
                        // Der Benutzer darf neue Konten erzeugen. Also wird ein neues Benutzerkonto ohne Adminrechte angelegt
                        createUserXML(userNameForPasswordChange, newPassword, false);
                    } catch (Exception e) {
                        _debug.error("Passwort ändern", e);
                        throw new RequestException(e);
                    }
                } else {
                    // Der Benutzer hat nicht das Recht neue Benutzerkonten zu erzeugen
                    throw new ConfigurationTaskException("Passwortänderung verworfen, da benötigte Rechte fehlen");
                }
            } else {
                // Es ist ein Fehler aufgetreten, der Fehler wird nun genauer spezifiziert
                if (!hasUserObject) {
                    // Es gibt kein Objekt
                    throw new ConfigurationTaskException("Kein Benutzerobjekt vorhanden");
                } else if (!_userAccounts.containsKey(userNameForPasswordChange)) {
                    // Das Benutzerkonto fehlt
                    throw new ConfigurationTaskException("Unbekannter Benutzer");
                }
            }
        } // synchronized (_userAccounts)
    }

    @Override
    public void changeUserName(String username, String oldUserName, String newUserName, String newPassword)
        throws ConfigurationTaskException, RequestException {
        // Die Pid des Benutzers ist unbekannt, darum ""
        final SystemObject userObject = getUserObject(oldUserName);
        
        if(userObject == null) {
            throw new ConfigurationTaskException("Kein Benutzerobjekt vorhanden");
        }



        synchronized (_userAccounts) {
            if(_userAccounts.containsKey(newUserName)) {
                throw new ConfigurationTaskException("Neuer Benutzername ist bereits vergeben");
            }
            if (!_userAccounts.containsKey(oldUserName)) {
                // Das Benutzerkonto fehlt
                throw new ConfigurationTaskException("Unbekannter Benutzer");
            }
            
            // Das Objekt ist vorhanden und es gibt Benutzerdaten zu dem Benutzer, der geändert werden soll (Das ist der Normalfall)

            if (!isAdmin(username) && !username.equals(oldUserName)) {
                // Der Benutzer hat nicht das Recht das Passwort zu ändern
                throw new ConfigurationTaskException("Keine Berechtigung");
            }

            try {
                UserAccount oldUserAccount = _userAccounts.get(oldUserName);
                // Merken, ob der Benutzer ein Admin ist
                boolean isAdmin = oldUserAccount.isAdmin();
                
                // Neuen XML-Benutzer erstellen
                createUserXML(newUserName, newPassword, isAdmin);

                // Benutzerobjekt umbenennen
                try {
                    userObject.setName(newUserName);
                }
                catch (ConfigurationChangeException e) {
                    deleteUserXML(newUserName);
                    throw e;
                }
                
                // Alten XML-Benutzer löschen
                deleteUserXML(oldUserName);
            } catch (Exception e) {
                _debug.error("Passwort ändern", e);
                throw new RequestException(e);
            }
        } // synchronized (_userAccounts)
    }

    /**
     * @param username             Benutzer, der den Auftrag erteilt hat (dieser muss Adminrechte besitzen)
     * @param usernameChangeRights Benutzer, dessen Rechte geändert werden soll
     * @param newUserRights        Neue Rechte des Benutzers (true = Admin-Rechte, false = normaler Benutzerrechte
     *
     * @throws ConfigurationTaskException Der Benutzer ist unbekannt oder der Auftraggeber besitzt nicht die nötigen Rechte
     * @throws RequestException           Fehler beim Zugriff auf die XML-Datei
     */
    @Override
    public void changeUserRights(String username, String usernameChangeRights, boolean newUserRights)
        throws ConfigurationTaskException, RequestException {
        if (isAdmin(username)) {
            // Admin versucht die Rechte zu ändern
            if (_userAccounts.containsKey(usernameChangeRights)) {
                // Der Benutzer existiert
                try {
                    _userAccounts.get(usernameChangeRights).setAdminRights(newUserRights);
                } catch (Exception e) {
                    _debug.error("Benutzerrechte ändern", e);
                    throw new RequestException(e);
                }
            } else {
                throw new ConfigurationTaskException("Unbekannter Benutzer");
            }
        } else {
            // Der Benutzer besitzt nicht die nötigen Rechte
            throw new ConfigurationTaskException("Der Benutzer besitzt nicht die nötgen Rechte");
        }
    }

    /**
     * Erzeugt einen Desirialisierer auf den mit den üblichen Methoden zugegriffen werden kann. Dafür wird der übergebene, verschlüsselte Text
     * entschlüsselt.
     *
     * @param encryptedMessage            Verschlüsselte Nachricht, diese wird entschlüsselt
     * @param decryptenText               Text, mit dem die verschlüsselte Nachricht entschlüsselt wird
     * @param authentificationProcessName Verfahren, mit dem die Nachricht verschlüsselt wurde
     *
     * @return Deserialisierer
     *
     * @throws Exception Fehler beim entschlüsseln oder beim erstellen des Desirialisierers
     */
    private final Deserializer getDeserializer(byte[] encryptedMessage, String decryptenText, String authentificationProcessName) throws Exception {
        final byte[] decryptedMessage =
            DecryptFactory.getDecryptInstance(isEncryptDecryptProcedureAllowed(authentificationProcessName)).decrypt(encryptedMessage, decryptenText);

        final int serializerVersion = getSerializerVersion(decryptedMessage);

        InputStream in = new ByteArrayInputStream(removeFirst4Bytes(decryptedMessage));

        //deserialisieren
        return SerializingFactory.createDeserializer(serializerVersion, in);
    }

    /**
     * Löscht einen angegebenen Benutzer. Diese Aktion kann nur von Administratoren ausgeführt werden.
     *
     * @param username     Veranlasser der Aktion
     * @param userToDelete Benutzername des Benutzers, der gelöscht werden soll
     *
     * @throws RequestException           Das Löschen kann aufgrund eines Problems nicht durchgeführt werden
     * @throws ConfigurationTaskException Die Anfrage ist fehlerhaft weil der Veranlasser nicht die nötigen Rechte hat oder der zu löschende Benutzer
     *                                    nicht existiert
     */
    @Override
    public void deleteUser(String username, String userToDelete) throws RequestException, ConfigurationTaskException {
        if (isAdmin(username)) {
            final boolean userHasObject = userHasObject(userToDelete, "");

            if (userHasObject && _userAccounts.containsKey(userToDelete)) {
                try {
                    deleteUserObject(userToDelete);
                    deleteUserXML(userToDelete);
                } catch (Exception e) {
                    _debug.error("Benutzer löschen, XML und Objekt", e);
                    throw new RequestException(e);
                }
            } else if (userHasObject) {
                // Fall 2, das Objekt fehlt
                deleteUserObject(userToDelete);
            } else if (_userAccounts.containsKey(userToDelete)) {
                try {
                    // Fall 3, der Eintrag in der XML-Datei fehlt
                    deleteUserXML(userToDelete);
                } catch (Exception e) {
                    _debug.error("Benutzer löschen, XML", e);
                    throw new RequestException(e);
                }
            } else {
                throw new ConfigurationTaskException("Der Benutzer existiert nicht");
            }
        } else {
            throw new ConfigurationTaskException("Der Benutzer hat nicht die nötigen Rechte");
        }
    }

    @Override
    public SrpVerifierAndUser getSrpVerifierData(final String authenticatedUser, String userName, final int passwordIndex)
        throws ConfigurationTaskException {

        if (!isAdmin(authenticatedUser) && !authenticatedUser.equals(userName)) {
            throw new ConfigurationTaskException("Der Benutzer hat nicht die nötigen Rechte");
        }

        return getVerifier(userName, getUserLogin(userName), passwordIndex);
    }

    @Override
    public int setOneTimePasswords(final String authenticatedUser, final String usernamePassword, final List<String> passwords, final boolean append)
        throws ConfigurationTaskException, RequestException {
        // prüfen, ob der Benutzer überhaupt ein Einmal-Passwort erzeugen darf
        if (isAdmin(authenticatedUser)) {
            // Einmal-Passwort erzeugen
            if (_userAccounts.containsKey(usernamePassword)) {
                return _userAccounts.get(usernamePassword).createNewSingleServingPasswords(passwords, append);
            } else {
                // Der Benutzer, für den das Passwort angelegt werden soll, existiert nicht
                throw new ConfigurationTaskException("Unbekannter Benutzer");
            }
        } else {
            throw new ConfigurationTaskException("Benutzer verfügt nicht über die benötigten Rechte");
        }
    }

    @Override
    public void disableSingleServingPassword(final String authenticatedUser, final String usernamePassword, final int passwordIndex)
        throws ConfigurationTaskException, RequestException {
        if (isAdmin(authenticatedUser) || authenticatedUser.equals(usernamePassword)) {
            // Der Benutzer darf ein Einmal-Passwort deaktivieren

            // Einmal-Passwort erzeugen
            if (_userAccounts.containsKey(usernamePassword)) {
                _userAccounts.get(usernamePassword).disableSingleServingPassword(passwordIndex);
            } else {
                // Der Benutzer, für den das Passwort angelegt werden soll, existiert nicht
                throw new ConfigurationTaskException("Unbekannter Benutzer");
            }
        } else {
            throw new ConfigurationTaskException("Benutzer verfügt nicht über die benötigten Rechte");
        }
    }

    public UserLogin getUserLogin(final String userName) {
        if (_userAccounts.containsKey(userName)) {
            SystemObject userObject = getUserObject(userName);
            if (userObject != null) {
                return UserLogin.user(userObject.getId());
            }
        }
        return UserLogin.notAuthenticated();
    }

    @Override
    public SystemObject getUserObject(final String userName) {
        List<SystemObject> matchingUsers = new ArrayList<>();
        for (final SystemObject userObject : _dataModel.getType("typ.benutzer").getObjects()) {
            // Objekt, das in der Konfiguration gespeichert ist und einen Benutzer darstellt
            if (userObject.getName().equals(userName)) {
                matchingUsers.add(userObject);
            }
        }
        switch (matchingUsers.size()) {
            case 0:
                return null;
            case 1:
                return matchingUsers.get(0);
        }

        List<SystemObject> localUsers = matchingUsers.stream().filter(this::isLocalUser).collect(Collectors.toList());
        matchingUsers.sort(Comparator.comparing(SystemObject::getPidOrId));
        localUsers.sort(Comparator.comparing(SystemObject::getPidOrId));
        StringBuilder stringBuilder =
            new StringBuilder().append("Zum Benutzernamen \"").append(userName).append("\" gibt es ").append(matchingUsers.size())
                .append(" SystemObjekte, davon sind ").append(localUsers.size()).append(" der lokalen AOE zugeordnet: ");
        for (SystemObject matchingUser : matchingUsers) {
            stringBuilder.append("\n").append(matchingUser.getPidOrId()).append(" (")
                .append(matchingUser.getConfigurationArea().getConfigurationAuthority().getPidOrId()).append(")");
        }
        if (localUsers.size() == 1) {
            stringBuilder.append("\nDer Benutzer des lokalen AOE ").append(_dataModel.getConfigurationAuthority().getPidOrId())
                .append(" wird verwendet");
            _debug.warning(stringBuilder.toString());
            return localUsers.get(0);
        } else {
            if (!localUsers.isEmpty()) {
                return chooseRandomUser(localUsers, stringBuilder);
            } else {
                return chooseRandomUser(matchingUsers, stringBuilder);
            }
        }
    }

    /**
     * Prüft, ob ein Benutzerobjekt unter der lokalen AOE konfiguriert wurde
     *
     * @param userObject Benutzerobjekt
     *
     * @return true: Unte der lokalen AOE, sonst false
     */
    private boolean isLocalUser(final SystemObject userObject) {
        return Objects.equals(userObject.getConfigurationArea().getConfigurationAuthority(), _dataModel.getConfigurationAuthority());
    }

    private SrpVerifierAndUser getVerifier(final String userName, final UserLogin userLogin, final int passwordIndex) {
        UserAccount userAccount = _userAccounts.get(userName);
        if (userAccount == null) {
            return new SrpVerifierAndUser(userLogin,
                                          fakeVerifier(userName, secretHash(userName, passwordIndex), ClientCredentials.ofString(_secretToken)),
                                          false);
        }
        try {
            return new SrpVerifierAndUser(userLogin, userAccount.getSrpVerifier(passwordIndex), false);
        } catch (IllegalArgumentException ignored) {
            // Kein SRP-Format, Passwort liegt im Klartext vor.
            // Passenden SRP-Verifier erzeugen, damit der Benutzer sich authentifizieren kann.
            // Dem Datenverteiler ist es egal, ob dieser Verifier in der benutzerverwaltung.xml gespeichert war, oder hier erzeugt wurde.
            // Tatsächlich kann er es gar nicht unterscheiden.

            ClientCredentials clientCredentials = userAccount.getClientCredentials(passwordIndex);
            if (clientCredentials != null) {
                return new SrpVerifierAndUser(userLogin, fakeVerifier(userName, secretHash(userName, passwordIndex), clientCredentials), true);
            } else {
                // Passwort ist leer ("")
                // Fake-Verifier erzeugen
                return new SrpVerifierAndUser(userLogin,
                                              fakeVerifier(userName, secretHash(userName, passwordIndex), ClientCredentials.ofString(_secretToken)),
                                              false);
            }
        }
    }

    private byte[] secretHash(final String userName, final int passwordIndex) {
        return SrpUtilities
            .generatePredictableSalt(getCryptoParameters(), (userName + _secretToken + passwordIndex).getBytes(StandardCharsets.UTF_8));
    }

    private SrpCryptoParameter getCryptoParameters() {
        return SrpCryptoParameter.getDefaultInstance();
    }

    /**
     * Löscht einen Benutzer aus der XML-Datei
     *
     * @param userToDelete Benutzer, der gelöscht werden soll
     *
     * @throws TransformerException  Fehler beim XML-Zugriff
     * @throws FileNotFoundException XMl-Datei nciht gefunden
     */
    private void deleteUserXML(final String userToDelete) throws TransformerException, FileNotFoundException {
        try {
            synchronized (_xmlDocument) {
                final NodeList childNodes = _xmlDocument.getDocumentElement().getChildNodes();
                for (int i = 0; i < childNodes.getLength(); i++) {
                    final Node node = childNodes.item(i);
                    if (node.hasAttributes()) {
                        final NamedNodeMap attributes = node.getAttributes();
                        final Node name = attributes.getNamedItem("name");
                        if (name != null && name.getNodeValue().equals(userToDelete)) {
                            _xmlDocument.getDocumentElement().removeChild(node);
                            saveXMLFile();
                            return;
                        }
                    }
                }
            }
            _debug.error("deleteUserXML: Konnte Benutzer nicht aus XML-Datei löschen. Knoten wurde nicht gefunden.", userToDelete);
        } finally {
            _userAccounts.keySet().remove(userToDelete);
        }
    }

    /**
     * Entfernt das dynamische Benutzerobjekt aus dem Datenmodell
     *
     * @param userToDelete Benutzer, der gelöscht werden soll
     *
     * @throws ConfigurationChangeException Fehler beim durchführen der Aktion
     */
    private void deleteUserObject(final String userToDelete) throws ConfigurationChangeException {
        for (final SystemObject configUser : _dataModel.getType("typ.benutzer").getObjects()) {
            if (configUser.getName().equals(userToDelete)) {
                configUser.invalidate();
            }
        }
    }

    /**
     * Speichert alle Benutzerdaten in einer XML-Datei.
     *
     * @throws TransformerException
     * @throws FileNotFoundException
     */
    private void saveXMLFile() throws TransformerException, FileNotFoundException {

        TransformerFactory transformerFactory = TransformerFactory.newInstance();
        Transformer transformer = transformerFactory.newTransformer();
        transformer.setOutputProperty(OutputKeys.ENCODING, "ISO-8859-1"); // ISO-Kodierung für westeuropäische Sprachen
        transformer.setOutputProperty(OutputKeys.INDENT, "yes");
        transformer.setOutputProperty(OutputKeys.STANDALONE, "no");       // DTD ist in einer separaten Datei
        transformer.setOutputProperty("{http://xml.apache.org/xslt}indent-amount", "4");

        synchronized (_xmlDocument) {

            // DOCTYPE bestimmen
            final DocumentType documentType = _xmlDocument.getDoctype();
            String publicID = null;
            String systemID = null;
            if (documentType != null) {
                publicID = documentType.getPublicId();
                systemID = documentType.getSystemId();
            }
            if (publicID != null) {
                transformer.setOutputProperty(OutputKeys.DOCTYPE_PUBLIC, publicID);
            } else {
                // DOCTYPE PUBLIC_ID ist nicht vorhanden -> erstellen
                transformer.setOutputProperty(OutputKeys.DOCTYPE_PUBLIC, "-//K2S//DTD Authentifizierung//DE");
            }
            if (systemID != null) {
                transformer.setOutputProperty(OutputKeys.DOCTYPE_SYSTEM, systemID);
            } else {
                // DOCTYPE SYSTEM_ID ist nicht vorhanden -> erstellen
                transformer.setOutputProperty(OutputKeys.DOCTYPE_SYSTEM, "authentication.dtd");
            }

            DOMSource source = new DOMSource(_xmlDocument);

	        try (BufferedOutputStream outputStream = new BufferedOutputStream(new FileOutputStream(_xmlFile))) {
                StreamResult result = new StreamResult(outputStream);    // gibt die XML-Struktur in einem Stream (Datei) aus
                transformer.transform(source, result);
            } catch (IOException e) {
                throw new TransformerException("IO-Fehler beim Speichern von " + _xmlFile, e);
            }
        }
    }

    /**
     * Sicher die Benutzerverwaltungsdatei in das angegebene Verzeichnis
     *
     * @param targetDirectory Zielverzeichnis
     *
     * @throws IOException IO-Fehler
     */
    public void createBackupFile(File targetDirectory) throws IOException {
        final String fileName = _xmlFile.getName();

        try {
            saveXMLFile();
        } catch (TransformerException e) {
            e.printStackTrace();
            throw new IOException("Konnte XML-Datei nicht sichern: " + e.getMessage()); // wg. java 1.5
        }

        // Datei kopieren
        try (FileOutputStream fileOutputStream = new FileOutputStream(new File(targetDirectory, fileName))) {
            try (FileInputStream inputStream = new FileInputStream(_xmlFile)) {
                byte[] buf = new byte[1024];
                int len;
                while ((len = inputStream.read(buf)) > 0) {
                    fileOutputStream.write(buf, 0, len);
                }
            }
        }
    }

    /**
     * Erzeugt ein XML Objekt, das einem Einmal-Passwort entspricht.
     *
     * @param newPassword   Passwort des neuen Einmal-Passworts
     * @param passwortindex Index des Einmal-Passworts
     * @param usable        ja = Das Einmal-Passwort ist noch zu benutzen; nein = Das Einmal-Passwort kann nicht mehr benutzt werden
     *
     * @return XML-Objekt, das einem Einmal-Passwort entspricht
     */
    private Element createXMLSingleServingPasswort(String newPassword, int passwortindex, String usable) {
        synchronized (_xmlDocument) {
            Element xmlSingleServingPassword = _xmlDocument.createElement("autorisierungspasswort");
            xmlSingleServingPassword.setAttribute("passwort", newPassword);
            xmlSingleServingPassword.setAttribute("passwortindex", String.valueOf(passwortindex));
            xmlSingleServingPassword.setAttribute("gueltig", usable);
            return xmlSingleServingPassword;
        }
    }

    /**
     * Erzeugt ein XML Objekt, das einem Benutzerkonto entspricht. Einmal-Passwörter müssen mit der entsprechenden Methode erzeugt werden.
     *
     * @param name     Name des Benutzers
     * @param password Passwort des Benutzers (in Klarschrift)
     * @param admin    ja = Der Benutzer besitzt Admin-Rechte; nein = Der Benutzer besitzt keine Admin-Rechte
     *
     * @return XML Objekt, das einem Benutzerkonto entspricht
     */
    private synchronized Element createXMLUserAccount(String name, String password, String admin) {
        synchronized (_xmlDocument) {
            Element xmlSingleServingPassword = _xmlDocument.createElement("benutzeridentifikation");
            xmlSingleServingPassword.setAttribute("name", name);
            xmlSingleServingPassword.setAttribute("passwort", password);
            xmlSingleServingPassword.setAttribute("admin", admin);
            return xmlSingleServingPassword;
        }
    }

    /**
     * Diese Klasse Speichert alle Informationen, die zu Benutzerkonto gehören. Dies beinhaltet:
     * <p>
     * Benutzername
     * <p>
     * Benutzerpasswort
     * <p>
     * Adminrechte
     * <p>
     * Liste von Einmal-Passwörtern (siehe TPuK1-130)
     * <p>
     * Sollen Änderungen an einem dieser Informationen vorgenommen werden, führt dies erst dazu, dass die Daten persistent in einer XML-Datei
     * gespeichert werden. Ist dies erfolgreich, wird die Änderung auch an den Objekten durchgeführt. Kann die Änderungen nicht gespeichert werden,
     * wird ein entsprechender Fehler ausgegeben und die Änderung nicht durchgeführt
     */
    private final class UserAccount {

        private static final int NO_RESULT = -1;
        /** Benutzername des Accounts */
        private final String _username;
        /**
         * Liste, die alle benutzbaren Einmalpasswörter enthält.
         */
        private final Collection<SingleServingPassword> _usableSingleServingPasswords = new HashSet<>();
        /**
         * Speichert alle Einmal-Passwörter . Soll ein neues Einmal-Passwort erzeugt werden, und das Passwort befindet sich bereits in dieser Menge,
         * dann darf das neue Einmal-Passwort nicht angelegt werden. Dadurch wird verhindert, dass ein Passwort oder Überprüfungscode doppelt
         * verwendet wird.
         */
        private final Set<String> _allSingleServingPasswords = new HashSet<>();
        /** XML-Objekt, dieses muss zuerst verändert und gespeichert werden, bevor die Objekte im Speicher geändert werden */
        private final Element _xmlObject;
        /** Passwort oder SRP-Überprüftungscode, so wie er in der Datei steht */
        private String _xmlVerifierText;
        /** true = Der Benutzer ist ein Admin und darf Einstellungen bei anderen Benutzern vornehmen */
        private boolean _admin;
        /**
         * Speichert den größten Index, der bisher für ein Einmal-Passwort benutzt wurde. Das nächste Einmal-Passwort hätte als Index
         * "_greatestSingleServingPasswordIndex++".
         * <p>
         * Wird mit -1 initialisiert. Das erste Passwort erhält also Index 0.
         * <p>
         * Der Wert wird im Konstruktor, falls Einmal-Passwörter vorhanden sind, auf den größten vergebenen Index gesetzt.
         */
        private int _greatestSingleServingPasswordIndex = -1;

        /**
         * @param username                  Benutzername
         * @param xmlPassword               Passwort, wie es in der XML-Datei gespeichert wurde
         * @param admin                     Ob der Benutzer Admin_Rechte hat
         * @param allSingleServingPasswords Alle Einmal-Passwörter
         * @param xmlObject                 XML-Objekt, aus dem die obigen Daten ausgelesen wurden
         */
        public UserAccount(String username, String xmlPassword, boolean admin, List<SingleServingPassword> allSingleServingPasswords,
                           Element xmlObject) {
            _username = username;
            _xmlVerifierText = xmlPassword;
            _xmlObject = xmlObject;
            _admin = admin;

            for (SingleServingPassword singleServingPassword : allSingleServingPasswords) {
                // Damit dieses Passwort nicht noch einmal vergeben werden kann
                _allSingleServingPasswords.add(singleServingPassword.getXmlVerifierText());

                if (singleServingPassword.getIndex() > _greatestSingleServingPasswordIndex) {
                    _greatestSingleServingPasswordIndex = singleServingPassword.getIndex();
                }

                if (singleServingPassword.isPasswordUsable()) {
                    // Das Passwort kann noch benutzt werden
                    _usableSingleServingPasswords.add(singleServingPassword);
                }
            }
        }

        /**
         * Benutzername
         *
         * @return s.o.
         */
        public String getUsername() {
            return _username;
        }

        /**
         * Unverschlüsseltes Passwort des Benutzers
         *
         * @return s.o.
         *
         * @deprecated In der Übergangsphase kann in der XML-Datei noch ein Klartextpasswort drin stehen
         */
        @Deprecated
        public ClientCredentials getClientCredentials() {
            return getClientCredentials(-1);
        }

        /**
         * Unverschlüsseltes Passwort des Benutzers
         *
         * @param passwordIndex
         *
         * @return s.o.
         *
         * @deprecated In der Übergangsphase kann in der XML-Datei noch ein Klartextpasswort drin stehen
         */
        @Deprecated
        public ClientCredentials getClientCredentials(final int passwordIndex) {
            try {
                getSrpVerifier(passwordIndex);
            } catch (IllegalArgumentException ignored) {
                return ClientCredentials.ofPassword(getXmlVerifierText(passwordIndex).toCharArray());
            }
            throw new IllegalArgumentException("Das Passwort am Benutzer " + _username +
                                               " ist verschlüsselt gespeichert, eine Authentifizierung über das veraltete HMAC-Verfahren ist damit " +
                                               "nicht mehr möglich. Datenverteiler und Applikationsfunktionen müssen ggf. aktualisiert werden.");
        }

        /**
         * Gibt den SRP-Überprüfungscode des Standardpassworts zurück
         *
         * @return den SRP-Verifier
         *
         * @throws IllegalArgumentException Bei einem String, der nicht dem erwarteten Format entspricht (also wenn es sich bspw. um ein
         *                                  Klartextpasswort handelt(
         */
        public SrpVerifierData getSrpVerifier() {
            return getSrpVerifier(-1);
        }

        /**
         * Gibt den SRP-Überprüfungscode zurück
         *
         * @param passwordIndex Passwort-Index (-1 für Standardpasswort)
         *
         * @return den SRP-Verifier
         *
         * @throws IllegalArgumentException Bei einem String, der nicht dem erwarteten Format entspricht (also wenn es sich bspw. um ein
         *                                  Klartextpasswort handelt(
         */
        public SrpVerifierData getSrpVerifier(final int passwordIndex) {
            return new SrpVerifierData(getXmlVerifierText(passwordIndex));
        }

        /**
         * Ändert das Passwort und speichert das neue Passwort in einer XML-Datei
         *
         * @param xmlVerifierText Neues Passwort
         */
        public void setXmlVerifierText(String xmlVerifierText) throws FileNotFoundException, TransformerException {
            _xmlObject.setAttribute("passwort", xmlVerifierText);
            saveXMLFile();

            // Erst nach dem das neue Passwort gespeichert wurde, wird die Änderung im Speicher übernommen
            _xmlVerifierText = xmlVerifierText;
        }

        /** @return true = Der Benutzer darf die Eigenschaften anderer Benutzer ändern; false = Der Benutzer darf nur sein Passwort ändern */
        public boolean isAdmin() {
            return _admin;
        }

        /**
         * Legt fest, ob ein Benutzer Admin-Rechte besitzt. Die Änderung wird sofort in der XML-Datei gespeichert.
         *
         * @param adminRights true = Der Benutzer besitzt Admin Rechte; false = Der Benutzer besitzt keine Admin-Rechte
         */
        public void setAdminRights(boolean adminRights) throws FileNotFoundException, TransformerException {
            if (adminRights) {
                _xmlObject.setAttribute("admin", "ja");
            } else {
                _xmlObject.setAttribute("admin", "nein");
            }

            saveXMLFile();

            _admin = adminRights;
        }

        /**
         * Erzeugt ein neues Einmal-Passwort. Der Index wird automatisch angepasst.
         *
         * @param newPassword Passwort des Einmal-Passworts
         *
         * @throws RequestException           Fehler beim Speichern des neuen Passworts, das Passwort wurde nicht angelegt.
         * @throws ConfigurationTaskException Das Passwort wurde bereits vergeben, es wurde kein neues Passwort angelegt.
         */
        public synchronized void createNewSingleServingPassword(final String newPassword) throws ConfigurationTaskException, RequestException {
            if (!_allSingleServingPasswords.contains(newPassword)) {
                // Das Passwort wurde noch nicht vergeben.

                // An das XML-Objekt ein neues Element hängen
                final Element xmlSingleServingPassword = createXMLSingleServingPasswort(newPassword, _greatestSingleServingPasswordIndex + 1, "ja");
                _xmlObject.appendChild(xmlSingleServingPassword);

                // XML Datei neu speichern
                try {
                    saveXMLFile();

                    // Das Speichern hat geklappt, nun alle Objekte im Speicher ändern

                    // Jetzt wird es gesperrt, damit es nicht noch einmal vergeben
                    // werden kann.
                    _allSingleServingPasswords.add(newPassword);
                    _usableSingleServingPasswords
                        .add(new SingleServingPassword(newPassword, _greatestSingleServingPasswordIndex + 1, true, xmlSingleServingPassword));
                    _greatestSingleServingPasswordIndex++;
                } catch (Exception e) {
                    // Das Passwort wurde nicht angelegt
                    _debug.error("Fehler beim Anlegen eines Einmal-Passworts", e);
                    throw new RequestException(e);
                }
            } else {
                // Das Passwort wurde bereits vergeben
                throw new ConfigurationTaskException("Das Passwort wurde bereits vergeben");
            }
        }

        public synchronized int createNewSingleServingPasswords(final List<String> passwords, final boolean append)
            throws ConfigurationTaskException, RequestException {
            if (!append) {
                try {
                    clearSingleServingPasswords();
                } catch (TransformerException | FileNotFoundException e) {
                    throw new ConfigurationChangeException("Konnte Einmalpasswörter nicht löschen", e);
                }
            } else {
                for (String password : passwords) {
                    if (_allSingleServingPasswords.contains(password)) {
                        // Das Passwort wurde bereits vergeben
                        throw new ConfigurationTaskException("Ein Passwort wurde bereits vergeben");
                    }
                }
            }
            int firstInsertIndex = _greatestSingleServingPasswordIndex + 1;
            final List<Element> xmlElements = new ArrayList<>(passwords.size());
            for (String password : passwords) {
                final Element xmlSingleServingPassword = createXMLSingleServingPasswort(password, _greatestSingleServingPasswordIndex + 1, "ja");
                _xmlObject.appendChild(xmlSingleServingPassword);
                xmlElements.add(xmlSingleServingPassword);
                _greatestSingleServingPasswordIndex++;
            }
            // XML Datei neu speichern
            try {
                saveXMLFile();

                // Das Speichern hat geklappt, nun alle Objekte im Speicher ändern
                _allSingleServingPasswords.addAll(passwords);
                for (int i = 0; i < passwords.size(); i++) {
                    final String password = passwords.get(i);
                    _usableSingleServingPasswords.add(new SingleServingPassword(password, firstInsertIndex + i, true, xmlElements.get(i)));
                }
            } catch (Exception e) {
                // Die Passwörter wurden nicht angelegt
                _debug.error("Fehler beim Anlegen eines Einmal-Passworts", e);
                throw new RequestException(e);
            }
            return firstInsertIndex;
        }

        /**
         * Versucht ein Einmal-Passwort zu benutzen. Ist dies möglich, wird das Einmal-Passwort als gebraucht markiert. Wurde eine falsches Passwort
         * übergeben, so wird eine Exception geworfen.
         *
         * @param encryptedPassword           Einmal-Passwort, das vom Benutzer eingegeben wurde
         * @param authentificationText        Text mit dem das Einmal-Passwort verschlüsselt wurde
         * @param authentificationProcessName Name des benutzten Verschlüsselungsverfahren
         *
         * @throws IllegalArgumentException     Falsches Einmal-Passwort
         * @throws NoSuchAlgorithmException     Unbekanntes Verschlüsselungsverfahren
         * @throws UnsupportedEncodingException
         * @throws InvalidKeyException
         * @throws FileNotFoundException
         * @throws TransformerException
         * @deprecated Wird nur von der alten Hmac-basierten Authentifizierung benutzt
         */
        @Deprecated
        public synchronized void useSingleServingPassword(byte[] encryptedPassword, String authentificationText, String authentificationProcessName)
            throws NoSuchAlgorithmException, UnsupportedEncodingException, InvalidKeyException, FileNotFoundException, TransformerException {

            for (SingleServingPassword singleServingPassword : _usableSingleServingPasswords) {
                final SecretKey secretKey =
                    new SecretKeySpec(singleServingPassword.getXmlVerifierText().getBytes("ISO-8859-1"), authentificationProcessName);
                Mac mac = Mac.getInstance("HmacMD5");
                mac.init(secretKey);

                if (Arrays.equals(encryptedPassword, mac.doFinal(authentificationText.getBytes("ISO-8859-1")))) {
                    // Das Passwort ist gültig. Also das Passwort sperren
                    singleServingPassword.setPasswortInvalid();
                    // Speichern war erfolgreich, also kann das Passwort entfernt werden
                    _usableSingleServingPasswords.remove(singleServingPassword);
                    return;
                }
            }
            _debug.warning("Authentifizierungsversuch eines registrierten Benutzers fehlgeschlagen, Benutzername", getUsername());
            throw new IllegalArgumentException("Benutzername/Passwort ist falsch");
        }

        /**
         * Löscht alle Einmalpasswörter eines Benutzers und markiert diese als ungültig
         *
         * @throws TransformerException
         * @throws FileNotFoundException
         */
        public synchronized void clearSingleServingPasswords() throws TransformerException, FileNotFoundException {
            // Alle Kindknoten (Einmalpasswörter löschen)
            while (_xmlObject.hasChildNodes()) {
                _xmlObject.removeChild(_xmlObject.getFirstChild());
            }
            saveXMLFile();
            _allSingleServingPasswords.clear();
            _usableSingleServingPasswords.clear();
            _greatestSingleServingPasswordIndex = -1;
        }

        /**
         * Gibt die Anzahl der verbleidenden, gültigen Einmalpasswörter zurück
         *
         * @return die Anzahl der verbleidenden, gültigen Einmalpasswörter
         */
        public synchronized int countSingleServingPasswords() {
            return _usableSingleServingPasswords.size();
        }

        /**
         * Gibt das Passwort oder den Verifier als Rohdatum mit dem angegebenen Index zurück
         *
         * @param passwordIndex Index (falls -1 wird das normale Passwort zurückgegeben, sonst ein Einmalpasswort mit angegebenem Index)
         *
         * @return Passwort oder leeren String falls kein Passwort vorhanden ist. Der Aufrufer muss sicherstellen, dass man sich nicht mit einem
         *     leeren Passwort einloggen kann.
         */
        private String getXmlVerifierText(final int passwordIndex) {
            if (passwordIndex == -1) {
                return _xmlVerifierText;
            } else {
                for (SingleServingPassword oneTimePassword : _usableSingleServingPasswords) {
                    if (oneTimePassword.getIndex() == passwordIndex) {
                        return oneTimePassword.getXmlVerifierText();
                    }
                }
            }
            _debug.warning("Angegebener Passwort-Index ist nicht am Benutzer " + _username + " vorhanden: " + passwordIndex);
            return "";
        }

        /**
         * Deaktiviert ein Einmalpasswort
         *
         * @param passwordIndex Index
         *
         * @throws RequestException
         */
        public synchronized void disableSingleServingPassword(final int passwordIndex) throws RequestException {
            try {
                if (passwordIndex == -1) {
                    throw new IllegalArgumentException("Das Standard-Passwort kann nicht deaktiviert werden");
                } else {
                    for (Iterator<SingleServingPassword> iterator = _usableSingleServingPasswords.iterator(); iterator.hasNext(); ) {
                        final SingleServingPassword usableSingleServingPassword = iterator.next();
                        if (usableSingleServingPassword.getIndex() == passwordIndex) {
                            usableSingleServingPassword.setPasswortInvalid();
                            iterator.remove();
                            return;
                        }
                    }
                }
                // Das Passwort wurde schon deaktiviert.
                _debug.warning(
                    "Kann Einmalpasswort nicht deaktivieren, Passwort-Index ist nicht am Benutzer " + _username + " vorhanden: " + passwordIndex);
            } catch (Exception e) {
                _debug.error("Fehler beim Deaktivieren eines Einmal-Passworts", e);
                throw new RequestException(e);
            }
        }

        /**
         * Gibt die IDs der benutzbaren Einmalpasswörter zurück
         *
         * @return IDs
         */
        public int[] getUsableIDs() {
            return _usableSingleServingPasswords.stream().mapToInt(SingleServingPassword::getIndex).sorted().toArray();
        }
    }

    /** Speichert alle Informationen zu einem "Einmal-Passwort" (Passwort, Index, "schon gebraucht") */
    private final class SingleServingPassword {

        /** Passwort in Klarschrift */
        private final String _xmlVerifierText;

        /** Index des Passworts */
        private final int _index;
        /** XML Objekt, das die Daten speichert */
        private final Element _xmlObject;
        /** Wurde das Passwort schon einmal benutzt */
        private boolean _passwordUsable;

        /**
         * @param xmlVerifierText Password des Einmal-Passworts, ausgelesen aus der XML-Datei
         * @param index           Index des Passworts
         * @param passwordUsable  Kann das Passwort noch benutzt werden. true = es kann noch benutzt werden; false = es wurde bereits benutzt und kann
         *                        nicht noch einmal benutzt werden
         * @param xmlObject       XML-Objekt, das dem Einmal-Passwort entspricht
         */
        public SingleServingPassword(String xmlVerifierText, int index, boolean passwordUsable, Element xmlObject) {
            _xmlVerifierText = xmlVerifierText;
            _index = index;
            _passwordUsable = passwordUsable;
            _xmlObject = xmlObject;
        }

        /**
         * Passwort-String-Wert in der XML-Datei (entweder ein Klartextpasswort oder ein SRP-Verifier)
         *
         * @return s.o
         */
        private String getXmlVerifierText() {
            return _xmlVerifierText;
        }

        /**
         * Index des Einmal-Passworts
         *
         * @return s.o
         */
        public int getIndex() {
            return _index;
        }

        /**
         * Kann das Passwort noch benutzt werden.
         *
         * @return true = ja; false = nein, es wurde bereits benutzt und darf nicht noch einmal benutzt werden
         */
        public synchronized boolean isPasswordUsable() {
            return _passwordUsable;
        }

        /** Setzt ein Einmal-Passwort auf ungültig und speichert diese Information in der XML-Datei (erst speichern, dann Objekte im Speicher 
         * ändern) */
        public synchronized void setPasswortInvalid() throws FileNotFoundException, TransformerException {
            _xmlObject.setAttribute("gueltig", "nein");
            saveXMLFile();
            _passwordUsable = false;
        }

        @Override
        public String toString() {
            return "SingleServingPassword{" + "_password='" + _xmlVerifierText + '\'' + ", _index=" + _index + ", _passwordUsable=" +
                   _passwordUsable + ", _xmlObject=" + _xmlObject + '}';
        }
    }

    /**
     * Implementierung eines EntityResolvers, der Referenzen auf den Public-Identifier "-//K2S//DTD Verwaltung//DE" ersetzt durch die
     * verwaltungsdaten.dtd Resource-Datei in diesem Package.
     */
    private class ConfigAuthenticationEntityResolver implements EntityResolver {

        /**
         * Löst Referenzen auf external entities wie z.B. DTD-Dateien auf.
         * <p>
         * Angegebene Dateien werden, falls sie im Suchverzeichnis gefunden werden, von dort geladen. Ansonsten wird der normale Mechanismus zum Laden
         * von externen Entities benutzt.
         *
         * @param publicId Der public identifier der externen Entity oder null falls dieser nicht verfügbar ist.
         * @param systemId Der system identifier aus dem XML-Dokument.
         *
         * @return Für Referenzen im Suchverzeichnis wird ein InputSource-Objekt, das mit der entsprechenden Datei im Suchverzeichnis verbunden ist,
         *     zurückgegeben.
         *
         * @throws SAXException Bei Fehlern beim Zugriff auf externe Entities.
         * @throws IOException
         */
        @Override
        public InputSource resolveEntity(String publicId, String systemId) throws SAXException, IOException {
            if (publicId != null && publicId.equals("-//K2S//DTD Authentifizierung//DE")) {
                URL url = this.getClass().getResource("authentication.dtd");
                assert url != null : this.getClass();
                return new InputSource(url.toExternalForm());
            }
            return null;
        }
    }
}
