/*
 * Copyright 2004 by Kappich+Kniß Systemberatung Aachen (K2S)
 * Copyright 2007-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.sys.funclib.concurrent.
 *
 * de.bsvrz.sys.funclib.concurrent is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.sys.funclib.concurrent is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.sys.funclib.concurrent; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.sys.funclib.concurrent;

import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;

/**
 * Queue zum Austausch von Nachrichten zwischen Threads. Mit der Methode {@link #put} können beliebige Objekte (Nachrichten) in die Queue eingetragen
 * (gesendet) werden und i.a. von einem anderen Thread mit den Methoden {@link #take} oder {@link #poll} aus der Queue entnommen (empfangen) werden.
 * Die generische Queue kann durch Angabe eines Typs auf diesen bestimmten Objekttypen arbeiten. Die Anzahl der Nachrichten in der Queue ist nicht
 * beschränkt. Die Nachrichten werden in der Reihenfolge empfangen in der sie versendet wurden (first in first out, FIFO).
 *
 * @author Kappich Systemberatung
 */
public final class UnboundedQueue<E> {

    /**
     * {@link LinkedBlockingQueue} erlaubt keine null-Elemente, die UnboundedQueue aus historischen Gründen aber schon. Also nulls durch dieses
     * Platzhalterobjekt ersetzen.
     */
    private static final Object NULL_OBJECT = new Object();
    private final LinkedBlockingQueue<Object> _queue = new LinkedBlockingQueue<>();

    /**
     * Sendet eine Nachricht an die Queue.
     *
     * @param message Zu versendende Nachricht.
     */
    public void put(E message) {
        Object toInsert = message;
        if (toInsert == null) {
            toInsert = NULL_OBJECT;
        }
        if (!_queue.offer(toInsert)) {
            while (true) {
                try {
                    _queue.put(toInsert);
                    break;
                } catch (InterruptedException ignored) {
                }
            }
        }
    }

    /**
     * Empfängt und entfernt eine Nachricht aus der Queue. Wenn die Queue leer ist, dann wartet die Methode, bis eine Nachricht in die Queue gesendet
     * wurde. Wenn mehrere Nachrichten in der Queue vorhanden sind, wird die Nachricht empfangen und entfernt, die als erstes in die Queue gesendet
     * wurde.
     *
     * @return Empfangene Nachricht.
     *
     * @throws InterruptedException Wenn der Thread unterbrochen wurde während auf Nachrichten gewartet wurde.
     */
    public E take() throws InterruptedException {
        Object taken = _queue.take();
        if (NULL_OBJECT.equals(taken)) {
            return null;
        }
        return (E) taken;
    }

    /**
     * Empfängt und entfernt eine Nachricht aus der Queue. Wenn die Queue leer ist, dann wartet die Methode, bis eine Nachricht in die Queue gesendet
     * wurde oder bis die angegebene Zeit verstrichen ist. Wenn mehrere Nachrichten in der Queue vorhanden sind, wird diejenige Nachricht empfangen
     * und entfernt, die als erstes in die Queue gesendet wurde.
     *
     * @param timeout Zeit in Millisekunden, für die bei leerer Queue auf neue Nachrichten gewartet wird.
     *
     * @return Empfangene Nachricht oder {@code null}, wenn nach Ablauf des Timeouts immer noch keine Nachricht in der Queue vorhanden ist.
     *
     * @throws InterruptedException Wenn der Thread unterbrochen wurde während auf Nachrichten gewartet wurde.
     */
    public E poll(long timeout) throws java.lang.InterruptedException {
        Object taken = _queue.poll(timeout, TimeUnit.MILLISECONDS);
        if (NULL_OBJECT.equals(taken)) {
            return null;
        }
        return (E) taken;
    }

    /**
     * Bestimmt die Größe dieser Queue.
     *
     * @return Anzahl verbleibender Elemente in dieser Queue.
     */
    public int size() {
        return _queue.size();
    }
}
