/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.choose;

import com.google.common.base.Splitter;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.kappich.pat.gnd.displayObjectToolkit.DisplayObject;
import de.kappich.pat.gnd.displayObjectToolkit.OnlineDisplayObject;
import de.kappich.pat.gnd.documentation.GndHelp;
import de.kappich.pat.gnd.gnd.GenericNetDisplay;
import de.kappich.pat.gnd.gnd.MapPane;
import de.kappich.pat.gnd.utils.view.GndDialog;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.io.IOException;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;
import java.util.regex.Pattern;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.SwingUtilities;
import javax.swing.border.EmptyBorder;

/**
 * Dieser Dialog gestattet es dem Benutzer Namen, Pids oder IDs von {@link DisplayObject DisplayObjects} oder {@link SystemObject SystemObjects}
 * einzugeben, und dann von der {@link GenericNetDisplay}-Instanz zentriert anzeigen zu lassen.
 *
 * @author Kappich Systemberatung
 */
public class ChooseByDialog extends GndDialog {

    private final GenericNetDisplay _genericNetDisplay;
    private final Collection<Collection<DisplayObject>> _displayObjectCollections;

    private final JTextArea _textArea = new JTextArea("", 30, 40);
    private final JTextArea _notFoundTextArea = new JTextArea("", 5, 40);
    private final JButton _chooseButton = new JButton("Gehe zu");
    private final JRadioButton _nameRadioButton = new JRadioButton("nur Namen");
    private final JRadioButton _pidRadioButton = new JRadioButton("nur Pids");
    private final JRadioButton _idRadioButton = new JRadioButton("nur IDs");
    private final JRadioButton _allRadioButton = new JRadioButton("egal");
    private final JButton _helpButton = new JButton("Hilfe");

    public ChooseByDialog(final GenericNetDisplay genericNetDisplay, Collection<Collection<DisplayObject>> displayObjectLists) {
        super("ChooseByDialog", genericNetDisplay.getFrame(), false);
        _genericNetDisplay = genericNetDisplay;
        _displayObjectCollections = displayObjectLists;
        initLayout();
        initLogic();
        setPositionAndSize();
    }

    private void initLayout() {
        setTitle("Gehe zu");
        JPanel allPanel = new JPanel();
        allPanel.setLayout(new BoxLayout(allPanel, BoxLayout.Y_AXIS));
        add(allPanel);

        JLabel label = new JLabel("Namen, Pids oder IDs eingeben:");
        JPanel labelPanel = new JPanel();
        labelPanel.setLayout(new BoxLayout(labelPanel, BoxLayout.X_AXIS));
        labelPanel.add(label);
        labelPanel.add(Box.createHorizontalGlue());
        labelPanel.setBorder(new EmptyBorder(5, 5, 5, 5));
        allPanel.add(labelPanel);

        JPanel mainPanel = new JPanel();
        mainPanel.setLayout(new BoxLayout(mainPanel, BoxLayout.X_AXIS));
        allPanel.add(mainPanel);

        JPanel textAreasPanel = new JPanel();
        textAreasPanel.setLayout(new BoxLayout(textAreasPanel, BoxLayout.Y_AXIS));
        textAreasPanel.add(new JScrollPane(_textArea));
        textAreasPanel.add(Box.createRigidArea(new Dimension(0, 10)));
        JLabel notFoundLabel = new JLabel("Nicht gefunden:");
        JPanel notFoundPanel = new JPanel();
        notFoundPanel.add(notFoundLabel);
        textAreasPanel.add(notFoundPanel);
        textAreasPanel.add(new JScrollPane(_notFoundTextArea));
        textAreasPanel.setBorder(new EmptyBorder(0, 5, 5, 0));

        mainPanel.add(textAreasPanel);

        JPanel buttonPanel = new JPanel();
        buttonPanel.setLayout(new BoxLayout(buttonPanel, BoxLayout.Y_AXIS));
        buttonPanel.add(Box.createVerticalGlue());
        buttonPanel.add(_nameRadioButton);
        buttonPanel.add(Box.createRigidArea(new Dimension(0, 10)));
        buttonPanel.add(_pidRadioButton);
        buttonPanel.add(Box.createRigidArea(new Dimension(0, 10)));
        buttonPanel.add(_idRadioButton);
        buttonPanel.add(Box.createRigidArea(new Dimension(0, 10)));
        buttonPanel.add(_allRadioButton);
        buttonPanel.add(Box.createRigidArea(new Dimension(0, 50)));
        buttonPanel.add(_chooseButton);
        buttonPanel.add(Box.createRigidArea(new Dimension(0, 30)));
        buttonPanel.add(_helpButton);
        buttonPanel.add(Box.createRigidArea(new Dimension(0, 120)));
        buttonPanel.setBorder(new EmptyBorder(10, 10, 10, 10));

        mainPanel.add(buttonPanel);
    }

    private void readTextAreaLineByLine(final Collection<String> collection) {
        Iterable<String> iterable1 = Splitter.on(Pattern.compile("\r?\n")).split(_textArea.getText());
        // Erwischt \r\n und \n. Alte Macs nutzen nur \r, weshalb nochmal gesplittet wird:
        for (final String anIterable1 : iterable1) {
            Iterable<String> iterable2 = Splitter.on(Pattern.compile("\r")).split(anIterable1);
            for (final String anIterable2 : iterable2) {
                collection.add(anIterable2);
            }
        }
        // Alternative Implementation. Nachteil: wirft aber IOException:
// 		try (LineNumberReader lnr = new LineNumberReader(new StringReader(_textArea.getText()))) {
//			String line;
//			while ((line = lnr.readLine()) != null) {
//				collection.add(line);
//			}
//		}
    }

    @SuppressWarnings("OverlyLongMethod")
    private void initLogic() {
        ButtonGroup bg = new ButtonGroup();
        bg.add(_nameRadioButton);
        bg.add(_pidRadioButton);
        bg.add(_idRadioButton);
        bg.add(_allRadioButton);
        _allRadioButton.setSelected(true);

        _chooseButton.addActionListener(e -> {
            Set<String> nameOrPidOrIdSet = new HashSet<>();
            readTextAreaLineByLine(nameOrPidOrIdSet);
            if (nameOrPidOrIdSet.isEmpty()) {
                return;
            }

            Set<DisplayObject> displayObjectsSet = new HashSet<>();
            Set<String> foundSet = new HashSet<>();
            for (Collection<DisplayObject> displayObjectCollection : _displayObjectCollections) {
                for (DisplayObject displayObject : displayObjectCollection) {
	                if (displayObject instanceof OnlineDisplayObject onlineDisplayObject) {
                        SystemObject systemObject = onlineDisplayObject.getSystemObject();
                        if (_allRadioButton.isSelected() || _nameRadioButton.isSelected()) {
                            if (nameOrPidOrIdSet.contains(systemObject.getName())) {
                                displayObjectsSet.add(displayObject);
                                foundSet.add(systemObject.getName());
                            }
                        }
                        if (_allRadioButton.isSelected() || _pidRadioButton.isSelected()) {
                            if (nameOrPidOrIdSet.contains(systemObject.getPid())) {
                                displayObjectsSet.add(displayObject);
                                foundSet.add(systemObject.getPid());
                            }
                        }
                        if (_allRadioButton.isSelected() || _idRadioButton.isSelected()) {
                            if (nameOrPidOrIdSet.contains(Long.toString(systemObject.getId()))) {
                                displayObjectsSet.add(displayObject);
                                foundSet.add(Long.toString(systemObject.getId()));
                            }
                        }
                    } else {
                        if (_allRadioButton.isSelected() || _nameRadioButton.isSelected()) {
                            if (nameOrPidOrIdSet.contains(displayObject.getName())) {
                                displayObjectsSet.add(displayObject);
                                foundSet.add(displayObject.getName());
                            }
                        }
                    }
                }
            }
            StringBuilder sb = new StringBuilder();
            for (String found : foundSet) {
                sb.append(found).append("\n");
            }
            _textArea.setText(sb.toString());
            _notFoundTextArea.setText("");
            if (nameOrPidOrIdSet.size() > foundSet.size()) {
                for (String nameOrPidOrId : nameOrPidOrIdSet) {
                    if (nameOrPidOrId != null && !nameOrPidOrId.isEmpty() && !foundSet.contains(nameOrPidOrId)) {
                        _notFoundTextArea.append(nameOrPidOrId + System.lineSeparator());
                    }
                }
            }
            MapPane mapPane = _genericNetDisplay.getMapPane();
            mapPane.setSelectedObjects(displayObjectsSet);
            mapPane.focusOnObjects(displayObjectsSet);
        });

        _helpButton.addActionListener(e -> GndHelp.openHelp("#gotoByNamePidOrId"));

        _textArea.addMouseListener(new MouseListener() {
            @Override
            public void mouseClicked(final MouseEvent e) {
            }

            @Override
            public void mousePressed(final MouseEvent e) {
                if (SwingUtilities.isRightMouseButton(e)) {
                    try {
                        String data = (String) Toolkit.getDefaultToolkit().getSystemClipboard().getData(DataFlavor.stringFlavor);
                        _textArea.append(data);
                        _textArea.append("\n");
                    } catch (UnsupportedFlavorException | IOException ignore) {
                        JOptionPane.showMessageDialog(ChooseByDialog.this.getDialog(),
                                                      "Der Inhalt der Zwischenablage konnte nicht in das Textfeld kopiert werden.", "Fehler",
                                                      JOptionPane.ERROR_MESSAGE);
                    }
                }
            }

            @Override
            public void mouseReleased(final MouseEvent e) {
            }

            @Override
            public void mouseEntered(final MouseEvent e) {
            }

            @Override
            public void mouseExited(final MouseEvent e) {
            }
        });
    }

    private void setPositionAndSize() {
        setPositionAndSize(0, 0, 0, 0, true, 0, 0);
    }

    @Override
    public String toString() {
        return "ChooseByDialog{}";
    }
}
