/*
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.coorTransform;

/**
 * Eine Klasse für WGS84-Koordinaten. Sie könnte auch für andere Ellipsoid-Koordinaten verwendet werden.
 *
 * @author Kappich Systemberatung
 * @author BitCtrl Systems GmbH, Gieseler
 */
public class WGS84Coordinate {

    /**
     * kleinster zulässiger Wert für die Länge.
     */
    private static final double MIN_LAENGE = -180;

    /**
     * größter zulässiger Wert für die Länge.
     */
    private static final double MAX_LAENGE = 180;

    /**
     * kleinster zulässiger Wert für die Breite.
     */
    private static final double MIN_BREITE = -90;

    /**
     * größter zulässiger Wert für die Breite.
     */
    private static final double MAX_BREITE = 90;

    /**
     * Länge.
     */
    private double _laenge;

    /**
     * Breite.
     */
    private double _breite;

    /**
     * Konstruktor für eine WGS84-Koordinate.
     *
     * @param laenge geographische Länge in Dezimalgrad
     * @param breite geographische Breite in Dezimalgrad
     */
    public WGS84Coordinate(double laenge, double breite) {
        if (testBreite(breite)) {
            throw new IllegalArgumentException("Der Wert für die Breite ist ungültig!"); //$NON-NLS-1$
        }

        if (testLaenge(laenge)) {
            throw new IllegalArgumentException("Der Wert für die Länge ist ungültig!"); //$NON-NLS-1$
        }

        _laenge = laenge;
        _breite = breite;
    }

    /**
     * Testet die Breite auf Gültigkeit.
     *
     * @param tbreite Breite
     *
     * @return true, wenn ok sonst false
     */
    private static boolean testBreite(double tbreite) {
        return (tbreite < MIN_BREITE) || (tbreite > MAX_BREITE);
    }

    /**
     * Testet die Länge auf Gültigkeit.
     *
     * @param tlaenge Länge
     *
     * @return true, wenn ok sonst false
     */
    private static boolean testLaenge(double tlaenge) {
        return (tlaenge < MIN_LAENGE) || (tlaenge > MAX_LAENGE);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final WGS84Coordinate other = (WGS84Coordinate) obj;
        if (Double.doubleToLongBits(_breite) != Double.doubleToLongBits(other._breite)) {
            return false;
        }
        return Double.doubleToLongBits(_laenge) == Double.doubleToLongBits(other._laenge);
    }

    /**
     * Gibt die geographische Breite zurück.
     *
     * @return geographische Breite in Dezimalgrad
     */
    public double getBreite() {
        return _breite;
    }

    /**
     * Setzt die geographische Breite.
     *
     * @param neuebreite neue geographische Breite in Dezimalgrad
     */
    public void setBreite(double neuebreite) {
        if (testBreite(neuebreite)) {
            throw new IllegalArgumentException("Der Wert für die Breite ist ungültig!"); //$NON-NLS-1$
        }

        _breite = neuebreite;
    }

    /**
     * Gibt die geographische Länge zurück.
     *
     * @return geographische Länge in Dezimalgrad
     */
    public double getLaenge() {
        return _laenge;
    }

    /**
     * Setzt die geographische Länge.
     *
     * @param neuelaenge neue geographische Länge in Dezimalgrad
     */
    public void setLaenge(double neuelaenge) {
        if (testLaenge(neuelaenge)) {
            throw new IllegalArgumentException("Der Wert für die Länge ist ungültig!"); //$NON-NLS-1$
        }

        _laenge = neuelaenge;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
	    result = prime * result + Double.hashCode(_breite);
	    result = prime * result + Double.hashCode(_laenge);
        return result;
    }

    @Override
    public String toString() {
        return "WGS84Coordinate{" + "_laenge=" + _laenge + ", _breite=" + _breite + '}';
    }
}
