/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2021 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.kappich.pat.gnd.csv;

import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.kappich.pat.gnd.gnd.PreferencesHandler;
import de.kappich.pat.gnd.utils.view.PreferencesDeleter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;
import javax.swing.table.AbstractTableModel;

/**
 * @author Kappich Systemberatung
 */
public final class CsvFormatManager extends AbstractTableModel {

    // Statische statt lazy Initialisierung da lazy nicht thread-safe. Statische Initialisierung
    // hat allerdings den Nachteil bei Exceptions. Deshalb immer auch Debug benutzen.
    private static final CsvFormatManager _instance = new CsvFormatManager();

    private static final Debug _debug = Debug.getLogger();
    private static final String[] COLUMN_NAMES = {"Name des CSV-Formats"};
    @SuppressWarnings("SerializableNonStaticInnerClassWithoutSerialVersionUID")
    private final List<CsvFormat> _formats = new ArrayList<>() {
        public boolean add(CsvFormat format) {
            int index = Collections.binarySearch(this, format);
            if (index < 0) {
                index = ~index;
            }
            super.add(index, format);
            return true;
        }
    };
    private final Set<String> _unchangeables = new HashSet<>();
    private final Map<String, CsvFormat> _formatHashMap = new HashMap<>();
    private final List<CsvFormatManagerChangeListener> _listeners = new CopyOnWriteArrayList<>();

    /*
     * Ein CsvFormatManager verwaltet alle zur GND gehörenden CsvFormats.
     */
    private CsvFormatManager() {
        initUserDefinedFormats();
    }

    /**
     * Die für ein Singleton übliche Methode, um an die einzige Instanz der Klasse zu gelangen.
     *
     * @return den CsvFormatManager
     */
    public static CsvFormatManager getInstance() {
        return _instance;
    }

    /**
     * Mit Hilfe dieser Methode kann man den CsvFormatManager dazu zwingen, sich erneut zu konstruieren, was etwa nach dem Importieren von Präferenzen
     * sinnvoll ist.
     */
    public static void refreshInstance() {
        getInstance()._formats.clear();
        getInstance()._unchangeables.clear();
        getInstance()._formatHashMap.clear();
        getInstance().initUserDefinedFormats();
    }

    /**
     * Gibt das Preferences-Objekt für den Ausgangspunkt zur Ablage der Präferenzen des CsvFormatManager zurück.
     *
     * @return den Ausgangsknoten
     */
    private static Preferences getPreferenceStartPath() {
        return PreferencesHandler.getInstance().getPreferenceStartPath().node("CSV-Formate");
    }

    /**
     * Gibt eine sortierte Kopie der Liste aller CsvFormate zurück.
     *
     * @return die Liste aller CsvFormate
     */
    public List<CsvFormat> getCsvFormats() {
        return Collections.unmodifiableList(_formats);
    }

    /**
     * Gibt ein sortiertes Array aller CsvFormate-Namen zurück.
     *
     * @return das rray aller CsvFormat-Namen
     */
    public String[] getCsvFormatNames() {
        String[] names = new String[_formats.size()];
        for (int i = 0; i < _formats.size(); ++i) {
            names[i] = _formats.get(i).getName();
        }
        return names;
    }

    /**
     * Gibt das CsvFormat mit dem übergebenen Namen zurück.
     *
     * @param formatName der Name
     *
     * @return den geforderten CsvFormat
     */
    public CsvFormat getCsvFormat(String formatName) {
        return _formatHashMap.get(formatName);
    }

    /**
     * Gibt das CsvFormat an der i-ten Stelle der CsvFormat-Liste zurück, wobei die Zählung mit 0 beginnt.
     *
     * @param i ein Index
     *
     * @return den geforderten CsvFormat
     */
    public CsvFormat getCsvFormat(int i) {
        return _formats.get(i);
    }

    /**
     * Gibt {@code true} zurück, wenn es ein CsvFormat gibt, dessen Name sich allenfalls bezüglich Klein-Groß-Schreibung unterseheidet. Sonst {@code
     * false}.
     *
     * @param formatName
     *
     * @return s.o.
     */
    public boolean hasCsvFormatToLowerCase(String formatName) {
        for (CsvFormat f : _formats) {
            if (f.getName().toLowerCase().equals(formatName.toLowerCase())) {
                return true;
            }
        }
        return false;
    }

    /*
     * Gehört zur Implementation des TableModel.
     */
    @Override
    public int getColumnCount() {
        return COLUMN_NAMES.length;
    }

    /*
     * Gehört zur Implementation des TableModel.
     */
    @Override
    public int getRowCount() {
        return _formats.size();
    }

    /*
     * Gehört zur Implementation des TableModel.
     */
    @Override
    public Object getValueAt(int rowIndex, int columnIndex) {
        return _formats.get(rowIndex).getName();
    }

    /**
     * Definiert den Tooltipp für die Felder der Tabelle. Gehört zur Implementation des TableModel.
     *
     * @param rowIndex ein Zeilenindex
     *
     * @return ein Tooltipp
     */
    @Nullable
    public String getTooltipAt(int rowIndex) {
        if (rowIndex >= 0 && rowIndex < getRowCount()) {
            return _formats.get(rowIndex).getInfo();
        } else {
            return null;
        }
    }

    /*
     * Gehört zur Implementation des TableModel.
     */
    @Override
    public String getColumnName(int columnIndex) {
        return COLUMN_NAMES[columnIndex];
    }

    /**
     * Fügt das CsvFormat der CsvFormat-Liste an, wenn nicht schon ein gleichnamiges CsvFormat existiert.
     *
     * @param format ein CsvFormat
     *
     * @throws IllegalArgumentException wenn bereits ein gleichnamiges CsvFormat existiert
     */
    public void addCsvFormat(CsvFormat format) throws IllegalArgumentException {
        if (_formatHashMap.containsKey(format.getName())) {
            throw new IllegalArgumentException("Ein CsvFormat mit diesem Namen existiert bereits.");
        }
        // Klein-/Großschreibung nicht (mehr) signifikant:
        for (CsvFormat f : _formats) {
            if (format.equals(f)) {
                throw new IllegalArgumentException(
                    "Es existiert bereits ein CsvFormat, dessen Name sich nur bezüglich Klein-/Großschreibung unterscheidet.");
            }
        }
        _formats.add(format);
        _formatHashMap.put(format.getName(), format);
        format.putPreferences(getPreferenceStartPath());
//		final int lastIndex = _formats.size()-1;
//		fireTableRowsInserted( lastIndex, lastIndex);   // das ist falsch, da _formats sortiert ist
        fireTableDataChanged();
        notifyChangeListenersCsvFormatAdded(format);
    }

    /**
     * Ändert das gleichnamige CsvFormat.
     *
     * @param format ein CsvFormat
     *
     * @throws IllegalArgumentException wenn das CsvFormat nicht bekannt ist
     */
    public void changeCsvFormat(CsvFormat format) throws IllegalArgumentException {
        final String name = format.getName();
        if (!_formatHashMap.containsKey(name)) {
            throw new IllegalArgumentException("Ein CsvFormat mit diesem Namen existiert nicht.");
        }
        final CsvFormat existingFormat = _formatHashMap.get(name);
        existingFormat.deletePreferences(getPreferenceStartPath());
        CsvFormat.copy(existingFormat, format, false);
        for (int i = 0; i < _formats.size(); i++) {
            if (_formats.get(i).getName().equals(name)) {
                fireTableRowsUpdated(i, i);
                break;
            }
        }
        format.putPreferences(getPreferenceStartPath());
        notifyChangeListenersCsvFormatChanged(format);
    }

    /**
     * Entfernt das übergebene CsvFormat auf Basis eines Namensvergleichs aus der Liste aller CsvFormate und damit auch aus den Präferenzen.
     * Entspricht einer kompletten Löschung des CsvFormats. Wirkt aber nicht für im Kode definierte CsvFormate. Wird ein CsvFormat gelöscht, so erhält
     * man den Rückgabewert {@code true}, sonst {@code false}.
     *
     * @param format ein CsvFormat
     *
     * @return {@code true} genau dann, wenn der CsvFormat gelöscht wurde
     */
    public boolean removeCsvFormat(CsvFormat format) {
        final String name = format.getName();
        if (!_unchangeables.contains(name)) {
            final int index = remove(name);
            if (index > -1) {
                format.deletePreferences(getPreferenceStartPath());
                fireTableRowsDeleted(index, index);
                notifyChangeListenersCsvFormatRemoved(name);
                return true;
            }
        }
        return false;
    }

    /**
     * Löscht alle benutzerdefinierten CsvFormate.
     */
    public void clearCsvFormats() {
        for (CsvFormat f : _formats) {
            removeCsvFormat(f);
        }
    }

    private void initUserDefinedFormats() {
        Preferences classPrefs = getPreferenceStartPath();
        String[] formats;
        try {
            formats = classPrefs.childrenNames();
        } catch (BackingStoreException | IllegalStateException ignored) {
            PreferencesDeleter pd = new PreferencesDeleter("Die benutzer-definierten CSV-Formate können nicht geladen werden.", classPrefs);
            pd.run();
            return;
        }
        for (String formatName : formats) {
            Preferences formatPrefs = classPrefs.node(formatName);
            CsvFormat format = new CsvFormat();
            if (format.initializeFromPreferences(formatPrefs)) {
                _formats.add(format);
                _formatHashMap.put(format.getName(), format);
            }
        }
    }

    private int remove(final String formatName) {
        int index = 0;
        for (CsvFormat format : _formats) {
            if (formatName.equals(format.getName())) {
                _formats.remove(index);
                _formatHashMap.remove(formatName);
                return index;
            }
            index++;
        }
        return -1;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("[CvsFormatManager: ");
        for (CsvFormat format : _formats) {
            sb.append(format.toString());
        }
        if (!_unchangeables.isEmpty()) {
            sb.append("[Unveränderebare CsvFormate: ");
            for (String name : _unchangeables) {
                sb.append("[").append(name).append("]");
            }
        }
        sb.append("]");
        return sb.toString();
    }

    /**
     * Gibt {@code true} zurück, wenn das CsvFormat veränderbar ist. Im Moment ist ein CsvFormat genau dann unveränderbar, wenn er im Kode definiert
     * ist.
     *
     * @param format ein CsvFormat
     *
     * @return {@code true} genau dann, wenn das CsvFormat veränderbar ist
     */
    public boolean isChangeable(CsvFormat format) {
        return !_unchangeables.contains(format.getName());
    }

    /**
     * Fügt das übergebene Objekt der Liste der auf CsvFormat-Änderungen angemeldeten Objekte hinzu.
     *
     * @param listener ein Listener
     */
    public void addChangeListener(CsvFormatManagerChangeListener listener) {
        _listeners.add(listener);
    }

    /**
     * Entfernt das übergebene Objekt aus der Liste der auf CsvFormat-Änderungen angemeldeten Objekte.
     *
     * @param listener ein Listener
     */
    public void removeChangeListener(CsvFormatManagerChangeListener listener) {
        _listeners.remove(listener);
    }

    /**
     * Informiert die auf CsvFormat-Änderungen angemeldeten Objekte über ein neu hinzugefügtes CsvFormat.
     *
     * @param format ein CsvFormat
     */
    private void notifyChangeListenersCsvFormatAdded(final CsvFormat format) {
        for (CsvFormatManagerChangeListener changeListener : _listeners) {
            changeListener.csvFormatAdded(format);
        }
    }

    /**
     * Informiert die auf CsvFormat-Änderungen angemeldeten Objekte über ein geändertes CsvFormat.
     *
     * @param format ein CsvFormat
     */
    private void notifyChangeListenersCsvFormatChanged(final CsvFormat format) {
        for (CsvFormatManagerChangeListener changeListener : _listeners) {
            changeListener.csvFormatChanged(format);
        }
    }

    /**
     * Informiert die auf CsvFormat-Änderungen angemeldeten Objekte über ein gelöschtes CsvFormat.
     *
     * @param formatName ein CsvFormat-Name
     */
    private void notifyChangeListenersCsvFormatRemoved(final String formatName) {
        for (CsvFormatManagerChangeListener changeListener : _listeners) {
            changeListener.csvFormatRemoved(formatName);
        }
    }

    /**
     * Ein Interface für Listener, die über das Hinzufügen, Löschen und Ändern von CsvFormaten informiert werden wollen.
     *
     * @author Kappich Systemberatung
     */
    public interface CsvFormatManagerChangeListener {
        /**
         * Diese Methode wird aufgerufen, wenn das CsvFormat hinzugefügt wurde.
         *
         * @param format ein CsvFormat
         */
        void csvFormatAdded(final CsvFormat format);

        /**
         * Diese Methode wird aufgerufen, wenn das CsvFormat geändert wurde.
         *
         * @param format ein CsvFormat
         */
        void csvFormatChanged(final CsvFormat format);

        /**
         * Diese Methode wird aufgerufen, wenn das CsvFormat gelöscht wurde.
         *
         * @param formatName ein CsvFormat-Name
         */
        void csvFormatRemoved(final String formatName);
    }

}
