/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.csv.view;

import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.filechooser.AwtFileChooser;
import de.kappich.pat.gnd.csv.CsvFormat;
import de.kappich.pat.gnd.csv.CsvFormatManager;
import de.kappich.pat.gnd.csv.CsvPriority;
import de.kappich.pat.gnd.csv.CsvQuote;
import de.kappich.pat.gnd.csv.CsvSeparator;
import de.kappich.pat.gnd.documentation.GndHelp;
import de.kappich.pat.gnd.extLocRef.ReferenceHierarchyManager;
import de.kappich.pat.gnd.utils.SpringUtilities;
import de.kappich.pat.gnd.utils.view.GndFrame;
import de.kappich.sys.funclib.csv.CsvReader;
import de.kappich.sys.funclib.csv.IterableCsvData;
import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;
import javax.swing.BorderFactory;
import javax.swing.ComboBoxModel;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.SpringLayout;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.filechooser.FileNameExtensionFilter;

/**
 * Der Dialog zur Definition und Bearbeitung eines {@link CsvFormat}.
 *
 * @author Kappich Systemberatung
 */
public class CsvFormatDefinitionDialog extends GndFrame {

    private static final Debug _debug = Debug.getLogger();
    private static final int _lowerHeightBound = 460;
    private static final int _upperHeightBound = Toolkit.getDefaultToolkit().getScreenSize().height;
    private final List<EditableListenButton> _listeningButtons = new ArrayList<>();
    private final JTextField _nameTextField = new JTextField();
    private final JTextField _infoTextField = new JTextField();
    private final JTextField _csvTextField = new JTextField();
    private final JButton _csvOpenButton = new JButton("Öffnen");
    private final JComboBox<String> _nameComboBox = new JComboBox<>();
    private final JComboBox<String> _xComboBox = new JComboBox<>();
    private final JComboBox<String> _yComboBox = new JComboBox<>();
    private final JComboBox<String> _lineComboBox = new JComboBox<>();
    private final JComboBox<String> _offsetComboBox = new JComboBox<>();
    private final JComboBox<String> _objectComboBox = new JComboBox<>();
    private final JComboBox<String> _refHierarchyComboBox = new JComboBox<>();
    private final JComboBox<CsvPriority> _priorityComboBox = new JComboBox<>(CsvPriority.values());
    private final EditableListenButton _saveButton = new EditableListenButton("Speichern");
    private final JButton _cancelButton = new JButton("Dialog schließen");
    private final JButton _helpButton = new JButton("Hilfe");
    private CsvFormat _scratchFormat;
    private CsvFormat _unchangeableOriginalCsvFormat;
    private boolean _editable;
    private boolean _nameChangeable;
    private boolean _somethingChanged;

    /**
     * Konstruktor zum Anlegen eines neuen CsvFormat-Editors.
     *
     * @param format         ein CsvFormat
     * @param editable       ist das CsvFormat veränderbar?
     * @param nameChangeable ist der Name und damit die Identität änderbar?
     * @param title          der Titel des Fensters
     */
    public CsvFormatDefinitionDialog(final CsvFormat format, final boolean editable, final boolean nameChangeable, final String title) {
        // Parameterbehandlung
        super("CsvFormatDefinitionDialog", title);
        if (format != null) {
            _scratchFormat = format.getCopy();
            _unchangeableOriginalCsvFormat = _scratchFormat.getCopy();
        } else {
            String err = "Ein CsvFormatDefinitionDialog kann nicht mit einem Null-CSV-Format konstruiert werden.";
            _debug.error(err);
            throw new IllegalArgumentException(err);
        }
        _editable = editable;
        _nameChangeable = nameChangeable;

        // Layout:
        initLayout();

        // Logik:
        initLogic();

        // Weitere Erscheinung:
        setEditable(_editable, _nameChangeable);
        setPositionAndSize();
        setVisible(true);
        toFront();
    }

    private static void selectSomething(final JComboBox<String> comboBox, final String prefix) {
        ComboBoxModel<String> model = comboBox.getModel();
        String pf = prefix.toLowerCase();
        for (int i = 0; i < model.getSize(); ++i) {
            String s = model.getElementAt(i);
            if (s.toLowerCase().startsWith(pf)) {
                comboBox.setSelectedItem(s);
            }
        }
    }

    @SuppressWarnings("OverlyLongMethod")
    private void initLayout() {
        setLayout(new BorderLayout());
        Dimension labelSize = new Dimension(130, 20);

        // Oberer Teil mit Name, Info usw.
        {
            JLabel nameLabel = new JLabel("Name: ");
            nameLabel.setPreferredSize(labelSize);
            _nameTextField.setText(_scratchFormat.getName());
            _nameTextField.setEditable(_nameChangeable);

            JLabel infoLabel = new JLabel("Info: ");
            _infoTextField.setText(_scratchFormat.getInfo());
            infoLabel.setPreferredSize(labelSize);

            JLabel csvLabel = new JLabel("CSV-Datei: ");
            csvLabel.setPreferredSize(labelSize);
            JPanel csvPanel = new JPanel();
            csvPanel.setLayout(new SpringLayout());
            csvPanel.add(_csvTextField);
            csvPanel.add(_csvOpenButton);
            SpringUtilities.makeCompactGrid(csvPanel, 2, 5, 5);

            JPanel upperPanel = new JPanel();
            upperPanel.setLayout(new SpringLayout());
            upperPanel.add(nameLabel);
            upperPanel.add(_nameTextField);
            upperPanel.add(infoLabel);
            upperPanel.add(_infoTextField);
            upperPanel.add(csvLabel);
            upperPanel.add(csvPanel);
            upperPanel.setBorder(BorderFactory.createEmptyBorder(5, 10, 5, 10));
            SpringUtilities.makeCompactGrid(upperPanel, 2, 5, 5);
            add(upperPanel, BorderLayout.NORTH);
        }

        // Mittlerer Teil mit Spaltendefinitionen
        {
            JLabel nameColumnLabel = new JLabel("Name:");
            nameColumnLabel.setPreferredSize(labelSize);
            nameColumnLabel.setToolTipText("Spaltenname der Namensspalte");
            JLabel xColumnLabel = new JLabel("X (WGS-Länge): ");
            xColumnLabel.setPreferredSize(labelSize);
            xColumnLabel.setToolTipText("Spaltenname der x-Koordinate (Länge)");
            JLabel yColumnLabel = new JLabel("Y (WGS-Breite):");
            yColumnLabel.setPreferredSize(labelSize);
            yColumnLabel.setToolTipText("Spaltenname der y-Koordinate (Breite)");
            JLabel lineColumnName = new JLabel("Liniereferenz (Pid):");
            lineColumnName.setPreferredSize(labelSize);
            lineColumnName.setToolTipText("Spaltenname der Linienreferenzen");
            JLabel offsetColumnName = new JLabel("Offset:");
            offsetColumnName.setPreferredSize(labelSize);
            offsetColumnName.setToolTipText("Spaltenname des Linienoffset");
            JLabel objectColumnName = new JLabel("Objektreferenz (Pid):");
            objectColumnName.setPreferredSize(labelSize);
            objectColumnName.setToolTipText("Spaltenname für Objektreferenz:");
            JLabel referenceHierarchyLabel = new JLabel("EOR-Hierarchie:");
            referenceHierarchyLabel.setPreferredSize(labelSize);
            referenceHierarchyLabel.setToolTipText("Eine EOR-Hierarchie für Objektreferenzen");
            JLabel priorityLabel = new JLabel("Priorität: ");
            priorityLabel.setPreferredSize(labelSize);
            priorityLabel.setToolTipText("Die Priorität der verschiedenen Georeferenzen");

            JPanel middlePanel = new JPanel();
            middlePanel.setLayout(new SpringLayout());
            middlePanel.add(nameColumnLabel);
            middlePanel.add(_nameComboBox);
            middlePanel.add(xColumnLabel);
            middlePanel.add(_xComboBox);
            middlePanel.add(yColumnLabel);
            middlePanel.add(_yComboBox);
            middlePanel.add(lineColumnName);
            middlePanel.add(_lineComboBox);
            middlePanel.add(offsetColumnName);
            middlePanel.add(_offsetComboBox);
            middlePanel.add(objectColumnName);
            middlePanel.add(_objectComboBox);
            middlePanel.add(referenceHierarchyLabel);
            middlePanel.add(_refHierarchyComboBox);
            middlePanel.add(priorityLabel);
            middlePanel.add(_priorityComboBox);
            middlePanel.setBorder(BorderFactory.createTitledBorder("Spaltenfestlegungen"));
            SpringUtilities.makeCompactGrid(middlePanel, 2, 5, 5);
            add(middlePanel, BorderLayout.CENTER);
        }

        // Unterer Teil mit Buttons

        JPanel buttonsPanel = new JPanel();
        buttonsPanel.setLayout(new SpringLayout());

        buttonsPanel.add(_saveButton);
        buttonsPanel.add(_cancelButton);
        buttonsPanel.add(_helpButton);

        buttonsPanel.setBorder(BorderFactory.createEmptyBorder(5, 10, 5, 10));
        SpringUtilities.makeCompactGrid(buttonsPanel, 3, 20, 5);
        add(buttonsPanel, BorderLayout.SOUTH);

    }

    private void initLogic() {
        addChangeListeners();
        addFrameListener();
        initComboBoxes();
        initFileChooser();
        addButtonListener(_saveButton, _cancelButton, _helpButton);
        _somethingChanged = false;
    }

    public final void setPositionAndSize() {
        Dimension d = new Dimension(0, 0);   // TODO
        setPositionAndSize(690, 202 + (int) d.getHeight(), 660, 0, false, 690,
                           Math.min(Math.max(202 + (int) d.getHeight(), _lowerHeightBound), _upperHeightBound));
    }

    private void addChangeListeners() {
        {
            _nameTextField.getDocument().addDocumentListener(new DocumentListener() {
                @Override
                public void changedUpdate(DocumentEvent e) {
                    _somethingChanged = true;
                }

                @Override
                public void removeUpdate(DocumentEvent e) {
                    _somethingChanged = true;
                }

                @Override
                public void insertUpdate(DocumentEvent e) {
                    _somethingChanged = true;
                }
            });
        }

        {
            _infoTextField.getDocument().addDocumentListener(new DocumentListener() {
                @Override
                public void changedUpdate(DocumentEvent e) {
                    _somethingChanged = true;
                }

                @Override
                public void removeUpdate(DocumentEvent e) {
                    _somethingChanged = true;
                }

                @Override
                public void insertUpdate(DocumentEvent e) {
                    _somethingChanged = true;
                }
            });
        }
        {
            ActionListener nameColumnListener = e -> _somethingChanged = true;
            _nameComboBox.addActionListener(nameColumnListener);
        }
        {
            ActionListener xColumnListener = e -> _somethingChanged = true;
            _xComboBox.addActionListener(xColumnListener);
        }
        {
            ActionListener yColumnListener = e -> _somethingChanged = true;
            _yComboBox.addActionListener(yColumnListener);
        }
        {
            ActionListener lineColumnListener = e -> _somethingChanged = true;
            _lineComboBox.addActionListener(lineColumnListener);
        }
        {
            ActionListener offsetColumnListener = e -> _somethingChanged = true;
            _offsetComboBox.addActionListener(offsetColumnListener);
        }
        {
            ActionListener objectColumnListener = e -> _somethingChanged = true;
            _objectComboBox.addActionListener(objectColumnListener);
        }
        {
            ActionListener refHierarchyColumnListener = e -> _somethingChanged = true;
            _refHierarchyComboBox.addActionListener(refHierarchyColumnListener);
        }
        {
            ActionListener priorityListener = e -> _somethingChanged = true;
            _priorityComboBox.addActionListener(priorityListener);
        }
    }

    /**
     * Setzt die Felder des CSV-Format-Editors mit den Informationen des übergebenen CSV-Formats und aktiviert die Veränderbarkeit gemäß der zwei
     * boolschen Werte.
     *
     * @param format         ein CSV-Format
     * @param editable       ist das CSV-Format veränderbar?
     * @param nameChangeable ist der Name und damit die Identität des CSV-Formats änderbar?
     */
    public void setCsvFormat(CsvFormat format, boolean editable, boolean nameChangeable) {
        _scratchFormat = format;
        _unchangeableOriginalCsvFormat = _scratchFormat.getCopy();
        _editable = editable;
        _nameTextField.setText(_scratchFormat.getName());
        _infoTextField.setText(_scratchFormat.getInfo());
        _csvTextField.setText("");
        _nameComboBox.getEditor().setItem(_scratchFormat.getNameColumn());
        _xComboBox.getEditor().setItem(_scratchFormat.getXColumn());
        _yComboBox.getEditor().setItem(_scratchFormat.getYColumn());
        _lineComboBox.getEditor().setItem(_scratchFormat.getLineColumn());
        _offsetComboBox.getEditor().setItem(_scratchFormat.getOffsetColumn());
        _objectComboBox.getEditor().setItem(_scratchFormat.getObjectColumn());
        _refHierarchyComboBox.setSelectedItem(_scratchFormat.getReferenceHierarchy());
        _priorityComboBox.setSelectedItem(_scratchFormat.getCsvPriority());
        setEditable(editable, nameChangeable);
        for (EditableListenButton elButton : _listeningButtons) {
            elButton.formatsEditingStateChanged(_editable);
        }
        _somethingChanged = false;
    }

    /**
     * Setzt den Wert der internen Variable, die darüber entscheidet, ob die Informationen des angezeigten CsvFormats veränderbar sind, und macht
     * Textfelder veränderbar oder nicht, aktiviert bzw. deaktiviert Knöpfe usw.
     *
     * @param editable       ist das CsvFormat veränderbar?
     * @param nameChangeable ist der Name und damit die Identität des CsvFormats änderbar?
     */
    public final void setEditable(boolean editable, boolean nameChangeable) {
        _editable = editable;
        _nameChangeable = nameChangeable;
        for (EditableListenButton elButton : _listeningButtons) {
            elButton.formatsEditingStateChanged(_editable);
        }
        _nameTextField.setEditable(_nameChangeable);
        _infoTextField.setEditable(_editable);
    }

    private void addButtonListener(final JButton saveButton, final JButton cancelButton, final JButton helpButton) {

        ActionListener actionListenerSave = e -> saveCsvFormat();
        saveButton.addActionListener(actionListenerSave);

        ActionListener actionListenerCancel = e -> {
            if (_editable && _somethingChanged) {
                if (!saveChangesWanted()) {    // Es wird nicht gespeichert: Änderungen rückgängig machen!
                    setCsvFormat(_unchangeableOriginalCsvFormat.getCopy(), _editable, _nameChangeable);
                } else {  // es wird gespeichert
                    saveCsvFormat();
                }
            }
            setVisible(false);
            storePreferenceBounds();
            dispose();
        };
        cancelButton.addActionListener(actionListenerCancel);

        ActionListener actionListenerHelp = e -> GndHelp.openHelp("#csvFormatDefinitionDialog");
        helpButton.addActionListener(actionListenerHelp);
    }

    @SuppressWarnings("OverlyLongMethod")
    private void saveCsvFormat() {
        if (!_editable) {    // Sollte nie passieren, da der Button disabled ist.
            JOptionPane.showMessageDialog(this.getFrame(), "Dieses CSV-Format ist nicht veränderbar!", "Fehler", JOptionPane.ERROR_MESSAGE);
            return;
        }
        String formatName = _nameTextField.getText();
        if (formatName.isEmpty()) {
            JOptionPane.showMessageDialog(this.getFrame(), "Bitte geben Sie einen Namen ein!", "Fehler", JOptionPane.ERROR_MESSAGE);
            return;
        } else if (formatName.length() > 40) {    // Präferenzen erlauben 80 Zeichen als Node-Name, doch hier wird 40 gewählt.
            JOptionPane
                .showMessageDialog(this.getFrame(), "Bitte verwenden Sie einen Namen mit höchstens 40 Zeichen!", "Fehler", JOptionPane.ERROR_MESSAGE);
            return;
        }
        String infoText = _infoTextField.getText();
        if (infoText.isEmpty()) {
            infoText = null;
        }

        if (null == _xComboBox.getEditor().getItem() || ((String) _xComboBox.getEditor().getItem()).isEmpty()) {
            if (null == _yComboBox.getEditor().getItem() || ((String) _yComboBox.getEditor().getItem()).isEmpty()) {
                if (null == _lineComboBox.getEditor().getItem() || ((String) _lineComboBox.getEditor().getItem()).isEmpty()) {
                    if (null == _offsetComboBox.getEditor().getItem() || ((String) _offsetComboBox.getEditor().getItem()).isEmpty()) {
                        if (null == _objectComboBox.getEditor().getItem() || ((String) _objectComboBox.getEditor().getItem()).isEmpty()) {
                            JOptionPane.showMessageDialog(this.getFrame(),
                                                          "Bitte geben Sie X- und Y-Spalte an oder Linien- und Offsetspalte oder Objektspalte!",
                                                          "Fehler", JOptionPane.ERROR_MESSAGE);
                            return;
                        }
                    }
                }
            }
        }

        if (null == _xComboBox.getEditor().getItem() || ((String) _xComboBox.getEditor().getItem()).isEmpty()) {
            if (null != _yComboBox.getEditor().getItem() && !((String) _yComboBox.getEditor().getItem()).isEmpty()) {
                JOptionPane.showMessageDialog(this.getFrame(), "Bitte geben Sie X- und Y-Spalte an oder lassen beide frei!", "Fehler",
                                              JOptionPane.ERROR_MESSAGE);
                return;
            }
        }
        if (null == _yComboBox.getEditor().getItem() || ((String) _yComboBox.getEditor().getItem()).isEmpty()) {
            if (null != _xComboBox.getEditor().getItem() && !((String) _xComboBox.getEditor().getItem()).isEmpty()) {
                JOptionPane.showMessageDialog(this.getFrame(), "Bitte geben Sie X- und Y-Spalte an oder lassen beide frei!", "Fehler",
                                              JOptionPane.ERROR_MESSAGE);
                return;
            }
        }

        if (null == _lineComboBox.getEditor().getItem() || ((String) _lineComboBox.getEditor().getItem()).isEmpty()) {
            if (null != _offsetComboBox.getEditor().getItem() && !((String) _offsetComboBox.getEditor().getItem()).isEmpty()) {
                JOptionPane.showMessageDialog(this.getFrame(), "Bitte geben Sie Linien- und Offset-Spalte an oder lassen beide frei!", "Fehler",
                                              JOptionPane.ERROR_MESSAGE);
                return;
            }
        }
        if (null == _offsetComboBox.getEditor().getItem() || ((String) _offsetComboBox.getEditor().getItem()).isEmpty()) {
            if (null != _lineComboBox.getEditor().getItem() && !((String) _lineComboBox.getEditor().getItem()).isEmpty()) {
                JOptionPane.showMessageDialog(this.getFrame(), "Bitte geben Sie Linien- und Offset-Spalte an oder lassen beide frei!", "Fehler",
                                              JOptionPane.ERROR_MESSAGE);
                return;
            }
        }

        if (null == _objectComboBox.getEditor().getItem() || ((String) _objectComboBox.getEditor().getItem()).isEmpty()) {
            if (null != _refHierarchyComboBox.getEditor().getItem() && !((String) _refHierarchyComboBox.getEditor().getItem()).isEmpty()) {
                JOptionPane
                    .showMessageDialog(this.getFrame(), "Eine EOR-Hierachie ist nur in Verbindung mit einer Objektreferenz sinnvoll!", "Fehler",
                                       JOptionPane.ERROR_MESSAGE);
                return;
            }
        }

        final CsvFormat existingFormat = CsvFormatManager.getInstance().getCsvFormat(formatName);
        if (existingFormat != null) {
            if (CsvFormatManager.getInstance().isChangeable(existingFormat)) {
                Object[] options = {"CSV-Format überschreiben", "Speichern abbrechen"};
                int n = JOptionPane
                    .showOptionDialog(this.getFrame(), "Soll das bestehende CSV-Format mit diesem Namen wirklich überschrieben werden?",
                                      "CSV-Format speichern", JOptionPane.YES_NO_CANCEL_OPTION, JOptionPane.QUESTION_MESSAGE, null, options,
                                      options[1]);
                if (n != 0) {
                    return;
                }
            } else {
                JOptionPane.showMessageDialog(this.getFrame(), "Das bestehende CSV-Format darf nicht überschrieben werden!", "Fehler",
                                              JOptionPane.ERROR_MESSAGE);
                return;
            }
        } else if (CsvFormatManager.getInstance().hasCsvFormatToLowerCase(formatName)) {
            JOptionPane.showMessageDialog(this.getFrame(),
                                          "Es existiert bereits ein CSV-Format, dessen Name sich nur bezüglich Klein-Groß-Schreibung unterscheidet!",
                                          "Fehler", JOptionPane.ERROR_MESSAGE);
            return;
        }

        CsvFormat newFormat = new CsvFormat(formatName, infoText);
        newFormat.setCharset(_scratchFormat.getCharset());
        newFormat.setSeparator(_scratchFormat.getSeparator());
        newFormat.setQuote(_scratchFormat.getQuote());
        if (null != _nameComboBox.getEditor().getItem()) {
            newFormat.setNameColumn((String) _nameComboBox.getEditor().getItem());
        } else {
            newFormat.setNameColumn("");
        }
        if (null != _xComboBox.getEditor().getItem()) {
            newFormat.setXColumn((String) _xComboBox.getEditor().getItem());
        } else {
            newFormat.setXColumn("");
        }
        if (null != _yComboBox.getEditor().getItem()) {
            newFormat.setYColumn((String) _yComboBox.getEditor().getItem());
        } else {
            newFormat.setYColumn("");
        }
        if (null != _lineComboBox.getEditor().getItem()) {
            newFormat.setLineColumn((String) _lineComboBox.getEditor().getItem());
        } else {
            newFormat.setLineColumn("");
        }
        if (null != _offsetComboBox.getEditor().getItem()) {
            newFormat.setOffsetColumn((String) _offsetComboBox.getEditor().getItem());
        } else {
            newFormat.setOffsetColumn("");
        }
        if (null != _objectComboBox.getEditor().getItem()) {
            newFormat.setObjectColumn((String) _objectComboBox.getEditor().getItem());
        } else {
            newFormat.setObjectColumn("");
        }
        if (null != _refHierarchyComboBox.getSelectedItem()) {
            newFormat.setReferenceHierarchy((String) _refHierarchyComboBox.getSelectedItem());
        } else {
            newFormat.setReferenceHierarchy("");
        }
        if (null != _priorityComboBox.getSelectedItem()) {
            newFormat.setCsvPriority((CsvPriority) _priorityComboBox.getSelectedItem());
        } else {
            newFormat.setCsvPriority(CsvPriority.getDefaultPrioroty());
        }

        // Nachdem alle Eingaben getestet wurden, werden nun die Csv-Eigenschaften abgefragt.
        EncodingDialog encodingDialog =
            new EncodingDialog(CsvFormatDefinitionDialog.this.getFrame(), newFormat.getCharset(), newFormat.getSeparator(), newFormat.getQuote());
        encodingDialog.setVisible(true);
        if (encodingDialog.isOkay()) {  // impliziert Charset, Separator und Quote sind nicht null
            Charset charset = encodingDialog.getCharset();
            if (charset != null) {
                newFormat.setCharset(charset);
            }
            CsvSeparator separator = encodingDialog.getSeparator();
            if (null != separator) {
                newFormat.setSeparator(separator);
            }
            CsvQuote quote = encodingDialog.getQuote();
            if (null != quote) {
                newFormat.setQuote(quote);
            }
        } else {
            return;
        }

        //noinspection VariableNotUsedInsideIf
        if (existingFormat != null) {
            CsvFormatManager.getInstance().changeCsvFormat(newFormat);
        } else {
            CsvFormatManager.getInstance().addCsvFormat(newFormat);
        }
        _scratchFormat = newFormat;
        _unchangeableOriginalCsvFormat = _scratchFormat.getCopy();
        _somethingChanged = false;
        // Damit der Benutzer dasselbe Objekt nicht unter anderem Namen speichern kann:
        _nameChangeable = false;
        _nameTextField.setEditable(false);
    }

    private void addFrameListener() {
        class FrameListener extends WindowAdapter {
            @Override
            public void windowClosing(WindowEvent e) {
                if (_somethingChanged && _editable) {
                    if (saveChangesWanted()) {
                        saveCsvFormat();
                        setVisible(false);
                        storePreferenceBounds();
                        dispose();
                    }
                }
            }
        }
        addWindowListener(new FrameListener());
    }

    private void initComboBoxes() {
        _nameComboBox.setEditable(true);
        if (!_scratchFormat.getNameColumn().isEmpty()) {
            _nameComboBox.getEditor().setItem(_scratchFormat.getNameColumn());
        }
        _xComboBox.setEditable(true);
        if (!_scratchFormat.getXColumn().isEmpty()) {
            _xComboBox.getEditor().setItem(_scratchFormat.getXColumn());
        }
        _yComboBox.setEditable(true);
        if (!_scratchFormat.getYColumn().isEmpty()) {
            _yComboBox.getEditor().setItem(_scratchFormat.getYColumn());
        }
        _lineComboBox.setEditable(true);
        if (!_scratchFormat.getLineColumn().isEmpty()) {
            _lineComboBox.getEditor().setItem(_scratchFormat.getLineColumn());
        }
        _offsetComboBox.setEditable(true);
        if (!_scratchFormat.getOffsetColumn().isEmpty()) {
            _offsetComboBox.getEditor().setItem(_scratchFormat.getOffsetColumn());
        }
        _objectComboBox.setEditable(true);
        if (!_scratchFormat.getObjectColumn().isEmpty()) {
            _objectComboBox.getEditor().setItem(_scratchFormat.getObjectColumn());
        }
        _refHierarchyComboBox.setModel(new DefaultComboBoxModel<>(ReferenceHierarchyManager.getInstance().getReferenceHierarchyNames(true)));
        if (!_scratchFormat.getReferenceHierarchy().isEmpty()) {
            _refHierarchyComboBox.setSelectedItem(_scratchFormat.getReferenceHierarchy());
        } else {
            _refHierarchyComboBox.setSelectedIndex(-1);
        }
        _priorityComboBox.setSelectedItem(_scratchFormat.getCsvPriority());
    }

    private void initFileChooser() {
        _csvOpenButton.addActionListener(e -> {
            final JFileChooser fileChooser;
            if (System.getProperty("os.name").toLowerCase().startsWith("mac")) {
                fileChooser = AwtFileChooser.createFileChooser();
            } else {
                fileChooser = new JFileChooser();
            }
            fileChooser.setFileFilter(new FileNameExtensionFilter("CSV-Datei", "csv", "CSV"));
            int returnVal = fileChooser.showOpenDialog(CsvFormatDefinitionDialog.this.getFrame());
            if (returnVal == JFileChooser.APPROVE_OPTION) {
                File file = fileChooser.getSelectedFile();
                _csvTextField.setText(file.getAbsolutePath());
                try (InputStream inputStream = new FileInputStream(file)) {
                    CsvReader reader = new CsvReader("UTF-8", inputStream, ';', '"');
                    IterableCsvData data = reader.read();
                    String[] headerCells = data.getHeaderCells();
                    if (null != headerCells && headerCells.length == 1 && headerCells[0].contains(",")) {
                        headerCells = headerCells[0].split(",");
                    }
                    if (null != headerCells) {
                        String[] headerCellsPlus = new String[headerCells.length + 1];
                        headerCellsPlus[0] = "";
                        System.arraycopy(headerCells, 0, headerCellsPlus, 1, headerCells.length);
                        _nameComboBox.setModel(new DefaultComboBoxModel<>(headerCellsPlus));
                        selectSomething(_nameComboBox, "name");
                        _xComboBox.setModel(new DefaultComboBoxModel<>(headerCellsPlus));
                        selectSomething(_xComboBox, "x");
                        _yComboBox.setModel(new DefaultComboBoxModel<>(headerCellsPlus));
                        selectSomething(_yComboBox, "y");
                        _lineComboBox.setModel(new DefaultComboBoxModel<>(headerCellsPlus));
                        selectSomething(_lineComboBox, "linie");
                        _offsetComboBox.setModel(new DefaultComboBoxModel<>(headerCellsPlus));
                        selectSomething(_offsetComboBox, "offset");
                        _objectComboBox.setModel(new DefaultComboBoxModel<>(headerCellsPlus));
                        selectSomething(_objectComboBox, "objekt");
                    }
                } catch (FileNotFoundException ignore) {
                    JOptionPane.showMessageDialog(CsvFormatDefinitionDialog.this.getFrame(),
                                                  "Die Datei '" + file.getAbsolutePath() + "' konnte nicht geöffnet werden.", "Fehler",
                                                  JOptionPane.ERROR_MESSAGE);
                } catch (IOException e1) {
                    JOptionPane.showMessageDialog(CsvFormatDefinitionDialog.this.getFrame(),
                                                  "IOException für die Datei '" + file.getAbsolutePath() + "':" + System.lineSeparator() +
                                                  e1.toString(), "Fehler", JOptionPane.ERROR_MESSAGE);
                }
            }
        });
    }

    private boolean saveChangesWanted() {
        Object[] options = {"Änderungen speichern", "Nicht speichern"};
        int n = JOptionPane.showOptionDialog(CsvFormatDefinitionDialog.this.getFrame(), "Änderungen speichern?",
                                             "Es wurden Änderungen an dem CSV-Format vorgenommen.", JOptionPane.YES_NO_CANCEL_OPTION,
                                             JOptionPane.QUESTION_MESSAGE, null, options, options[1]);
        return n == 0;
    }

    @Override
    public String toString() {
        return "CsvFormatDefinitionDialog{}";
    }

    private interface CsvFormatsEditingStateListener {
        @SuppressWarnings("unused")
        void formatsEditingStateChanged(boolean editable);
    }

    private class EditableListenButton extends JButton implements CsvFormatsEditingStateListener {

        EditableListenButton(String text) {
            super(text);
            _listeningButtons.add(this);

        }

        @Override
        public void formatsEditingStateChanged(boolean editable) {
            setEnabled(editable);
        }
    }

}
