/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.csv.view;

import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.kappich.pat.gnd.csv.CsvQuote;
import de.kappich.pat.gnd.csv.CsvSeparator;
import de.kappich.pat.gnd.utils.SpringUtilities;
import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.Frame;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import javax.swing.BorderFactory;
import javax.swing.ComboBoxModel;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.SpringLayout;
import javax.swing.event.ListDataListener;

/**
 * @author Kappich Systemberatung
 */
public class EncodingDialog {

    private final JDialog _dialog;

    private final JComboBox<Charset> _charSetComboBox = new JComboBox<>(new CharSetModel());
    private final JComboBox<CsvSeparator> _separatorComboBox = new JComboBox<>(CsvSeparator.values());
    private final JComboBox<CsvQuote> _quoteComboBox = new JComboBox<>(CsvQuote.values());
    private final JButton _cancelButton = new JButton("Abbrechen");
    private final JButton _okayButton = new JButton("Okay");
    private boolean _okay;

    /**
     * Der Konstruktor mit Identifier und den zwei Argumenten eines korrespondierenden {@code JDialog}-Konstruktors.
     *
     * @param owner der Owner-Frame
     */
    public EncodingDialog(final Frame owner, @Nullable final Charset charset, @Nullable final CsvSeparator sep, @Nullable final CsvQuote quote) {

        if (null != charset) {
            _charSetComboBox.setSelectedItem(charset);
        }
        if (null != sep) {
            _separatorComboBox.setSelectedItem(sep);
        }
        if (null != quote) {
            _quoteComboBox.setSelectedItem(quote);
        }
        _dialog = new JDialog(owner, true);
        _dialog.setTitle("CSV-Einstellungen");
        initLayout();
        initLogic();
        _dialog.pack();
        _dialog.setLocationRelativeTo(owner);
    }

    private void initLayout() {
        _dialog.setLayout(new BorderLayout());

        { // Oberer Teil:
            JLabel charSetLabel = new JLabel("Zeichensatz: ");
            Dimension labelSize = new Dimension(110, 20);
            charSetLabel.setPreferredSize(labelSize);

            JLabel separatorLabel = new JLabel("Feldtrenner: ");
            separatorLabel.setPreferredSize(labelSize);

            JLabel quoteLabel = new JLabel("Texttrenner: ");
            quoteLabel.setPreferredSize(labelSize);

            JPanel upperPanel = new JPanel();
            upperPanel.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
            upperPanel.setLayout(new SpringLayout());
            upperPanel.add(charSetLabel);
            upperPanel.add(_charSetComboBox);
            upperPanel.add(separatorLabel);
            upperPanel.add(_separatorComboBox);
            upperPanel.add(quoteLabel);
            upperPanel.add(_quoteComboBox);
            SpringUtilities.makeCompactGrid(upperPanel, 2, 5, 5);
            _dialog.add(upperPanel, BorderLayout.NORTH);
        }

        { // Unterer Teil:
            JPanel buttonsPanel = new JPanel();
            buttonsPanel.setLayout(new SpringLayout());

            buttonsPanel.add(_okayButton);
            buttonsPanel.add(_cancelButton);

            buttonsPanel.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
            SpringUtilities.makeCompactGrid(buttonsPanel, 2, 20, 5);
            _dialog.add(buttonsPanel, BorderLayout.SOUTH);
        }
    }

    private void initLogic() {
        initOkayButton();
        initCancelButton();
    }

    private void initOkayButton() {
        _okayButton.addActionListener(e -> {
            if (null == _charSetComboBox.getSelectedItem()) {
                JOptionPane
                    .showMessageDialog(EncodingDialog.this._dialog, "Bitte wählen Sie einen Zeichensatz aus!", "Fehler", JOptionPane.ERROR_MESSAGE);
                return;
            }
            if (null == _separatorComboBox.getSelectedItem()) {
                JOptionPane
                    .showMessageDialog(EncodingDialog.this._dialog, "Bitte wählen Sie einen Feldtrenner aus!", "Fehler", JOptionPane.ERROR_MESSAGE);
                return;
            }
            if (null == _quoteComboBox.getSelectedItem()) {
                JOptionPane
                    .showMessageDialog(EncodingDialog.this._dialog, "Bitte wählen Sie einen Texttrenner aus!", "Fehler", JOptionPane.ERROR_MESSAGE);
                return;
            }
            _okay = true;
            dispose();
        });

    }

    private void initCancelButton() {
        _cancelButton.addActionListener(e -> {
            _okay = false;
            dispose();
        });
    }

    public boolean isOkay() {
        return _okay;
    }

    public void setVisible(boolean b) {
        if (b) {
            _okay = false;
        }
        _dialog.setVisible(b);
    }

    public void dispose() {
        _dialog.dispose();
    }

    @Nullable
    public Charset getCharset() {
        Object selectedItem = _charSetComboBox.getModel().getSelectedItem();
        if (null != selectedItem) {
            return (Charset) selectedItem;
        } else {
            return null;
        }
    }

    @Nullable
    public CsvSeparator getSeparator() {
        Object selectedItem = _separatorComboBox.getSelectedItem();
        if (null != selectedItem) {
            return (CsvSeparator) selectedItem;
        } else {
            return null;
        }
    }

    @Nullable
    public CsvQuote getQuote() {
        Object selectedItem = _quoteComboBox.getSelectedItem();
        if (null != selectedItem) {
            return (CsvQuote) selectedItem;
        } else {
            return null;
        }
    }

    @Override
    public String toString() {
        return "EncodingDialog{}";
    }

    private static class CharSetModel implements ComboBoxModel<Charset> {

        private final List<Charset> _charsets = new ArrayList<>();
        private Charset _selectedItem = StandardCharsets.ISO_8859_1;

        CharSetModel() {
            SortedMap<String, Charset> availableCharsets = Charset.availableCharsets();
            for (final Map.Entry<String, Charset> stringCharsetEntry : availableCharsets.entrySet()) {
                _charsets.add(stringCharsetEntry.getValue());
            }
            _charsets.sort(Comparator.comparing(Charset::displayName));
        }

        @Override
        public Object getSelectedItem() {
            return _selectedItem;
        }

        @Override
        public void setSelectedItem(final Object anItem) {
            _selectedItem = (Charset) anItem;
        }

        @Override
        public int getSize() {
            return _charsets.size();
        }

        @Override
        public Charset getElementAt(final int index) {
            return _charsets.get(index);
        }

        @Override
        public void addListDataListener(final ListDataListener l) {
        }

        @Override
        public void removeListDataListener(final ListDataListener l) {
        }
    }
}
