/*
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */
package de.kappich.pat.gnd.displayObjectToolkit;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.kappich.pat.gnd.documentation.GndHelp;
import de.kappich.pat.gnd.gnd.PluginManager;
import de.kappich.pat.gnd.layerManagement.LayerManager;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectType;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectTypePlugin;
import de.kappich.pat.gnd.utils.SpringUtilities;
import de.kappich.pat.gnd.utils.view.GndFrame;
import de.kappich.pat.gnd.utils.view.GndTable;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Frame;
import java.awt.Rectangle;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Vector;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.SpringLayout;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.DefaultTableCellRenderer;


/**
 * Der Verwaltungsdialog für alle Darstellungstypen.
 *
 * @author Kappich Systemberatung
 * @version $Revision$
 */
@SuppressWarnings({"OverlyLongMethod", "UseOfObsoleteCollectionType"})
public class DOTManagerDialog {

    private final DOTManager _dotManager;
    private final ClientDavInterface _connection;
    private final GndFrame _frame;

    // Im Moment landen alle Frames, die zum Betrachten/Bearbeiten bestehender DOTs geöffnet
    // werden, in der folgenden Liste. Falls zu einem solchen DOT wieder ein Frame geöffnet werden
    // soll, so bekommt der bereits geöffnete den Fokus.
    private final Map<String, DOTDefinitionDialog> _dotDefinitionDialogFrames =
        new HashMap<>();

    /**
     * Der DOTManagerDialog dient zur grafischen Verwaltung und Bearbeitung aller vom
     * DOTManager verwalteten Darstellungstypen.
     *
     * @param dotManager die Darstellungstypenverwaltung
     * @param connection die Datenverteiler-Verbindung
     */
    public DOTManagerDialog(DOTManager dotManager, ClientDavInterface connection) {

        _dotManager = dotManager;
        _connection = connection;
        _frame = new GndFrame("DOTManagerDialog", "GND: Darstellungstypenverwaltung");
        _frame.setLayout(new BorderLayout());

        JButton newDOTButton = new JButton("Neuer Darstellungstyp");

        final JButton editButton = new JButton("Bearbeiten");
        final JButton copyButton = new JButton("Kopieren");
        final JButton deleteButton = new JButton("Löschen");
        final JButton helpButton = new JButton("Hilfe");

        JPanel buttonPanelEast = new JPanel();
        buttonPanelEast.setLayout(new BoxLayout(buttonPanelEast, BoxLayout.Y_AXIS));

        Dimension d = new Dimension(15, 15);
        buttonPanelEast.add(Box.createRigidArea(d));

        buttonPanelEast.add(newDOTButton);
        buttonPanelEast.add(Box.createRigidArea(d));

        buttonPanelEast.add(editButton);
        buttonPanelEast.add(Box.createRigidArea(d));

        buttonPanelEast.add(copyButton);
        buttonPanelEast.add(Box.createRigidArea(d));

        buttonPanelEast.add(deleteButton);
        buttonPanelEast.add(Box.createRigidArea(d));

        buttonPanelEast.add(helpButton);
        buttonPanelEast.add(Box.createRigidArea(d));

        buttonPanelEast.setBorder(BorderFactory.createEmptyBorder(5, 10, 5, 10));

        editButton.setEnabled(false);
        copyButton.setEnabled(false);
        deleteButton.setEnabled(false);

        JPanel buttonsPanelSouth = new JPanel();
        buttonsPanelSouth.setLayout(new SpringLayout());
        buttonsPanelSouth.setBorder(BorderFactory.createEmptyBorder(5, 10, 5, 10));

        JButton schliessenButton = new JButton("Dialog schließen");
        buttonsPanelSouth.add(schliessenButton);
        SpringUtilities.makeCompactGrid(buttonsPanelSouth, 1, 20, 5);

        final GndTable table = new GndTable(_dotManager);
        table.setVisibleRowCount(40);
        table.setFillsViewportHeight(true);
        table.setDefaultRenderer(Object.class, new DotTableRenderer());
        // vordefinierte Darstellungsobjekttypen dürfen nicht bearbeitet oder gelöscht werden
        final ListSelectionModel selectionModel = table.getSelectionModel();

        ListSelectionListener listSelectionListener = e -> {
            final int selectedRow = table.getSelectedRow();
            if(selectedRow == -1) {
                editButton.setEnabled(false);
                editButton.setToolTipText("Es ist kein Darstellungstyp ausgewählt worden");
                copyButton.setEnabled(false);
                copyButton.setToolTipText("Es ist kein Darstellungstyp ausgewählt worden");
                deleteButton.setEnabled(false);
                deleteButton.setToolTipText("Es ist kein Darstellungstyp ausgewählt worden");
            }
            else {
                final boolean changeable = _dotManager.isChangeable(_dotManager.getDisplayObjectType(selectedRow));
                if(!changeable) {
                    editButton.setText("Betrachten");
                    editButton.setEnabled(true);
                    editButton.setToolTipText("Details des ausgewählten Darstellungstypen betrachten");
                    copyButton.setEnabled(true);
                    copyButton.setToolTipText("Kopie des ausgewählten Darstellungstypen erstellen und bearbeiten");
                    deleteButton.setEnabled(false);
                    deleteButton.setToolTipText("Der ausgewählte Darstellungstyp kann nicht gelöscht werden");
                }
                else {
                    editButton.setText("Bearbeiten");
                    editButton.setEnabled(true);
                    editButton.setToolTipText("Den ausgewählten Darstellungstypen bearbeiten");
                    copyButton.setEnabled(true);
                    copyButton.setToolTipText("Kopie des ausgewählten Darstellungstypen erstellen und bearbeiten");
                    deleteButton.setEnabled(true);
                    deleteButton.setToolTipText("Den ausgewählten Darstellungstypen löschen");
                }
                deleteButton.setEnabled(changeable);
            }
        };

        final KeyAdapter keyAdapter = new KeyAdapter() {
            @Override
            public void keyPressed(KeyEvent e) {
                if(e.getKeyCode() == KeyEvent.VK_DELETE) {
                    int selectedRow = table.getSelectedRow();
                    if(selectedRow >= 0) {
                        final DisplayObjectType type = _dotManager.getDisplayObjectType(table.getSelectedRow());
                        if(_dotManager.isChangeable(type)) {
                            Object[] options = {"Ja", "Nein"};
                            int n = JOptionPane.showOptionDialog(
                                _frame.getFrame(),
                                "Wollen Sie den Darstellungstyp " + type.getName() + " wirklich löschen?",
                                "Darstellungstyp " + type.getName() + "löschen",
                                JOptionPane.YES_NO_OPTION,
                                JOptionPane.QUESTION_MESSAGE,
                                null,
                                options,
                                options[0]
                            );
                            if(n == 0) {
                                _dotManager.deleteDisplayObjectType(type);
                            }
                        }
                    }
                }
                else {
                    char c = e.getKeyChar();
                    if(Character.isLetter(c)) {
                        int index = _dotManager.getIndexOfFirstDot(c);
                        if(index >= 0) {
                            table.getSelectionModel().setSelectionInterval(index, index);
                            table.scrollRectToVisible(new Rectangle(table.getCellRect(index, 0, true)));
                        }
                    }
                }
            }
        };
        table.addKeyListener(keyAdapter);
        newDOTButton.addKeyListener(keyAdapter);
        editButton.addKeyListener(keyAdapter);
        copyButton.addKeyListener(keyAdapter);
        deleteButton.addKeyListener(keyAdapter);
        helpButton.addKeyListener(keyAdapter);


        table.addMouseListener(new MouseAdapter() {
            @Override
            public void mousePressed(final MouseEvent e) {
                if(e.getClickCount() > 1) {
                    editOrInspectAction(table);
                }
            }
        });

        selectionModel.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        selectionModel.addListSelectionListener(listSelectionListener);

        _frame.add(buttonPanelEast, BorderLayout.EAST);
        _frame.add(buttonsPanelSouth, BorderLayout.SOUTH);
        _frame.add(new JScrollPane(table), BorderLayout.CENTER);

        // Neu, Bearbeiten, Kopieren, Löschen
        ActionListener actionListenerNew = e -> {
            final JLabel label = new JLabel("Bitte wählen Sie eine Art aus:");
            final JPanel labelPanel = new JPanel();
            labelPanel.setLayout(new SpringLayout());
            labelPanel.add(label);
            SpringUtilities.makeCompactGrid(labelPanel, 1, 20, 5);
            final Vector<String> items = PluginManager.getAllPluginNames(false, false, false);
            final JComboBox<String> pluginComboBox = new JComboBox<>(items);
            final JButton chooseButton = new JButton("Auswählen");
            final JButton localHelpButton = new JButton("Hilfe");
            final JPanel choicePanel = new JPanel();
            choicePanel.setLayout(new SpringLayout());
            choicePanel.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));
            choicePanel.add(pluginComboBox);
            choicePanel.add(chooseButton);
            choicePanel.add(localHelpButton);
            SpringUtilities.makeCompactGrid(choicePanel, 3, 20, 5);
            final JPanel allPanel = new JPanel();
            allPanel.setLayout(new BoxLayout(allPanel, BoxLayout.Y_AXIS));
            allPanel.add(labelPanel);
            allPanel.add(choicePanel);
            allPanel.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));
            final JDialog newDialog = new JDialog(_frame.getFrame(), true);
            newDialog.setTitle("Art des Darstellungstyps auswählen");
            newDialog.add(allPanel);
            newDialog.pack();
            newDialog.setLocationRelativeTo(_frame.getFrame());

            ActionListener okayListener = f -> {
                final Object selectedItem = pluginComboBox.getSelectedItem();
                if(selectedItem == null) {
                    JOptionPane.showMessageDialog(
                        new JFrame(),
                        "Bitte wählen Sie eine Zeile aus der Liste aus!",
                        "Fehler",
                        JOptionPane.ERROR_MESSAGE
                    );
                    return;
                }
                final String s = (String) selectedItem;
                final DisplayObjectTypePlugin plugin = PluginManager.getPlugin(s);
                if(plugin == null) {
                    JOptionPane.showMessageDialog(
                        new JFrame(),
                        "Es wurde kein Plugin zu '" + s + "' gefunden!",
                        "Fehler",
                        JOptionPane.ERROR_MESSAGE
                    );
                    return;
                }
                final DisplayObjectType type = plugin.getDisplayObjectType();
                if(type == null) {
                    JOptionPane.showMessageDialog(
                        new JFrame(),
                        "Zu '" + s + "' kann kein Darstellungstyp angelegt werden!",
                        "Fehler",
                        JOptionPane.ERROR_MESSAGE
                    );
                    return;
                }
                newDialog.dispose();
                final String title = "GND: neuen Darstellungstyp bearbeiten";
                DOTDefinitionDialog dotDDFrame = new DOTDefinitionDialog(
                    DOTManagerDialog.this, _connection, _dotManager, type, true, false, title);
                dotDDFrame.setVisible(true);
                dotDDFrame.toFront();
            };
            chooseButton.addActionListener(okayListener);

            newDialog.setVisible(true);
        };
        newDOTButton.addActionListener(actionListenerNew);


        ActionListener actionListenerLoeschen = e -> {
            if(table.getSelectedRowCount() == 0) {
                JOptionPane.showMessageDialog(
                    new JFrame(),
                    "Bitte wählen Sie mindestens eine Zeile aus der Liste aus!",
                    "Fehler",
                    JOptionPane.ERROR_MESSAGE
                );
                return;
            }
            final int[] selectedRows = table.getSelectedRows();
            boolean allDOTsRemoved = true;
            DisplayObjectType notRemovableDOT = null;
            for(int i = selectedRows.length - 1; i >= 0; i--) {
                final DisplayObjectType type = _dotManager.getDisplayObjectType(selectedRows[i]);
                if(!_dotManager.deleteDisplayObjectType(type)) {
                    allDOTsRemoved = false;
                    notRemovableDOT = _dotManager.getDisplayObjectType(selectedRows[i]);
                }
            }
            if(!allDOTsRemoved) {
                List<String> layerNames = LayerManager.getInstance().getLayersUsingTheDisplayObjectType(
                    notRemovableDOT.getName());
                StringBuilder sb = new StringBuilder();
                sb.append("Der Darstellungstyp '").append(notRemovableDOT.getName())
                    .append("' kann nicht gelöscht werden, weil er in folgenden Layern verwendet wird: ");
                boolean commaNeeded = false;
                for(String layerName : layerNames) {
                    if(commaNeeded) {
                        sb.append(", ");
                    }
                    else {
                        commaNeeded = true;
                    }
                    sb.append(layerName);
                }
                JOptionPane.showMessageDialog(
                    new JFrame(),
                    sb.toString(),
                    "Fehler",
                    JOptionPane.ERROR_MESSAGE
                );
            }
        };
        deleteButton.addActionListener(actionListenerLoeschen);

        ActionListener actionListenerKopieren = e -> {
            if(table.getSelectedRowCount() != 1) {
                JOptionPane.showMessageDialog(
                    new JFrame(),
                    "Bitte wählen Sie genau eine Zeile aus der Liste aus!",
                    "Fehler",
                    JOptionPane.ERROR_MESSAGE
                );
                return;
            }
            final int selectedRow = table.getSelectedRow();
            final DisplayObjectType type = _dotManager.getDisplayObjectType(selectedRow);
            final DisplayObjectType newType = type.getCopy(type.getName() + " (Kopie)");
            final String title = "GND: kopierten Darstellungstyp bearbeiten";
            DOTDefinitionDialog dotDDFrame = new DOTDefinitionDialog(
                DOTManagerDialog.this, _connection, _dotManager, newType, true, false, title);
            dotDDFrame.setVisible(true);
            dotDDFrame.toFront();
        };
        copyButton.addActionListener(actionListenerKopieren);

        ActionListener actionListenerBearbeiten = e -> editOrInspectAction(table);
        editButton.addActionListener(actionListenerBearbeiten);

        ActionListener actionListenerHelp = e -> GndHelp.openHelp("#theDisplayObjectTypeManagerDialog");
        helpButton.addActionListener(actionListenerHelp);

        ActionListener actionListenerDialogSchliessen = e -> closeDialog();
        schliessenButton.addActionListener(actionListenerDialogSchliessen);

        class FrameListener extends WindowAdapter {
            @Override
            public void windowClosing(WindowEvent e) {
                /*
                 * wenn nur noch ein einziges Fenster geöffnet ist
                 * beendet sich das Programm beim Schließen des Fensters
                 */
                final Frame[] frames = Frame.getFrames();
                int length = frames.length - 1;

                for(Frame frame : frames) {
                    if(!frame.isVisible()) {
                        length--;
                    }
                }

                if(length == 0) {
                    _frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
                }
            }
        }
        _frame.addWindowListener(new FrameListener());

        table.setVisibleRowCount(Math.min(40, table.getModel().getRowCount()));
        _frame.setPositionAndSize(0, 0, 860, 50, true, 0, 0);
        _frame.setVisible(true);
    }

    private void editOrInspectAction(final JTable table) {
        if(table.getSelectedRowCount() != 1) {
            JOptionPane.showMessageDialog(
                new JFrame(),
                "Bitte wählen Sie genau eine Zeile aus der Liste aus!",
                "Fehler",
                JOptionPane.ERROR_MESSAGE
            );
            return;
        }
        final int selectedRow = table.getSelectedRow();
        final DisplayObjectType type = _dotManager.getDisplayObjectType(selectedRow);
        final String typeName = type.getName();
        final boolean changeable = _dotManager.isChangeable(type);
        final String title;
        if(changeable) {
            title = "GND: Darstellungstyp bearbeiten";
        }
        else {
            title = "GND: Darstellungstyp betrachten";
        }
        if(!_dotDefinitionDialogFrames.containsKey(typeName)) {
            DOTDefinitionDialog dotDDFrame = new DOTDefinitionDialog(
                DOTManagerDialog.this, _connection, _dotManager,
                type, changeable, true, title
            );
            dotDDFrame.setVisible(true);
            dotDDFrame.toFront();
            _dotDefinitionDialogFrames.put(typeName, dotDDFrame);
        }
        else {
            DOTDefinitionDialog dotDDFrame = _dotDefinitionDialogFrames.get(typeName);
            dotDDFrame.setTitle(title);
            dotDDFrame.setVisible(true);
            dotDDFrame.toFront();
            dotDDFrame.requestFocus();
        }
    }

    /*
     * CellRenderer für die JTable im Verwaltungsdialog
     *
     */
    private class DotTableRenderer extends DefaultTableCellRenderer {

        private static final long serialVersionUID = 1L;

        @Override
        public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus,
                                                       int row, int column) {
            super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);

            if(null != value) { // null passiert auf Macs
                setText(value.toString());
                if(_dotManager.containsDisplayObjectType(value.toString())) {
                    DisplayObjectType dot = _dotManager.getDisplayObjectType(value.toString());
                    setToolTipText(dot.getInfo() + " (Plugin: " + dot.getDisplayObjectTypePlugin().getName() + ")");
                    if(_dotManager.isChangeable(dot)) {
                        setForeground(Color.BLACK);
                        Font font = getFont();
                        setFont(new Font(font.getName(), Font.PLAIN, font.getSize()));
                    }
                    else {
                        setForeground(Color.GRAY);
                        Font font = getFont();
                        setFont(new Font(font.getName(), Font.ITALIC, font.getSize()));
                    }
                }
            }
            return this;
        }
    }

    /**
     * Gibt die Fensterkompenente zurück.
     *
     * @return die Fensterkompenente
     */
    public Component getComponent() {
        return _frame.getFrame();
    }
    /**
     * Macht den DOTManagerDialog sichtbar.
     */
    public void showDialog(boolean visible) {
        _frame.setState(Frame.NORMAL);
        _frame.setVisible(visible);
    }

    /**
     * Schließt den DOTManagerDialog.
     */
    public void closeDialog() {
        _frame.setVisible(false);
        _frame.dispose();
    }

    /**
     * Der übergebene DOTDefinitionPanel wurde geschlossen und muss aus der Verwaltung gelöscht werden.
     *
     * @param dotDDFrame der geschlossene DOTDefinitionPanel
     */
    public void dialogDisposed(DOTDefinitionDialog dotDDFrame) {
        final String dotName = dotDDFrame.getDisplayObjectType().getName();
        _dotDefinitionDialogFrames.remove(dotName);
    }

    /**
     * Der Darstellungstyp wurde verändert.
     *
     * @param formerDisplayObjectType der alte Darstellungstyp
     * @param newDisplayObjectType    der neue Darstellungstyp
     */
    public void dialogChanged(DisplayObjectType formerDisplayObjectType,
                              DisplayObjectType newDisplayObjectType) {
        if(formerDisplayObjectType == null) {
            return;
        }
        final String dotName = formerDisplayObjectType.getName();
        if(_dotDefinitionDialogFrames.containsKey(dotName)) {
            DOTDefinitionDialog dotDDFrame = _dotDefinitionDialogFrames.remove(dotName);
            if(newDisplayObjectType != null) {
                _dotDefinitionDialogFrames.put(newDisplayObjectType.getName(), dotDDFrame);
            }
        }
    }

    @Override
    public String toString() {
        return "DOTManagerDialog";
    }
}
