/*
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.displayObjectToolkit;

import de.bsvrz.dav.daf.main.ClientReceiverInterface;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.DataState;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.kappich.pat.gnd.gnd.MapPane;
import de.kappich.pat.gnd.needlePlugin.DOTNeedlePainter;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectPainter;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectType;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectType.DisplayObjectTypeItem;
import de.kappich.pat.gnd.utils.PointWithAngle;
import java.awt.Polygon;
import java.awt.Rectangle;
import java.awt.geom.Line2D;
import java.awt.geom.Path2D;
import java.awt.geom.PathIterator;
import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.TreeMap;

/**
 * Ein OnlineDisplayObject ist ein georeferenziertes SystemObject mit allen Informationen zu seiner Darstellung.
 *
 * @author Kappich Systemberatung
 */
public class OnlineDisplayObject implements DisplayObject, ClientReceiverInterface {

    private static final Debug _debug = Debug.getLogger();
    private final SystemObject _systemObject;
    private final Map<Integer, List<Object>> _coordinates = new TreeMap<>();
    private final Map<PrimitiveFormPropertyPair, Data> _values = new HashMap<>();
    private final DOTCollection _dotCollection;
    private final Map<Integer, Rectangle> _boundingRectangles = new HashMap<>();
    private final MapPane _mapPane;
    private final Map<DisplayObjectType, Map<PrimitiveFormPropertyPair, DisplayObjectTypeItem>> _resultCache;
    private DisplayObjectPainter _painter;
    private DisplayObjectType _currentDisplayObjectType;
    private Map<PrimitiveFormPropertyPair, DisplayObjectTypeItem> _displayObjectTypeItems;
    private int _defaultType;

    /**
     * Konstruiert ein OnlineDisplayObject. Hierzu müssen das zugehörige Systemobjekt, die Koordinaten, ein Painter, die {@link DOTCollection}, eine
     * speziell-aufbereitete Map mit mit Informationen welche {@link PrimitiveFormPropertyPair}-Objekte zu welchen Darstellungstypen gehören, und
     * schließlich die Kartenansicht, in der das Objekt gezeichnet werden soll, angegeben werden. Die Konstruktion dieser Objekte ist eine der
     * Aufgaben der Klasse {@link DisplayObjectManager}.
     *
     * @param systemObject               ein Systemobjekt
     * @param coordinates                die Koordinaten zum Default-Type
     * @param painter                    der Painter
     * @param dotCollection              die DOTCollcetion
     * @param primitiveFormPropertyPairs die Paare zu den Darstellungstypen
     * @param mapPane                    die Kartenansicht
     */
    public OnlineDisplayObject(SystemObject systemObject, List<Object> coordinates, DisplayObjectPainter painter, DOTCollection dotCollection,
                               Map<DisplayObjectType, List<PrimitiveFormPropertyPair>> primitiveFormPropertyPairs, MapPane mapPane) {
        super();
        _systemObject = systemObject;
        _coordinates.put(0, coordinates);
        _painter = painter;
        _dotCollection = dotCollection;

        _currentDisplayObjectType = null;

        _defaultType = 0;
        _mapPane = mapPane;
        _resultCache = new HashMap<>();
        for (final Map.Entry<DisplayObjectType, List<PrimitiveFormPropertyPair>> displayObjectTypeListEntry : primitiveFormPropertyPairs.entrySet()) {
            for (PrimitiveFormPropertyPair pfPropertyPair : displayObjectTypeListEntry.getValue()) {
                Map<PrimitiveFormPropertyPair, DisplayObjectTypeItem> currentDOTMap =
                    _resultCache.computeIfAbsent(displayObjectTypeListEntry.getKey(), k -> new HashMap<>());
                currentDOTMap.put(pfPropertyPair, null);
            }
        }
    }

    private static Data getSubItem(final Data data, final String attributeName) {
        Data dataItem = data;
        String s = attributeName;
        int index = s.indexOf('.');
        while (index > 0) {
            String itemName = s.substring(0, index);
            dataItem = dataItem.getItem(itemName);
            s = s.substring(index + 1);
            index = s.indexOf('.');
        }
        return dataItem.getItem(s);
    }

    @Nullable
    private static Point2D getReferencePoint(final Rectangle utmBounds, final List<Object> coordinates) {
        if (coordinates.isEmpty()) {
            return null;
        }
        for (Object o : coordinates) {
	        if (o instanceof PointWithAngle pointWithAngle) {
                if (utmBounds.contains(pointWithAngle.getPoint())) {
                    return pointWithAngle.getPoint();
                }
	        } else if (o instanceof Point2D p) {
                if (utmBounds.contains(p)) {
                    return p;
                }
	        } else if (o instanceof Path2D.Double polyline) {
                Point2D p = getReferencePoint(utmBounds, polyline);
                if (null != p) {
                    return p;
                }
	        } else if (o instanceof Polygon polygon) {
                if (polygon.intersects(utmBounds)) {
                    Rectangle rectangle = polygon.getBounds();
                    return new Point2D.Double(rectangle.getCenterX(), rectangle.getCenterY());
                }
            }
        }
        return null;
    }

    // Diese Methode ist für lange Linienobjekte noch nicht zufriedenstellend, weil die Namen
    // oft an ganz anderer Stelle angezeigt werden, als optimal wäre. Außerdem gibt es Grenzfälle,
    // in denen das arithmetische Mittel nicht mehr im übergebenen Rechteck liegt. Strenggenommen
    // ist das Ergebnis dann falsch.
    @Nullable
    private static Point2D getReferencePoint(final Rectangle utmBounds, final Path2D.Double polyline) {
        if (!polyline.intersects(utmBounds)) {
            // 'intersects' compares the interior of polyline and utmBounds; if polyline is a point, then it return false.
            final PathIterator pathIterator = polyline.getPathIterator(null);
            if (!pathIterator.isDone()) {
                double[] coordinates = new double[6];
                int type = pathIterator.currentSegment(coordinates);
                if (type == PathIterator.SEG_MOVETO) {
                    Point2D point = new Point2D.Double(coordinates[0], coordinates[1]);
                    if (utmBounds.contains(point)) {
                        return point;
                    }
                }
            }
            return null;
        }
        final PathIterator pathIterator = polyline.getPathIterator(null);
        double currentX = Double.MAX_VALUE;
        double currentY = Double.MAX_VALUE;
        while (!pathIterator.isDone()) {
            double[] coordinates = new double[6];
            int type = pathIterator.currentSegment(coordinates);
            switch (type) {
                case PathIterator.SEG_MOVETO:
                    currentX = coordinates[0];
                    currentY = coordinates[1];
                    break;
                case PathIterator.SEG_LINETO:
                    double newX = coordinates[0];
                    double newY = coordinates[1];
                    Line2D.Double line = new Line2D.Double(currentX, currentY, newX, newY);
                    if (line.intersects(utmBounds)) {
                        return new Point2D.Double((currentX + newX) / 2., (currentY + newY) / 2.);
                    }
                    currentX = coordinates[0];
                    currentY = coordinates[1];
                    break;
            }
            pathIterator.next();
        }
        return null;
    }

    private static DisplayObjectTypeItem getDOTItemForState(DisplayObjectType displayObjectType, ResultData result,
                                                            DOTSubscriptionData subscriptionData, PrimitiveFormPropertyPair pfPropertyPair) {
        final DataState dataState = result.getDataState();
        return displayObjectType
            .getDisplayObjectTypeItemForState(pfPropertyPair.getPrimitiveFormName(), pfPropertyPair.getProperty(), subscriptionData, dataState);
    }

    /**
     * Gibt das zugrundeliegende {@link SystemObject} zurück.
     *
     * @return das Systemobjekt
     */
    public SystemObject getSystemObject() {
        return _systemObject;
    }

    /**
     * Gibt das aktuell gültige {@link DisplayObjectTypeItem} für das {@link PrimitiveFormPropertyPair pair} zurück.
     *
     * @param pair das Paar
     *
     * @return das aktuell gültige DisplayObjectTypeItem oder {@code null}, wenn kein solches existiert
     */
    @Override
    @Nullable
    public DisplayObjectTypeItem getDisplayObjectTypeItem(PrimitiveFormPropertyPair pair) {
        if (_displayObjectTypeItems != null) {
            return _displayObjectTypeItems.get(pair);
        }
        return null;
    }

    /**
     * Gibt den aktuell gültigen Wert für das {@link PrimitiveFormPropertyPair pair} zurück.
     *
     * @param pair das Paar
     *
     * @return der aktuell gültige Wert oder {@code null}, wenn kein solcher existiert
     */
    @SuppressWarnings("VariableNotUsedInsideIf")
    @Nullable
    public Data getValue(@Nullable PrimitiveFormPropertyPair pair) {
        if (_displayObjectTypeItems != null) {
            return _values.get(pair);
        }
        return null;
    }

    @Override
    public String getName() {
        if (_systemObject.getName() != null && !_systemObject.getName().isEmpty()) {
            if (_systemObject.getPid() != null && !_systemObject.getPid().isEmpty()) {
                return _systemObject.getName() + " (" + _systemObject.getPid() + ")";
            } else {
                return _systemObject.getName();
            }
        } else if (_systemObject.getPid() != null && !_systemObject.getPid().isEmpty()) {
            return _systemObject.getPid();
        } else {
            return Long.toString(_systemObject.getId());
        }
    }

    /**
     * Gibt die Koordinaten zu dem dem übergebenen (Koordinaten-)Typ zurück. Bei Linien ist der Typ gleich dem Verschiebungswert.
     *
     * @param type der Koordinatentyp
     *
     * @return die Koordinaten
     */
    @Override
    public List<Object> getCoordinates(int type) {
        if (!_coordinates.containsKey(type)) {
            _coordinates.put(type, _painter.getCoordinates(_coordinates.get(0), type));
        }
        return _coordinates.get(type);
    }

    /**
     * Gibt die Koordinaten zum Default(-Koordinaten)-Typ zurück.
     *
     * @return die Default-Koordinaten
     */
    @Override
    public List<Object> getCoordinates() {
        return getCoordinates(_defaultType);
    }

    /**
     * Im Moment ist der SelectionPainter die einzige Anwendung, die die Referenzpunkte nutzt. Und wir geben nur einen Referenzpunkt zurück.
     *
     * @param utmBounds ein Rechteck, in dem die Referenzpunkte liegen sollen
     *
     * @return eine ein-elementige Liste von Referenzpunkten
     */
    @NotNull
    @Override
    public List<Point2D> getReferencePoints(final Rectangle utmBounds) {
        List<Point2D> pointList = new ArrayList<>(1);

        for (final Map.Entry<Integer, List<Object>> integerListEntry : _coordinates.entrySet()) {
            if (integerListEntry.getValue().isEmpty()) {
                continue;
            }
            Point2D p = getReferencePoint(utmBounds, integerListEntry.getValue());
            if (null != p) {
                pointList.add(p);
                return pointList;
            }
        }
        return pointList;
    }

    /**
     * Gibt das Painter-Objekt {@link DisplayObjectPainter} zu dieses OnlineDisplayObject zurück.
     *
     * @return den Painter
     */
    @Override
    public DisplayObjectPainter getPainter() {
        return _painter;
    }

    @Override
    public void setPainter(DisplayObjectPainter painter) {
        _painter = painter;
    }

    /**
     * Gibt die {@link DOTCollection} zu diesem OnlineDisplayObject zurück.
     *
     * @return die DOTCollection
     */
    @Override
    public DOTCollection getDOTCollection() {
        return _dotCollection;
    }

    /**
     * Gibt die umgebende Rechteck zu diesem OnlineDisplayObject für den angebenen (Koordinaten-)Typ zurück.
     *
     * @return das umgebende Rechteck
     */
    @Override
    public Rectangle getBoundingRectangle(int type) {
        if (_painter instanceof DOTNeedlePainter) {
            // Nadel-koordinaten nicht zwischenspeichern, da Nadeln in der Größe veränderlich sind
            return _painter.getBoundingRectangle(this, type);
        }
        if (!_boundingRectangles.containsKey(type)) {
            _boundingRectangles.put(type, _painter.getBoundingRectangle(this, type));
        }
        return _boundingRectangles.get(type);
    }

    /**
     * Gibt die umgebende Rechteck zu diesem OnlineDisplayObject für den Default-(Koordinaten-)Typ zurück.
     *
     * @return das umgebende Rechteck
     */
    @Override
    public Rectangle getBoundingRectangle() {
        return getBoundingRectangle(_defaultType);
    }

    /*
     * Dies ist die Methode, die für das ClientReceiverInterface implementiert wird.
     */
    @SuppressWarnings("BusyWait")
    @Override
    public void update(ResultData[] results) {
        try {
            int mapScale = _mapPane.getMapScale().intValue();
            // Wen die folgende Schleife überrascht: sie sollte tatsächlich NIE ausgeführt
            // werden, denn das mapPane kennt seinen Maßstab bevor überhaupt Anmeldungen
            // durchgeführt werden. Man könnte auch eine Exception werfen, aber zu so
            // drastischen Mitteln besteht kein Grund. Den SubscriptionDeliveryThread
            // zu blockieren ist natürlich nur eine Ausnahmelösung.
            while (mapScale == 0) {
                _debug.warning("Warten auf die Kartenansicht.");
                Thread.sleep(100);
                mapScale = _mapPane.getMapScale().intValue();
            }
            _currentDisplayObjectType = _dotCollection.getDisplayObjectType(mapScale);
            if (_currentDisplayObjectType == null) {
                return;
            }
            if (_displayObjectTypeItems == null) {
                _displayObjectTypeItems = _resultCache.get(_currentDisplayObjectType);
            }
            for (ResultData result : results) {
                final DataDescription dataDescription = result.getDataDescription();
                final AttributeGroup attributeGroup = dataDescription.getAttributeGroup();
                final Aspect aspect = dataDescription.getAspect();
                DOTSubscriptionData subscriptionData = new DOTSubscriptionData(attributeGroup.getPid(), aspect.getPid());
                update(result, subscriptionData);
                updateResultCache(result, subscriptionData);
            }
        } catch (Exception e) {
            _debug.warning("OnlineDisplayObject.update(): ein Update konnte nicht durchgeführt werden.", e);
        }
    }

    private void update(ResultData result, DOTSubscriptionData subscriptionData) {
        final Data data = result.getData();
        for (PrimitiveFormPropertyPair pfPropertyPair : _displayObjectTypeItems.keySet()) {
            if (data == null) {
                final DisplayObjectTypeItem dItem = getDOTItemForState(_currentDisplayObjectType, result, subscriptionData, pfPropertyPair);
                _displayObjectTypeItems.put(pfPropertyPair, dItem);
                _mapPane.updateDisplayObject(this);
            } else {
                List<String> attributeNames = _currentDisplayObjectType
                    .getAttributeNames(pfPropertyPair.getPrimitiveFormName(), pfPropertyPair.getProperty(), subscriptionData);
                for (String attributeName : attributeNames) {
                    if (DynamicDefinitionComponent.attributeNameIsState(attributeName)) {
                        continue;
                    }
                    final Data subItem = getSubItem(data, attributeName);
                    _values.put(pfPropertyPair, subItem);
                    double value = subItem.asUnscaledValue().doubleValue();
                    DisplayObjectTypeItem dItem = _currentDisplayObjectType
                        .getDOTItemForValue(pfPropertyPair.getPrimitiveFormName(), pfPropertyPair.getProperty(), subscriptionData, attributeName,
                                            value);
                    _displayObjectTypeItems.put(pfPropertyPair, dItem);    // auch bei null!
                    _mapPane.updateDisplayObject(this);
                }
            }
        }
    }

    private void updateResultCache(ResultData result, DOTSubscriptionData subscriptionData) {
        final Data data = result.getData();
        for (final Map.Entry<DisplayObjectType, Map<PrimitiveFormPropertyPair, DisplayObjectTypeItem>> displayObjectTypeMapEntry : _resultCache
            .entrySet()) {
            for (PrimitiveFormPropertyPair pfPropertyPair : _displayObjectTypeItems.keySet()) {
                if (data == null) {
                    DisplayObjectTypeItem dItem = getDOTItemForState(displayObjectTypeMapEntry.getKey(), result, subscriptionData, pfPropertyPair);
                    displayObjectTypeMapEntry.getValue().put(pfPropertyPair, dItem);
                } else {
                    List<String> attributeNames = displayObjectTypeMapEntry.getKey()
                        .getAttributeNames(pfPropertyPair.getPrimitiveFormName(), pfPropertyPair.getProperty(), subscriptionData);
                    for (String attributeName : attributeNames) {
                        if (DynamicDefinitionComponent.attributeNameIsState(attributeName)) {
                            continue;
                        }
                        final Data subItem = getSubItem(data, attributeName);
                        _values.put(pfPropertyPair, subItem);
                        double value = subItem.asUnscaledValue().doubleValue();
                        DisplayObjectTypeItem dItem = displayObjectTypeMapEntry.getKey()
                            .getDOTItemForValue(pfPropertyPair.getPrimitiveFormName(), pfPropertyPair.getProperty(), subscriptionData, attributeName,
                                                value);
                        displayObjectTypeMapEntry.getValue().put(pfPropertyPair, dItem);    // auch bei null!
                    }
                }
            }
        }
    }

    /**
     * Setzt den Default-Type.
     *
     * @param defaultType der Default-(Koordinaten-)Typ
     */
    @Override
    public void setDefaultType(int defaultType) {
        _defaultType = defaultType;
    }

    /*
     * Das ist die Methode, die für den MapPane.MapScaleListener implementiert werden muss.
     */
    @Override
    public void mapScaleChanged(double scale) {
        final DisplayObjectType displayObjectType = _dotCollection.getDisplayObjectType(_mapPane.getMapScale().intValue());
        if (!Objects.equals(displayObjectType, _currentDisplayObjectType)) {
            _currentDisplayObjectType = displayObjectType;
            _displayObjectTypeItems = _resultCache.get(_currentDisplayObjectType);
        }
    }

    /**
     * Eine ausgabefreundliche Beschreibung des Objekts.
     *
     * @return eine ausgabefreundliche Beschreibung
     */
    @Override
    public String toString() {
        return "[OnlineDisplayObject: [" + _systemObject.toString() + "][Number of Coordinates:" + _coordinates.size() + "]" +
               _dotCollection.toString() + "]";
    }

    /**
     * Gibt eine Referenz auf das MapPane-Objekt zurück.
     *
     * @return MapPane
     */
    public MapPane getMapPane() {
        return _mapPane;
    }
}
