/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2021 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.kappich.pat.gnd.extLocRef;

import de.bsvrz.dav.daf.main.config.DataModel;
import java.util.ArrayList;
import java.util.List;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;
import javax.swing.JOptionPane;

/**
 * Diese Klasse verkapselt die Definition einer Erweiterten Ortsreferenz (EOR). Das ist eine eine Liste von {@link DirectedReference
 * DirectedReferences}, deren Objekte eine *sinnvolle* Folge für eine Geo-Initialisierung bilden; die Bedeutung von sinnvoll in diesem Zusammenhang
 * ist allerdings schwierig. Wegen der möglichen Mehrfachvererbung, kann man schwer feststellen, ob eine solche Folge zu irgendeinem Ziel führt. Am
 * Ende der letzten DirectedSimpleReference muss in jedem Fall ein Subtyp von GeoReferenzObjekt stehen.
 *
 * @author Kappich Systemberatung
 */
public class ComposedReference implements Comparable<ComposedReference> {

    private static final String INFO = "info";
    private static final String GEOMETRY_TYPE = "geometryType";
    private static final String DIRECTED_REFERENCE = "dr";

    private String _name;
    private String _info;
    private String _geometryType;

    private DRCollection _directedReferences;

    public ComposedReference() {
        _name = "";
        _info = "";
        _geometryType = "";
        _directedReferences = new DRCollection();
    }

    public ComposedReference(String name, String info, String geometryType) {
        _name = name;
        _info = info;
        _geometryType = geometryType;
        _directedReferences = new DRCollection();
    }

    public ComposedReference(String name, String info, String geometryType, final List<DirectedReference> directedReferences) {
        _name = name;
        _info = info;
        _geometryType = geometryType;
        _directedReferences = new DRCollection(directedReferences);
    }

    public String getName() {
        return _name;
    }

    public void setName(String name) {
        _name = name;
    }

    public String getInfo() {
        return _info;
    }

    public void setInfo(String info) {
        _info = info;
    }

    public String getGeometryType() {
        return _geometryType;
    }

    public void setGeometryType(String geometryType) {
        _geometryType = geometryType;
    }

    @SuppressWarnings("ReturnOfCollectionOrArrayField")
    public DRCollection getDirectedReferences() {
        return _directedReferences;
    }

    @SuppressWarnings("SameParameterValue")
    public void setDirectedReferences(List<DirectedReference> directedReferences, boolean copy) {
        if (copy) {
            _directedReferences = new DRCollection();
            for (DirectedReference dr : directedReferences) {
                _directedReferences.add((DirectedReference) dr.getCopy());
            }
        } else {
            _directedReferences = new DRCollection(directedReferences);
        }
    }

    @SuppressWarnings("SameParameterValue")
    public void setDirectedReferences(DRCollection directedReferences, boolean copy) {
        if (copy) {
            _directedReferences = new DRCollection();
            for (int i = 0; i < directedReferences.size(); ++i) {
                _directedReferences.add((DirectedReference) directedReferences.get(i).getCopy());
            }
        } else {
            _directedReferences = new DRCollection(directedReferences.getDirectedReferences());
        }
    }

    // Diese Methode ist nach dem Muster in Layer.java gestrickt; natürlich ist das JOptionPane
    // Mist; bei Gelegenheit hier wie dort und sonstwo eine Rückgabewert boolean für Erfolg/Misserfolg
    // einführen, und der GUI-Komponente die Fehlermeldung überlassen. Ist aber aufwendig.
    public void deletePreferences(final Preferences prefs) {
        Preferences objectPrefs = prefs.node(getName());
        try {
            objectPrefs.removeNode();
        } catch (BackingStoreException e) {
            JOptionPane
                .showMessageDialog(null, "Das Löschen der Erweiterten Ortsreferenz war nicht erfolgreich. " + System.lineSeparator() + e.toString(),
                                   "Fehlermeldung", JOptionPane.ERROR_MESSAGE);
        }
    }

    public void putPreferences(final Preferences prefs) {
        deletePreferences(prefs);
        Preferences objectPrefs = prefs.node(getName());
        if (_info == null) {
            objectPrefs.put(INFO, "");
        } else {
            objectPrefs.put(INFO, _info);
        }
        if (_geometryType == null) {
            objectPrefs.put(GEOMETRY_TYPE, "");
        } else {
            objectPrefs.put(GEOMETRY_TYPE, _geometryType);
        }
        for (int i = 0; i < _directedReferences.size(); ++i) {
            Preferences drPrefs = objectPrefs.node(DIRECTED_REFERENCE + i);
            _directedReferences.get(i).putPreferences(drPrefs);
        }
    }

    public boolean initializeFromPreferences(final Preferences prefs, final DataModel configuration) {
        _name = prefs.name();
        _info = prefs.get(INFO, "");
        _geometryType = prefs.get(GEOMETRY_TYPE, "");
        int i = 0;
        String pathName = DIRECTED_REFERENCE + i;
        try {
            while (prefs.nodeExists(pathName)) {
                Preferences drPrefs = prefs.node(pathName);
                DirectedReference directedReference = new DirectedReference();
                if (!directedReference.initializeFromPreferences(drPrefs, configuration)) {
                    return false;
                }
                _directedReferences.add(directedReference);
                ++i;
                pathName = DIRECTED_REFERENCE + i;
            }
        } catch (BackingStoreException ignore) {
            return false;
        }
        return true;
    }

    @Override
    public int compareTo(final ComposedReference o) {
        return getName().toLowerCase().compareTo(o.getName().toLowerCase());
    }

    @Override
    public boolean equals(Object o) {
	    if (!(o instanceof ComposedReference other)) {
            return false;
        }
        return getName().toLowerCase().equals(other.getName().toLowerCase());
    }

    @Override
    public int hashCode() {
        return getName().toLowerCase().hashCode();
    }

    public ComposedReference getCopy() {
        List<DirectedReference> newList = new ArrayList<>();
        for (int i = 0; i < _directedReferences.size(); ++i) {
            newList.add((DirectedReference) _directedReferences.get(i).getCopy());
        }
        return new ComposedReference(_name, _info, _geometryType, newList);
    }

    @Override
    public String toString() {
        return "ComposedReference{" + "_name='" + _name + '\'' + ", _info='" + _info + '\'' + ", _geometryType='" + _geometryType + '\'' +
               ", _directedReferences=" + _directedReferences + '}';
    }
}
