/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.extLocRef;

import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.sys.funclib.debug.Debug;
import java.util.prefs.Preferences;

/**
 * Das Interface verkapselt die gerichtete Association zweier {@link SystemObjectType SystemObjectTypes}
 *
 * @author Kappich Systemberatung
 */
public class DirectedReference implements SimpleReference {

    private static final Debug _debug = Debug.getLogger();

    private static final String DESCRIPTION = "description";
    private static final String REVERSED = "reversed";
    private static final String SIMPLE_REFERENCE_CLASS = "SR_CLASS";
    private static final String SIMPLE_REFERENCE_NODE = "SR";

    private String _description;
    private SimpleReference _simpleReference;
    private boolean _reversed;

    DirectedReference() {
        _description = "";
        _simpleReference = null;
        _reversed = false;
    }

    public DirectedReference(final String description, final SimpleReference simpleReference, final boolean reversed) {
        _description = description;
        _simpleReference = simpleReference;
        _reversed = reversed;
    }

    @Override
    public SystemObjectType getFirstType() {
        if (!_reversed) {
            return _simpleReference.getFirstType();
        } else {
            return _simpleReference.getSecondType();
        }
    }

    @Override
    public SystemObjectType getSecondType() {
        if (!_reversed) {
            return _simpleReference.getSecondType();
        } else {
            return _simpleReference.getFirstType();
        }
    }

    @Override
    public void putPreferences(final Preferences prefs) {
        prefs.put(DESCRIPTION, _description);
        prefs.putBoolean(REVERSED, _reversed);
        prefs.put(SIMPLE_REFERENCE_CLASS, _simpleReference.getClass().getName());
        Preferences srPrefs = prefs.node(SIMPLE_REFERENCE_NODE);
        _simpleReference.putPreferences(srPrefs);
    }

    @Override
    public boolean initializeFromPreferences(final Preferences prefs, final DataModel configuration) {
        _description = prefs.get(DESCRIPTION, "");
        _reversed = prefs.getBoolean(REVERSED, false);
        String className = prefs.get(SIMPLE_REFERENCE_CLASS, null);
        if (null == className) {
            return false;
        }
        Class<?> srClass;
        try {
            srClass = Class.forName(className);
        } catch (ClassNotFoundException cnfe) {
            _debug.error("ClassNotFoundException in DirectedReference.initializeFromPreferences for '" + className + "'");
            _debug.error(cnfe.getMessage());
            return false;
        }
        Object srObject;
        try {
            srObject = srClass.newInstance();
        } catch (InstantiationException ie) {
            _debug.error("InstantiationException in DirectedReference.initializeFromPreferences for '" + className + "'");
            _debug.error(ie.getMessage());
            return false;
        } catch (IllegalAccessException iae) {
            _debug.error("IllegalAccessException in DirectedReference.initializeFromPreferences for '" + className + "'");
            _debug.error(iae.getMessage());
            return false;
        }
        Preferences srPrefs = prefs.node(SIMPLE_REFERENCE_NODE);
        boolean initSr;
        try {
            initSr = ((SimpleReference) srObject).initializeFromPreferences(srPrefs, configuration);
        } catch (ClassCastException ignore) {
            return false;
        }
        if (initSr) {
            _simpleReference = (SimpleReference) srObject;
        }
        return initSr;
    }

    @Override
    public SimpleReference getCopy() {
        return new DirectedReference(_description, _simpleReference.getCopy(), _reversed);
    }

    public String getDescription() {
        return _description;
    }

    public void setDescription(final String description) {
        _description = description;
    }

    public SimpleReference getSimpleReference() {
        return _simpleReference;
    }

    public boolean isReversed() {
        return _reversed;
    }

    @Override
    public String toString() {
        return "DirectedReference{" + "_description='" + _description + '\'' + ", _simpleReference=" + _simpleReference + ", _reversed=" + _reversed +
               '}';
    }
}
