/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.extLocRef;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.sys.funclib.debug.Debug;
import de.kappich.pat.gnd.gnd.PreferencesHandler;
import de.kappich.pat.gnd.utils.view.PreferencesDeleter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;
import javax.swing.table.AbstractTableModel;

/**
 * Dieser Manager verwaltet alle {@link ReferenceHierarchy ReferenceHierarchy-Objekte}, das sind die EOR-Hierarchien.
 *
 * @author Kappich Systemberatung
 */

@SuppressWarnings("serial")
public final class ReferenceHierarchyManager extends AbstractTableModel {

    private static final Debug _debug = Debug.getLogger();
    private static final Object _initializationAccess = new Object();
    private static final String[] COLUMN_NAMES = {"Name des Referenz-Hierarchie"};
    private static ReferenceHierarchyManager _instance;
    private final ClientDavInterface _connection;
	private final List<ReferenceHierarchy> _referenceHierarchies = new ArrayList<>() {
        @Override
        public boolean add(ReferenceHierarchy referenceHierarchy) {
            int index = Collections.binarySearch(this, referenceHierarchy);
            if (index < 0) {
                index = ~index;
            }
            super.add(index, referenceHierarchy);
            return true;
        }
    };
    private final Set<String> _unchangeables = new HashSet<>();
    private final Map<String, ReferenceHierarchy> _referenceHierarchiesMap = new HashMap<>();
    private final List<ReferenceHierarchyManager.RhmChangeListener> _listeners = new CopyOnWriteArrayList<>();

    private ReferenceHierarchyManager(ClientDavInterface connection) {
        _connection = connection;
        initDefaultHierarchies();
        initUserDefinedHierarchies();
    }

    public static ReferenceHierarchyManager getInstance(ClientDavInterface connection) {
        //noinspection SynchronizationOnStaticField
        synchronized (_initializationAccess) {
            if (null == _instance) {
                _instance = new ReferenceHierarchyManager(connection);
            }
            return _instance;
        }
    }

    public static ReferenceHierarchyManager getInstance() {
        //noinspection SynchronizationOnStaticField
        synchronized (_initializationAccess) {
            return _instance;
        }
    }

    /**
     * Gibt den Ausgangsknoten zum Abspeichern aller Präferenzen des ReferenceHierarchyManager an.
     *
     * @return gibt den Ausgangsknoten zum Abspeichern aller Präferenzen des ComposedReferenceManager zurück
     */
    private static Preferences getPreferenceStartPath() {
        return PreferencesHandler.getInstance().getPreferenceStartPath().node("RH");
    }

    public void addReferenceHierarchy(final ReferenceHierarchy referenceHierarchy) {
        if (_referenceHierarchiesMap.containsKey(referenceHierarchy.getName())) {
            throw new IllegalArgumentException("Ein EOR-Hierarchie mit diesem Namen existiert bereits.");
        }
        // Klein-/Großschreibung nicht signifikant:
        for (ReferenceHierarchy rh : _referenceHierarchies) {
            if (referenceHierarchy.equals(rh)) {
                throw new IllegalArgumentException(
                    "Es existiert bereits eine EOR-Hierarchie, dessen Name sich nur bezüglich Klein-/Großschreibung unterscheidet.");
            }
        }
        _referenceHierarchies.add(referenceHierarchy);
        _referenceHierarchiesMap.put(referenceHierarchy.getName(), referenceHierarchy);
        referenceHierarchy.putPrefernces(getPreferenceStartPath());
        fireTableDataChanged();
        notifyChangeListenerHierarchyAdded(referenceHierarchy);
    }

    public void changeReferenceHierarchy(final ReferenceHierarchy referenceHierarchy) {
        final String name = referenceHierarchy.getName();
        if (!_referenceHierarchiesMap.containsKey(name)) {
            throw new IllegalArgumentException("Ein EOR-Hierarchie mit diesem Namen existiert nicht.");
        }
        final ReferenceHierarchy existing = _referenceHierarchiesMap.get(name);
        existing.deletePreferences(getPreferenceStartPath());
        existing.setInfo(referenceHierarchy.getInfo());
        existing.setComposedReferences(referenceHierarchy.getComposedReferences().getComposedReferences(), true);
//		_referenceHierarchiesMap.put(referenceHierarchy.getName(), referenceHierarchy);
        for (int i = 0; i < _referenceHierarchies.size(); ++i) {
            if (_referenceHierarchies.get(i).getName().equals(name)) {
                fireTableRowsUpdated(i, i);
                break;
            }
        }
        referenceHierarchy.putPrefernces(getPreferenceStartPath());
        notifyChangeListenersHierarchyChanged(referenceHierarchy);
    }

    public boolean removeReferenceHierarchy(ReferenceHierarchy referenceHierarchy) {
        final String name = referenceHierarchy.getName();
        if (!_unchangeables.contains(name)) {
            final int index = remove(name);
            referenceHierarchy.deletePreferences(getPreferenceStartPath());
            fireTableRowsDeleted(index, index);
            notifyChangeListenersHierarchyRemoved(name);
            return true;
        }
        return false;
    }

    private int remove(final String name) {
        int index = 0;
        for (ReferenceHierarchy referenceHierarchy : _referenceHierarchies) {
            if (name.equals(referenceHierarchy.getName())) {
                _referenceHierarchies.remove(index);
                _referenceHierarchiesMap.remove(name);
                return index;
            }
            index++;
        }
        return -1;
    }

    public boolean hasReferenceHierarchie(final String name) {
        return _referenceHierarchiesMap.containsKey(name);
    }

    /**
     * Gibt eine sortierte Kopie der Liste aller ReferenceHierarchy-Objekte zurück.
     *
     * @return die Liste aller ReferenceHierarchy-Objekte
     */
    public List<ReferenceHierarchy> getReferenceHierarchies() {
        return Collections.unmodifiableList(_referenceHierarchies);
    }

    /**
     * Gibt eine Array mit allen Namen der ReferenceHierarchy-Objekte zurück.
     *
     * @return das Array
     */
    public String[] getReferenceHierarchyNames(final boolean withNoneEntry) {
        if (withNoneEntry) {
            String[] array = new String[_referenceHierarchies.size() + 1];
            array[0] = "";
            for (int i = 0; i < _referenceHierarchies.size(); ++i) {
                array[i + 1] = _referenceHierarchies.get(i).getName();
            }
            return array;
        } else {
            String[] array = new String[_referenceHierarchies.size()];
            for (int i = 0; i < _referenceHierarchies.size(); ++i) {
                array[i] = _referenceHierarchies.get(i).getName();
            }
            return array;
        }
    }

    /**
     * Gibt die ReferenceHierarchy mit dem übergebenen Namen zurück.
     *
     * @param name der Name
     *
     * @return die geforderte ReferenceHierarchy
     */
    public ReferenceHierarchy getReferenceHierarchy(String name) {
        return _referenceHierarchiesMap.get(name);
    }

    /**
     * Gibt die ReferenceHierarchy an der i-ten Stelle der ReferenceHierarchy-Liste zurück, wobei die Zählung mit 0 beginnt.
     *
     * @param i ein Index
     *
     * @return die geforderte ReferenceHierarchy
     */
    public ReferenceHierarchy getReferenceHierarchy(int i) {
        return _referenceHierarchies.get(i);
    }

    /**
     * Prüft, ob die übergebene {@link ComposedReference EOR} in einer {@link ReferenceHierarchy EOR-Hirarchie} benutzt wird.
     *
     * @param reference eine EOR
     *
     * @return {@code true}, falls ja
     */
    public boolean isUsed(final ComposedReference reference) {
        for (ReferenceHierarchy hierarchy : _referenceHierarchies) {
            if (hierarchy.getComposedReferences().contains(reference)) {
                return true;
            }
        }
        return false;
    }

    /*
     * Gehört zur Implementation des TableModel.
     */
    @Override
    public int getColumnCount() {
        return COLUMN_NAMES.length;
    }

    /*
     * Gehört zur Implementation des TableModel.
     */
    @Override
    public int getRowCount() {
        return _referenceHierarchies.size();
    }

    /*
     * Gehört zur Implementation des TableModel.
     */
    @Override
    public Object getValueAt(int rowIndex, int columnIndex) {
        return _referenceHierarchies.get(rowIndex).getName();
    }

    /*
     * Gehört zur Implementation des TableModel.
     */
    @Override
    public String getColumnName(int columnIndex) {
        return COLUMN_NAMES[columnIndex];
    }

    /**
     * Gibt {@code true} zurück, wenn die ReferenceHierarchy veränderbar ist. Im Moment ist eine ReferenceHierarchy genau dann unveränderbar, wenn er
     * im Kode definiert ist.
     *
     * @param rh eine ReferenceHierarchy
     *
     * @return {@code true} genau dann, wenn die ReferenceHierarchy veränderbar ist
     */
    public boolean isChangeable(ReferenceHierarchy rh) {
        return !_unchangeables.contains(rh.getName());
    }

    @SuppressWarnings("unused")
    public void addRhmChangeListener(final RhmChangeListener listener) {
        _listeners.add(listener);
    }

    @SuppressWarnings("unused")
    public void removeRhmChangeListener(final RhmChangeListener listener) {
        _listeners.remove(listener);
    }

    @SuppressWarnings("unused")
    public void clearRhmChangeListeners() {
        _listeners.clear();
    }

    /**
     * Informiert die auf Änderungen angemeldeten Listener über eine neu hinzugefügte {@link ReferenceHierarchy}.
     *
     * @param hierarchy eine ReferenceHierarchy
     */
    private void notifyChangeListenerHierarchyAdded(final ReferenceHierarchy hierarchy) {
        for (ReferenceHierarchyManager.RhmChangeListener listener : _listeners) {
            listener.referenceHierarchyAdded(hierarchy);
        }
    }

    /**
     * Informiert die auf Änderungen angemeldeten Listener über eine geänderte {@link ReferenceHierarchy}.
     *
     * @param hierarchy eine ReferenceHierarchy
     */
    private void notifyChangeListenersHierarchyChanged(final ReferenceHierarchy hierarchy) {
        for (ReferenceHierarchyManager.RhmChangeListener listener : _listeners) {
            listener.referenceHierarchyChanged(hierarchy);
        }
    }

    /**
     * Informiert die auf Änderungen angemeldeten Listener über eine gelöschte {@link ReferenceHierarchy} mit dem Namen.
     *
     * @param name ein Name
     */
    private void notifyChangeListenersHierarchyRemoved(final String name) {
        for (ReferenceHierarchyManager.RhmChangeListener listener : _listeners) {
            listener.referenceHierarchyRemoved(name);
        }
    }

    private void initDefaultHierarchies() {
        ComposedReferenceManager crm = ComposedReferenceManager.getInstance(_connection);
        ComposedReference cr1 = crm.getComposedReference("Fahrstreifen -> Messquerschnitt");
        if (null == cr1) {
            _debug.error("Achtung: die vordefinierte EOR 'Fahrstreifen -> Messquerschnitt' konnte nicht geladen werden.");
        }
        ComposedReference cr2 = crm.getComposedReference("Fahrstreifen -> DeLve");
        if (null == cr2) {
            _debug.error("Achtung: die vordefinierte EOR 'Fahrstreifen -> DeLve' konnte nicht geladen werden.");
        }
        ComposedReference cr3 = crm.getComposedReference("Fahrstreifen -> Eak (über DeLve und menge.de)");
        if (null == cr3) {
            _debug.error("Achtung: die vordefinierte EOR 'Fahrstreifen -> Eak (über DeLve und menge.de)' konnte nicht geladen werden.");
        }

        {
            List<ComposedReference> crl1 = new ArrayList<>();
            crl1.add(cr1);
            ReferenceHierarchy rh1 = new ReferenceHierarchy("Fahrstreifen -> Messquerschnitt", "Enthält nur die gleichnamige EOR", "Punkt", crl1);
            _referenceHierarchies.add(rh1);
            _referenceHierarchiesMap.put(rh1.getName(), rh1);
            _unchangeables.add(rh1.getName());
        }

        {
            List<ComposedReference> crl2 = new ArrayList<>();
            crl2.add(cr1);
            crl2.add(cr2);
            crl2.add(cr3);
            ReferenceHierarchy rh2 = new ReferenceHierarchy("Fahrstreifen -> Messquerschnitt, DeLve, Eak", "Enthält drei EOR", "Punkt", crl2);
            _referenceHierarchies.add(rh2);
            _referenceHierarchiesMap.put(rh2.getName(), rh2);
            _unchangeables.add(rh2.getName());
        }
    }

    //++++

    private void initUserDefinedHierarchies() {
        Preferences classPrefs = getPreferenceStartPath();
        String[] hierarchies;
        try {
            hierarchies = classPrefs.childrenNames();
        } catch (BackingStoreException | IllegalStateException ignored) {
            PreferencesDeleter pd = new PreferencesDeleter("Die benutzer-definierten EOR-Hierarchien können nicht geladen werden.", classPrefs);
            pd.run();
            return;
        }
        boolean success = true;
        for (String name : hierarchies) {
            Preferences hierarchyPrefs = classPrefs.node(name);
            ReferenceHierarchy hierarchy = new ReferenceHierarchy();
            if (hierarchy.initializeFromPreferences(hierarchyPrefs)) {
                addReferenceHierarchy(hierarchy);
            } else {
                success = false;
            }
        }
        if (!success) {
            PreferencesDeleter pd = new PreferencesDeleter("Nicht alle benutzer-definierten EOR-Hierarchien konnten geladen werden.", classPrefs);
            pd.run();
        }
    }

    @Override
    public String toString() {
        return "ReferenceHierarchyManager{" + "_referenceHierarchies=" + _referenceHierarchies + '}';
    }

    /**
     * Ein Interface für Listener, die über das Hinzufügen, Löschen und Ändern von ReferenceHierarchies informiert werden wollen.
     *
     * @author Kappich Systemberatung
     */
    public interface RhmChangeListener {
        /**
         * Diese Methode wird aufgerufen, wenn die EOR-Hierarchie hinzugefügt wurde.
         *
         * @param referenceHierarchy eine ReferenceHierarchy
         */
        void referenceHierarchyAdded(final ReferenceHierarchy referenceHierarchy);

        /**
         * Diese Methode wird aufgerufen, wenn die EOR-Hierarchie geändert wurde.
         *
         * @param referenceHierarchy eine ReferenceHierarchy
         */
        void referenceHierarchyChanged(final ReferenceHierarchy referenceHierarchy);

        /**
         * Diese Methode wird aufgerufen, wenn die EOR-Hierarchie mit dem Namen gelöscht wurde.
         *
         * @param name ein Name
         */
        void referenceHierarchyRemoved(final String name);
    }
}
