/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.extLocRef;

import de.bsvrz.dav.daf.main.config.Attribute;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.prefs.Preferences;

/**
 * Diese Klasse implentiert das Interface {@link SimpleReference} für durch Attributgruppen und Attribute gegebene Referenzen.
 *
 * @author Kappich Systemberatung
 */
public class SimpleAttributeReference implements SimpleReference {

    private static final String FIRST_TYPE = "firstType";
    private static final String SECOND_TYPE = "secondType";
    private static final String ATTRIBUTE_GROUP = "attributeGroup";
    private static final String ATTRIBUTE = "attribute";
    private static final String THE_LIST_KEYS = "theListKeys";

    private SystemObjectType _firstType;
    private SystemObjectType _secondType;
    private AttributeGroup _attributeGroup;
    private Attribute _attribute;
    private List<String> _theListKeys;

    /*
    Dieser Konstruktor wird tatsächlich gebraucht, und zwar in DirectedReference.initializeFromPreferences!
    Dort wird newInstance von Class aufgerufen.
     */
    @SuppressWarnings("unused")
    SimpleAttributeReference() {
        _firstType = null;
        _secondType = null;
        _attributeGroup = null;
        _attribute = null;
        _theListKeys = null;
    }

    SimpleAttributeReference(final SystemObjectType firstType, final SystemObjectType secondType, final AttributeGroup attributeGroup,
                             final Attribute attribute, final List<String> theListKeys) {
        _firstType = firstType;
        _secondType = secondType;
        _attributeGroup = attributeGroup;
        _attribute = attribute;
        _theListKeys = theListKeys;
    }

    @Override
    public SystemObjectType getFirstType() {
        return _firstType;
    }

    @Override
    public SystemObjectType getSecondType() {
        return _secondType;
    }

    @Override
    public void putPreferences(final Preferences prefs) {
        prefs.put(FIRST_TYPE, _firstType.getPid());
        prefs.put(SECOND_TYPE, _secondType.getPid());
        prefs.put(ATTRIBUTE_GROUP, _attributeGroup.getPid());
        prefs.put(ATTRIBUTE, _attribute.getName());
        for (int i = 0; i < _theListKeys.size(); ++i) {
            prefs.put(THE_LIST_KEYS + i, _theListKeys.get(i));
        }
    }

    @Override
    public boolean initializeFromPreferences(final Preferences prefs, final DataModel configuration) {
        String firstTypePid = prefs.get(FIRST_TYPE, null);
        String secondTypePid = prefs.get(SECOND_TYPE, null);
        String attributeGroupPid = prefs.get(ATTRIBUTE_GROUP, null);
        String attributeName = prefs.get(ATTRIBUTE, null);
        if (null == firstTypePid || null == secondTypePid || null == attributeGroupPid || null == attributeName) {
            return false;
        }
        _firstType = configuration.getType(firstTypePid);
        _secondType = configuration.getType(secondTypePid);
        _attributeGroup = configuration.getAttributeGroup(attributeGroupPid);
        if (null == _attributeGroup) {
            return false;
        }
        _attribute = _attributeGroup.getAttribute(attributeName);
        if (null != _theListKeys) {
            _theListKeys.clear();
        } else {
            _theListKeys = new ArrayList<>();
        }
        int i = 0;
        String currentListKey = prefs.get(THE_LIST_KEYS + i, null);
        while (currentListKey != null) {
            _theListKeys.add(currentListKey);
            ++i;
            currentListKey = prefs.get(THE_LIST_KEYS + i, null);
        }
        return !(null == _firstType || null == _secondType || null == _attributeGroup || null == _attribute);
    }

    @Override
    public SimpleReference getCopy() {
        return new SimpleAttributeReference(_firstType, _secondType, _attributeGroup, _attribute, _theListKeys);
    }

    public AttributeGroup getAttributeGroup() {
        return _attributeGroup;
    }

    public Attribute getAttribute() {
        return _attribute;
    }

    @SuppressWarnings("unused")
    public List<String> getTheListKeys() {
        return Collections.unmodifiableList(_theListKeys);
    }

    @Override
    public String toString() {
        return "1: " + _firstType.getNameOrPidOrId() + ", 2: " + _secondType.getNameOrPidOrId() + ", Atg: " + _attributeGroup.getNameOrPidOrId();
    }

    @SuppressWarnings("NonFinalFieldReferenceInEquals")
    @Override
    public boolean equals(Object o) {
	    if (!(o instanceof SimpleAttributeReference other)) {
            return false;
        }
        return _firstType.equals(other._firstType) && _secondType.equals(other._secondType) && _attributeGroup.equals(other._attributeGroup) &&
               _attribute.equals(other._attribute) && _theListKeys.equals(other._theListKeys);
    }

    @SuppressWarnings("NonFinalFieldReferencedInHashCode")
    @Override
    public int hashCode() {
        int result = _firstType.hashCode();
        result = 31 * result + _secondType.hashCode();
        result = 31 * result + _attributeGroup.hashCode();
        result = 31 * result + _attribute.hashCode();
        result = 31 * result + _theListKeys.hashCode();
        return result;
    }
}
