/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.extLocRef.view;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.kappich.pat.gnd.documentation.GndHelp;
import de.kappich.pat.gnd.extLocRef.ReferenceHierarchy;
import de.kappich.pat.gnd.extLocRef.ReferenceHierarchyManager;
import de.kappich.pat.gnd.utils.SpringUtilities;
import de.kappich.pat.gnd.utils.view.GndFrame;
import de.kappich.pat.gnd.utils.view.GndTable;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Frame;
import java.awt.Window;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.ArrayList;
import java.util.List;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.SpringLayout;
import javax.swing.WindowConstants;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.DefaultTableCellRenderer;

/**
 * Eine Dialog, der alle EOR-Hierarchien auflistet und über Schaltflächen Möglichkeiten bietet, einzelne zu betrachten,
 * zu bearbeiten, zu löschen oder neu einzuführen. Der RhmDialog zeigt die Inhalte der
 * {@link ReferenceHierarchyManager EOR-Hierarchie-Verwaltung}.
 *
 * @author Kappich Systemberatung
 * @version $Revision$
 */
@SuppressWarnings("OverlyLongMethod")
public class RhmDialog {

    private final ClientDavInterface _connection;

    private final GndFrame _frame;

    @SuppressWarnings("unused")
    private RhDefinitionDialog _rhDefinitionDialog = null;

    /**
     * Konstruiert einen neuen RhmDialog. Das ClientDavInterface ist z.B. notwendig, um über das
     * DataModel Informationen zu den Geo-Referenz-Objekten bekommen zu können.
     *
     * @param connection die Datenverteiler-Verbindung
     */
    public RhmDialog(ClientDavInterface connection) {

        _connection = connection;

        _frame = new GndFrame("RhmDialog", "GND: EOR-Hierarchien-Verwaltung");

        _frame.setLayout(new BorderLayout());

        final JButton buttonNeu = new JButton("Neue EOR-Hierarchie");
        buttonNeu.setActionCommand("Eine neue EOR-Hierarchie anlegen");

        final JButton buttonBearbeiten = new JButton("Bearbeiten");
        buttonBearbeiten.setActionCommand("Details der augewählten EOR-Hierarchie bearbeiten");

        final JButton buttonCopy = new JButton("Kopieren");
        buttonCopy.setActionCommand("Die ausgewählten EOR-Hierarchie kopieren und bearbeiten");

        final JButton buttonLoeschen = new JButton("Löschen");
        buttonLoeschen.setActionCommand("Die ausgewählte EOR-Hierarchie löschen");

        final JButton buttonHilfe = new JButton("Hilfe");
        buttonHilfe.setActionCommand("Die Online-Hilfe öffnen");

        buttonBearbeiten.setEnabled(false);
        buttonCopy.setEnabled(false);
        buttonLoeschen.setEnabled(false);

        JPanel buttonsPanelEast = new JPanel();
        buttonsPanelEast.setLayout(new BoxLayout(buttonsPanelEast, BoxLayout.Y_AXIS));

        Dimension d = new Dimension(15, 15);
        buttonsPanelEast.add(Box.createRigidArea(d));

        buttonsPanelEast.add(buttonNeu);
        buttonsPanelEast.add(Box.createRigidArea(d));

        buttonsPanelEast.add(buttonBearbeiten);
        buttonsPanelEast.add(Box.createRigidArea(d));

        buttonsPanelEast.add(buttonCopy);
        buttonsPanelEast.add(Box.createRigidArea(d));

        buttonsPanelEast.add(buttonLoeschen);
        buttonsPanelEast.add(Box.createRigidArea(d));

        buttonsPanelEast.add(buttonHilfe);
        buttonsPanelEast.add(Box.createRigidArea(d));

        buttonsPanelEast.setBorder(BorderFactory.createEmptyBorder(5, 10, 5, 10));

        JPanel buttonsPanelSouth = new JPanel();
        buttonsPanelSouth.setLayout(new SpringLayout());
        buttonsPanelSouth.setBorder(BorderFactory.createEmptyBorder(5, 10, 5, 10));

        JButton schliessenButton = new JButton("Dialog schließen");
        buttonsPanelSouth.add(schliessenButton);
        SpringUtilities.makeCompactGrid(buttonsPanelSouth, 1, 20, 5);

        final GndTable table = new GndTable(ReferenceHierarchyManager.getInstance(_connection));

        table.setFillsViewportHeight(true);
        table.setDefaultRenderer(Object.class, new RhTableRenderer());
        // vordefinierte Darstellungstypen dürfen nicht bearbeitet oder gelöscht werden
        final ListSelectionModel selectionModel = table.getSelectionModel();

        final List<ReferenceHierarchy> rhList = ReferenceHierarchyManager.getInstance(_connection).getReferenceHierarchies();

        ListSelectionListener listSelctionListener = e -> {
            final int selectedRow = table.getSelectedRow();

            if(selectedRow == -1) {
                buttonBearbeiten.setToolTipText("Es ist keine Ansicht ausgewählt worden");
                buttonBearbeiten.setEnabled(false);
                buttonCopy.setToolTipText("Es ist keine Ansicht ausgewählt worden");
                buttonCopy.setEnabled(false);
                buttonLoeschen.setToolTipText("Es ist keine Ansicht ausgewählt worden");
                buttonLoeschen.setEnabled(false);
            }
            else {
                final boolean changeable = ReferenceHierarchyManager.getInstance(_connection).isChangeable(rhList.get(selectedRow));
                if(!changeable) {
                    buttonBearbeiten.setText("Betrachten");
                    buttonBearbeiten.setToolTipText("Details der augewählten EOR-Hierarchie betrachten");
                    buttonBearbeiten.setEnabled(true);
                    buttonCopy.setToolTipText("Kopie der ausgewählten EOR-Hierarchie erstellen und bearbeiten");
                    buttonCopy.setEnabled(true);
                    buttonLoeschen.setToolTipText("Die ausgewählte EOR-Hierarchie ist nicht löschbar");
                    buttonLoeschen.setEnabled(false);
                }
                else {
                    buttonBearbeiten.setText("Bearbeiten");
                    buttonBearbeiten.setToolTipText("Details der augewählten EOR-Hierarchie bearbeiten");
                    buttonBearbeiten.setEnabled(true);
                    buttonCopy.setToolTipText("Kopie der ausgewählten EOR-Hierarchie erstellen und bearbeiten");
                    buttonCopy.setEnabled(true);
                    buttonLoeschen.setToolTipText("Die ausgewählten EOR-Hierarchie löschen");
                    buttonLoeschen.setEnabled(true);
                }
            }
        };
        selectionModel.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        selectionModel.addListSelectionListener(listSelctionListener);

        table.addKeyListener(new KeyAdapter() {
            @Override
            public void keyPressed(KeyEvent e) {
                if(e.getKeyCode() == KeyEvent.VK_DELETE) {
                    final ReferenceHierarchy rh = rhList.get(table.getSelectedRow());
                    final ReferenceHierarchyManager instance = ReferenceHierarchyManager.getInstance(_connection);
                    if(instance.isChangeable(rh)) {
                        instance.removeReferenceHierarchy(rh);
                    }
                }
            }
        });

        table.addMouseListener(new MouseAdapter() {
            @Override
            public void mousePressed(final MouseEvent e) {
                if(e.getClickCount() > 1) {
                    editOrInspectAction(table);
                }
            }
        });

        _frame.add(buttonsPanelEast, BorderLayout.EAST);
        _frame.add(buttonsPanelSouth, BorderLayout.SOUTH);
        _frame.add(new JScrollPane(table), BorderLayout.CENTER);

        // Neu, Bearbeiten, Kopieren, Löschen
        ActionListener actionListenerNew = e -> {
            ReferenceHierarchy newRh = new ReferenceHierarchy("", "", "");
            newRh.setComposedReferences(new ArrayList<>(), false);
            final String title = "GND: neue EOR-Hierarchie bearbeiten";
            _rhDefinitionDialog = new RhDefinitionDialog(newRh, true, true, title);
        };
        buttonNeu.addActionListener(actionListenerNew);


        ActionListener actionListenerLoeschen = e -> {
            if(table.getSelectedRowCount() == 0) {
                JOptionPane.showMessageDialog(
                    _frame.getFrame(),
                    "Bitte wählen Sie eine Zeile aus der Liste aus!",
                    "Fehler",
                    JOptionPane.ERROR_MESSAGE
                );
                return;
            }
            // Im Moment ist eh nur eine Row selektierbar.
            final int[] selectedRows = table.getSelectedRows();
            boolean allReferenceHierarchiesRemoved = true;
            final ReferenceHierarchyManager manager = ReferenceHierarchyManager.getInstance(_connection);
            for(int i = selectedRows.length - 1; i >= 0; i--) {
                final ReferenceHierarchy rh = manager.getReferenceHierarchy(selectedRows[i]);
                if(!manager.removeReferenceHierarchy(rh)) {
                    allReferenceHierarchiesRemoved = false;
                }
            }
            manager.fireTableDataChanged();
            if(!allReferenceHierarchiesRemoved) {
                JOptionPane.showMessageDialog(
                    _frame.getFrame(),
                    "Die ausgewählte EOR-Hierarchien koonte nicht gelöscht werden!",
                    "Fehler",
                    JOptionPane.ERROR_MESSAGE
                );
            }
        };
        buttonLoeschen.addActionListener(actionListenerLoeschen);

        ActionListener actionListenerKopieren = e -> {
            if(table.getSelectedRowCount() != 1) {
                JOptionPane.showMessageDialog(
                    _frame.getFrame(),
                    "Bitte wählen Sie eine Zeile aus der Liste aus!",
                    "Fehler",
                    JOptionPane.ERROR_MESSAGE
                );
                return;
            }
            final int selectedRow = table.getSelectedRow();
            final ReferenceHierarchy rh = ReferenceHierarchyManager.getInstance(_connection).getReferenceHierarchy(selectedRow);
            ReferenceHierarchy newRh = new ReferenceHierarchy(rh.getName() + " (Kopie)", rh.getInfo(), rh.getGeometryType());
            newRh.setComposedReferences(rh.getComposedReferences().getComposedReferences(), true);
            final String title = "GND: eine kopierte EOR-Hierarchie bearbeiten";
            _rhDefinitionDialog = new RhDefinitionDialog(newRh, true, true, title);
        };
        buttonCopy.addActionListener(actionListenerKopieren);

        ActionListener actionListenerBearbeiten = e -> editOrInspectAction(table);
        buttonBearbeiten.addActionListener(actionListenerBearbeiten);

        ActionListener actionListenerHelp = e -> GndHelp.openHelp("#theEorHierachyManagerDialog");
        buttonHilfe.addActionListener(actionListenerHelp);

        ActionListener actionListenerDialogSchliessen = e -> closeDialog();
        schliessenButton.addActionListener(actionListenerDialogSchliessen);

        class FrameListener extends WindowAdapter {
            @Override
            public void windowClosing(WindowEvent e) {
                /*
                 * wenn nur noch ein einziges Fenster geöffnet ist
                 * beendet sich das Programm beim Schließen des Fensters
                 */
                final Window[] windows = Window.getWindows();
                int length = windows.length - 1;
                for(Window window : windows) {
                    if(!window.isVisible()) {
                        length--;
                    }
                }
                if(length <= 0) {
                    _frame.setDefaultCloseOperation(WindowConstants.EXIT_ON_CLOSE);
                }
            }
        }
        _frame.addWindowListener(new FrameListener());

        table.setVisibleRowCount(Math.min(40, table.getModel().getRowCount()));
        _frame.setPositionAndSize(0, 0, 860, 50, true, 0, 0);
        _frame.setVisible(true);
    }

    private void editOrInspectAction(final JTable table) {
        final int selectedRow = table.getSelectedRow();
        if(selectedRow == -1) {
            JOptionPane.showMessageDialog(
                _frame.getFrame(),
                "Bitte wählen Sie eine Zeile aus der Liste aus!",
                "Fehler",
                JOptionPane.ERROR_MESSAGE
            );
            return;
        }
        final ReferenceHierarchy rh = ReferenceHierarchyManager.getInstance(_connection).getReferenceHierarchy(selectedRow);
        final boolean changeable = ReferenceHierarchyManager.getInstance(_connection).isChangeable(rh);
        final String title;
        if(changeable) {
            title = "GND: eine EOR-Hierarchie bearbeiten";
        }
        else {
            title = "GND: eine EOR-Hierarchie betrachten";
        }
        _rhDefinitionDialog = new RhDefinitionDialog(rh, changeable, false, title);
    }

    private class RhTableRenderer extends DefaultTableCellRenderer {

        private static final long serialVersionUID = 1L;

        @Override
        public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
            super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);

            if(null != value) { // null passiert auf Macs
	            if (value instanceof String name) {
                    // Text
                    setText(name);
                    if(ReferenceHierarchyManager.getInstance(_connection).hasReferenceHierarchie(name)) {
                        ReferenceHierarchy rh = ReferenceHierarchyManager.getInstance(_connection).getReferenceHierarchy(name);
                        // Font und Farbe
                        if(ReferenceHierarchyManager.getInstance(_connection).isChangeable(rh)) {
                            setForeground(Color.BLACK);
                            Font font = getFont();
                            setFont(new Font(font.getName(), Font.PLAIN, font.getSize()));
                        }
                        else {
                            setForeground(Color.GRAY);
                            Font font = getFont();
                            setFont(new Font(font.getName(), Font.ITALIC, font.getSize()));
                        }
                        // Tooltipp
                        setToolTipText(rh.getInfo() + " (Geometrie-Typ: " + rh.getGeometryType() + ")");
                    }
                    else { // sollte nicht passieren
                        setForeground(Color.RED);
                        Font font = getFont();
                        setFont(new Font(font.getName(), Font.BOLD, font.getSize()));
                    }
                }
                else { // sollte nicht passieren
                    setText(value.toString());
                    setForeground(Color.RED);
                    Font font = getFont();
                    setFont(new Font(font.getName(), Font.BOLD, font.getSize()));
                }
            }
            return this;
        }
    }

    /**
     * Macht den EOR-Hierarchien-Verwaltungsdialog sichtbar.
     */
    public void showDialog(boolean visible) {
        _frame.setState(Frame.NORMAL);
        _frame.setVisible(visible);
    }

    /**
     * Beendet den EOR-Hierarchien-Verwaltungsdialog.
     */
    private void closeDialog() {
        _frame.setVisible(false);
        _frame.storePreferenceBounds();
        _frame.dispose();
    }

    @Override
    public String toString() {
        return "RhmDialog";
    }

}
