/*
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.gnd;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.kappich.pat.gnd.layerManagement.Layer;
import de.kappich.pat.gnd.layerManagement.LayerDefinitionDialog;
import de.kappich.pat.gnd.layerManagement.LayerManager;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectType;
import de.kappich.pat.gnd.viewManagement.NoticeLayer;
import de.kappich.pat.gnd.viewManagement.View;
import de.kappich.pat.gnd.viewManagement.ViewDialog;
import de.kappich.pat.gnd.viewManagement.ViewEntry;
import de.kappich.pat.gnd.viewManagement.ViewManager;
import de.kappich.pat.gnd.viewManagement.ViewManagerDialog;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.HashMap;
import java.util.List;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JTree;
import javax.swing.SwingUtilities;
import javax.swing.ToolTipManager;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;

/**
 * Die Legende der Kartendarstellung.
 * <p>
 * Das LegendPane steht für den Legendenbaum in der Kartendarstellung der GND. Es ist als JTree realisiert und besitzt die von DefaultTreeModel
 * abgeleitete Klasse LegendTreeModel, die aus den LegendTreeNodes der Layer bei jeder Änderungkomplett neu zusammengebaut werden. Dies geschieht im
 * Konstruktor von LegendTreeModel. Ein Update wird durch eine Änderung einer Ansicht oder des Anzeigemaßstabs augelöst.
 * <p>
 * Beim Neuaufbau des Legendenbaums bleiben alle nicht-expandierten Knoten in diesem Zustand, während alle anderen, also insbesondere neu hinzugefügte
 * Knoten, expandiert werden.
 *
 * @author Kappich Systemberatung
 */
@SuppressWarnings("serial")
public class LegendPane extends JTree implements View.ViewChangeListener, MapPane.MapScaleListener {

    private final ClientDavInterface _connection;
    private final GenericNetDisplay _genericNetDisplay;
    private final View _view;
    private Double _mapScale = 0.;
    private TreeModel _treeModel;

    /**
     * Konstruiert ein Objekt aus der übergebenen Ansicht, wobei allerdings die Initialisierung noch ausbleibt (s. {@link #init}).
     *
     * @param view eine Ansicht
     */
    public LegendPane(final ClientDavInterface connection, final GenericNetDisplay genericNetDisplay, final View view) {
        _connection = connection;
        _genericNetDisplay = genericNetDisplay;
        _view = view;
    }

    /**
     * Initialisiert das Objekt.
     *
     * @param mapScale der Maßstab für die Kartendarstellung
     */
    public void init(final Double mapScale) {
        _mapScale = mapScale;

        setEditable(false);
        getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
        _view.addChangeListener(this);
        _treeModel = new LegendTreeModel();
        setModel(_treeModel);
        expandAll();
        addMouseListener(new MyMouseListener());
        setCellRenderer(new LegendCellRenderer());
        ToolTipManager.sharedInstance().registerComponent(this);
    }

    @Override
    public void viewEntriesSwitched(View view, int i, int j) {
        recreateLegendWithRespectToExpansion();
    }

    @Override
    public void viewEntryDefinitionChanged(View view, int i) {
        recreateLegendWithRespectToExpansion();
    }

    @Override
    public void viewEntryPropertyChanged(View view, int i) {
        recreateLegendWithRespectToExpansion();
    }

    @Override
    public void viewEntryRemoved(View view, int i) {
        recreateLegendWithRespectToExpansion();
    }

    @Override
    public void viewEntryInserted(View view, final int newIndex) {
        recreateLegendWithRespectToExpansion();
    }

    private void recreateLegendWithRespectToExpansion() {
        HashMap<String, Boolean> expansionMap = new HashMap<>(getRowCount());
        for (int i = 0; i < getRowCount(); i++) {
            TreePath path = getPathForRow(i);
            final Object lastPathComponent = path.getLastPathComponent();
	        if (lastPathComponent instanceof LegendTreeNodes.LegendTreeNode node) {
                expansionMap.put(node.getNameOrText(), isExpanded(path) || node.isLeaf());
            }
        }
        LegendTreeModel newTreeModel = new LegendTreeModel();
        setModel(newTreeModel);
        for (int i = 0; i < getRowCount(); i++) {
            TreePath path = getPathForRow(i);
            final Object lastPathComponent = path.getLastPathComponent();
	        if (lastPathComponent instanceof LegendTreeNodes.LegendTreeNode node) {
                final String nameOrText = node.getNameOrText();
                // neue Layer expandiert
                setExpandedState(path, expansionMap.getOrDefault(nameOrText, true));
            }
        }
    }

    private void expandAll() {
        for (int row = 0; row < getRowCount(); row++) {
            expandRow(row);
        }
    }

    /**
     * Setzt den Maßstabsfaktor.
     *
     * @param mapScale den neue Maßstabsfaktor
     */
    public void setMapScale(double mapScale) {
        _mapScale = mapScale;
    }

    /*
     * Die Implementation von MapPane.MapScaleListener
     */
    @Override
    public void mapScaleChanged(double scale) {
        _mapScale = scale;
        recreateLegendWithRespectToExpansion();
    }

    @Override
    public String toString() {
        return "LegendPane{" + "_view=" + _view + ", _mapScale=" + _mapScale + ", _treeModel=" + _treeModel + '}';
    }

    /**
     * Der LegendCellRenderer legt fest wie die Teile der Legende angezeigt werden.
     *
     * @author Kappich Systemberatung
     */
    public static class LegendCellRenderer extends DefaultTreeCellRenderer {
        /**
         * Der Konstruktor.
         */
        public LegendCellRenderer() {
            // Versichern, dass der Hintergrund gezeichnet wird
            setOpaque(true);
            setPreferredSize(new Dimension(300, 20));
        }

        @Override
        public Component getTreeCellRendererComponent(JTree tree, Object value, boolean isSelected, boolean expanded, boolean leaf, int row,
                                                      boolean hasFocusArg) {

            // Die Originalmethode die Standardeinstellungen machen lassen
            super.getTreeCellRendererComponent(tree, value, isSelected, expanded, leaf, row, hasFocusArg);

            boolean colorsSet = false;
	        if (value instanceof LegendTreeNodes.TextTreeNode node) {
                setText(node.getNameOrText());
                Font font = getFont();
                if (!isSelected) {
                    setFont(new Font(font.getName(), node.getStyle(), font.getSize()));
                } else {
                    setFont(new Font(font.getName(), Font.BOLD, font.getSize()));
                }
                setForeground(node.getColor());
                setBackground(tree.getBackground());
                colorsSet = true;
	        } else if (value instanceof LegendTreeNodes.IconTreeNode node) {
                JLabel newLabel = new JLabel(node.getIcon());
                newLabel.setToolTipText(node.getInfo());
                return newLabel;
	        } else if (value instanceof LegendTreeNodes.RootNode node) {
                final View view = (View) node.getUserObject();
                setText(view.getName());
                Font font = getFont();
                if (!isSelected) {
                    setFont(new Font(font.getName(), Font.PLAIN, font.getSize()));
                } else {
                    setFont(new Font(font.getName(), Font.BOLD, font.getSize()));
                }
	        } else if (value instanceof LegendTreeNodes.LegendTreeNode node) { // Oberklasse natürlich zuletzt.
                setText(node.getNameOrText());
                setToolTipText(node.getInfo());
                Font font = getFont();
                if (!isSelected) {
                    setFont(new Font(font.getName(), Font.PLAIN, font.getSize()));
                } else {
                    setFont(new Font(font.getName(), Font.BOLD, font.getSize()));
                }
            }

            // Juli/2017: Das ist alter Kode, dessen Bedeutung unklar ist. GND-intern gibt es keine
            // DefaultMutableTreeNode deren userObject vom Typ Color ist.
            if (null != value && !colorsSet) {
                DefaultMutableTreeNode node = (DefaultMutableTreeNode) value;
                Object inside = node.getUserObject();
                // Wenn der Knoten eine Farbe ist: den Hintergrund entsprechend setzen
                // Andernfalls soll der Hintergrund zum Baum passen.
                if (inside instanceof Color) {
                    setBackground((Color) inside);
                    setForeground(Color.BLACK);
                } else {
                    setBackground(tree.getBackground());
                    setForeground(tree.getForeground());
                }
            }
            return this;
        }
    }

    /**
     * Diese Klasse unterscheidet sich von einem DefaultTreeModel allein durch den Konstruktor, in dem die Daten aus den Membern von LegendPane
     * ermittelt werden.
     *
     * @author Kappich Systemberatung
     */
    public class LegendTreeModel extends DefaultTreeModel {
        /**
         * Konstruiert ein TreeModel aus den Daten von LegendPane.
         */
        public LegendTreeModel() {
            super(new LegendTreeNodes.RootNode(_view.getName(), null, _view));
            DefaultMutableTreeNode ourRoot = (DefaultMutableTreeNode) getRoot();
            for (ViewEntry viewEntry : _view.getViewEntriesWithoutNoticeEntries()) {
                if (viewEntry.isVisible(_mapScale.intValue())) {
                    final Layer layer = viewEntry.getLayer();
                    DefaultMutableTreeNode currentNode = new LegendTreeNodes.LegendTreeNode(layer.getName(), layer.getInfo(), viewEntry);
                    ourRoot.add(currentNode);
                    final DisplayObjectType displayObjectType = layer.getDisplayObjectType(_mapScale.intValue());
                    if (displayObjectType == null) {
                        continue;
                    }
                    LegendTreeNodes nodes = displayObjectType.getLegendTreeNodes();
                    for (LegendTreeNodes.LegendTreeNode newNode : nodes.getOrderedNodes()) {
                        currentNode.add(newNode);
                        Integer levelChange = nodes.getLevelChange(newNode);
                        if (levelChange < 0) {
                            currentNode = newNode;
                        }
                        while (levelChange > 0) {
                            currentNode = (DefaultMutableTreeNode) currentNode.getParent();
                            levelChange--;
                        }
                    }
                }
            }
        }
    }

    private class MyMouseListener extends MouseAdapter {

        @Override
        public void mousePressed(final MouseEvent e) {
            if (!SwingUtilities.isRightMouseButton(e)) {
                return;
            }
            Object source = e.getSource();
            TreePath selPath = getPathForLocation(e.getX(), e.getY());
	        if (source instanceof JTree tree) {
                tree.setSelectionPath(selPath);
            }
            if (null != selPath) {
                Object selObject = selPath.getLastPathComponent();
	            if (selObject instanceof LegendTreeNodes.LegendTreeNode node) {
                    final Object userObject = node.getUserObject();
                    if (userObject instanceof ViewEntry) {
                        showViewEntryPopup(e, (ViewEntry) userObject);
                    } else if (userObject instanceof View) {
                        showViewPopup(e, (View) userObject);
                    }
//					else if(userObject instanceof DisplayObjectType) {
//						showDOTPopup(e, (DisplayObjectType) userObject);
//					}
                }
            }
        }

        private void moveToPosition(int currIndex, int newIndex) {
            if (currIndex < newIndex) {
                for (int i = currIndex; i < newIndex; ++i) {
                    _view.switchTableLines(i, i + 1);
                }
            } else if (currIndex > newIndex) {
                for (int i = currIndex; i > newIndex; --i) {
                    _view.switchTableLines(i - 1, i);
                }
            }
        }

        @SuppressWarnings("OverlyLongMethod")
        private void showViewEntryPopup(MouseEvent e, ViewEntry viewEntry) {
            final Layer layer = viewEntry.getLayer();
            final boolean changeable = LayerManager.getInstance().isChangeable(layer);
            JPopupMenu popupMenu = new JPopupMenu();
            JMenuItem editMenuItem;
            if (changeable) {
                editMenuItem = new JMenuItem("Einstellungen bearbeiten");
            } else {
                editMenuItem = new JMenuItem("Einstellungen betrachten");
            }
            editMenuItem.addActionListener(e1 -> {
                final String title;
                if (changeable) {
                    title = "GND: einen Layer bearbeiten";
                } else {
                    title = "GND: einen Layer betrachten";
                }
                LayerDefinitionDialog definitionDialog = new LayerDefinitionDialog(_connection, layer, changeable, false, title);
                definitionDialog.setVisible(true);
            });
            popupMenu.add(editMenuItem);

            JMenuItem removeFromViewItem = new JMenuItem("Aus der Ansicht entfernen");
            removeFromViewItem.addActionListener(e12 -> {
                _view.removeLayer(layer.getName());
                if (ViewManager.getInstance().isChangeable(_view)) {
                    _view.putPreferences(ViewManager.getPreferenceStartPath());
                }
            });
            popupMenu.add(removeFromViewItem);

            JMenuItem hideLayerButton = new JMenuItem("Zeitweilig ausblenden");
            hideLayerButton.addActionListener(e13 -> viewEntry.setVisible(false));
            popupMenu.add(hideLayerButton);

            popupMenu.addSeparator();

            final JCheckBoxMenuItem selectableMenuItem = new JCheckBoxMenuItem("Selektierbar", viewEntry.isSelectable());
            selectableMenuItem.addActionListener(e14 -> viewEntry.setSelectable(selectableMenuItem.getState()));
            popupMenu.add(selectableMenuItem);

            popupMenu.addSeparator();

            List<ViewEntry> viewEntryList = _view.getViewEntriesWithoutNoticeEntries();

            int index = _view.getIndex(viewEntry);
            if (index > 0) {
                int[] visisblePredecessors = new int[index];
                int counter = 0;
                for (int i = index - 1; i >= 0; --i) {
                    if (viewEntryList.get(i).isVisible(_mapScale.intValue())) {
                        visisblePredecessors[counter] = i;
                        ++counter;
                    }
                }
                if (counter > 2) {
                    JMenuItem threeUpItem = new JMenuItem("Aufwärts (3x)");
                    threeUpItem.addActionListener(e15 -> {
                        moveToPosition(index, visisblePredecessors[2]);
                    });
                    popupMenu.add(threeUpItem);
                }
                if (counter > 1) {
                    JMenuItem twoUpItem = new JMenuItem("Aufwärts (2x)");
                    twoUpItem.addActionListener(e16 -> {
                        moveToPosition(index, visisblePredecessors[1]);
                    });
                    popupMenu.add(twoUpItem);
                }
                if (counter > 0) {
                    JMenuItem oneUpItem = new JMenuItem("Aufwärts");
                    oneUpItem.addActionListener(e17 -> {
                        moveToPosition(index, visisblePredecessors[0]);
                    });
                    popupMenu.add(oneUpItem);
                }
            }

            int numberOfViewEntries = viewEntryList.size();
            if (index + 1 < numberOfViewEntries) {
                int[] visisbleSuccessors = new int[numberOfViewEntries - index - 1];
                int counter = 0;
                for (int i = index + 1; i < numberOfViewEntries; ++i) {
                    if (viewEntryList.get(i).isVisible(_mapScale.intValue())) {
                        visisbleSuccessors[counter] = i;
                        ++counter;
                    }
                }
                if (counter > 0) {
                    JMenuItem oneDownItem = new JMenuItem("Abwärts");
                    oneDownItem.addActionListener(e18 -> {
                        moveToPosition(index, visisbleSuccessors[0]);
                    });
                    popupMenu.add(oneDownItem);
                }
                if (counter > 1) {
                    JMenuItem twoDownItem = new JMenuItem("Abwärts (2x)");
                    twoDownItem.addActionListener(e19 -> {
                        moveToPosition(index, visisbleSuccessors[1]);
                    });
                    popupMenu.add(twoDownItem);
                }
                if (counter > 2) {
                    JMenuItem threeDownItem = new JMenuItem("Abwärts (3x)");
                    threeDownItem.addActionListener(e18 -> {
                        moveToPosition(index, visisbleSuccessors[2]);
                    });
                    popupMenu.add(threeDownItem);
                }
            }

            popupMenu.show(LegendPane.this, e.getX(), e.getY());
        }

        private void showViewPopup(MouseEvent e, View view) {
            final boolean changeable = ViewManager.getInstance().isChangeable(view);
            JPopupMenu popupMenu = new JPopupMenu();
            JMenuItem editMenuItem = new JMenuItem("Einstellungen bearbeiten");
            editMenuItem.addActionListener(e1 -> {
                final String title = "GND: aktuelle Ansicht bearbeiten";
                ViewDialog.runDialog(ViewManager.getInstance(), view, true, changeable, false, title);
            });
            popupMenu.add(editMenuItem);

            JMenuItem saveItem = new JMenuItem("Einstellungen speichern");
            saveItem.setEnabled(changeable);
            saveItem.addActionListener(e13 -> {
                view.putPreferences(ViewManager.getPreferenceStartPath());
            });
            popupMenu.add(saveItem);

            boolean addSeparator = true;
            for (ViewEntry entry : view.getAllViewEntries()) {
                if (entry.getLayer() instanceof NoticeLayer) {
                    continue;
                }
                if (!entry.isVisible()) {
                    JMenuItem makeVisibleItem = new JMenuItem("Layer '" + entry.getLayer().getName() + "' einblenden");
                    makeVisibleItem.addActionListener(e12 -> {
                        entry.setVisible(true);
                    });
                    if (addSeparator) {
                        popupMenu.addSeparator();
                        addSeparator = false;
                    }
                    popupMenu.add(makeVisibleItem);
                }
            }

            popupMenu.addSeparator();

            JMenuItem changeViewItem = new JMenuItem("Ansicht wechseln");
            changeViewItem.addActionListener(e14 -> {
                ViewManagerDialog viewManagerDialog = new ViewManagerDialog(_genericNetDisplay, false);
                viewManagerDialog.showDialog();
            });
            popupMenu.add(changeViewItem);

            popupMenu.show(LegendPane.this, e.getX(), e.getY());
        }
    }

}
