/*
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.gnd;

import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.kappich.pat.gnd.viewManagement.View;
import java.awt.Color;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.swing.Icon;
import javax.swing.tree.DefaultMutableTreeNode;

/**
 * Eine Klasse für eine Liste von Objekten in der Legende der Kartendarstellung. Die einzelnen Knoten in dieser Liste haben Informationen über die
 * Tiefe innerhalb der Legendenbaumes, die jeweils relativ zum Vorgänger berechnet ist. Der allgemeineren Verständnis: jeder Darstellungstyp muss die
 * Methode getLegendTreeNodes implementieren, die ein Objekt von LegendTreeNodes zurückliefert, das gerade für den Teilbaum der Legende, den dieser
 * Darstellungstyp festlegt, steht.
 *
 * @author Kappich Systemberatung
 */
public class LegendTreeNodes {

    private final List<LegendTreeNode> _legendTreeNodes = new ArrayList<>();
    private final Map<LegendTreeNode, Integer> _levelChanges = new HashMap<>();

    /**
     * Ein LegendTreeNodes-Objekt verwaltet die Knoten des Legendenbaums als Liste. Jeder solche Knoten ist ein LegendTreeNode-Objekt mit der
     * Information, in welchem Umfang sich der Level des aktuellen Knoten von dem seines Vorgängers unterscheidet. Dabei bedeutet eine 0, dass sich
     * der Knoten auf demselben Level wie sein Vorgänger befindet, während -1 bedeutet, dass er einen Level tiefer angeordnet wird, und eine Positive
     * Zahl i bedeutet, dass der Knoten i Level höher anzuordnen ist.
     */
    @SuppressWarnings("RedundantNoArgConstructor")
    public LegendTreeNodes() {
    }

    /**
     * Fügt einen neuen Knoten mit dem übergebenen Level-Änderung relativ zum Vorgänger hinzu.
     *
     * @param node        der neue Knoten
     * @param levelChange die Level-Änderung
     */
    public void add(LegendTreeNode node, Integer levelChange) {
        _legendTreeNodes.add(node);
        _levelChanges.put(node, levelChange);
    }

    /**
     * Gibt die Liste der Knoten zurück.
     *
     * @return die Liste aller Knoten
     */
    public List<LegendTreeNode> getOrderedNodes() {
        return Collections.unmodifiableList(_legendTreeNodes);
    }

    /**
     * Gibt für übergebenen Knoten die Leveländerung relativ zu seinem Vorgänger zurück.
     *
     * @param node ein Knoten
     *
     * @return die Leveländerung relativ zu seinem Vorgänger
     */
    public Integer getLevelChange(LegendTreeNode node) {
        return _levelChanges.get(node);
    }

    /**
     * Gibt an, ob
     *
     * @return
     */
    public boolean isEmpty() {
        return _legendTreeNodes.isEmpty();
    }

    @Override
    public String toString() {
        return "LegendTreeNodes{" + "_legendTreeNodes=" + _legendTreeNodes + ", _levelChanges=" + _levelChanges + '}';
    }

    /**
     * Eine Klasse für einzelne Objekte in der Legende der Kartendarstellung.
     * <p>
     * Ein LegendTreeNode verkörpert einen Knoten im Legendenbaum. Jeder Knoten hat einen Namen, d.i. der Text, der in der Legende angezeigt wird, und
     * einen Infotext, der als Tooltipp verwendet wird.
     *
     * @author Kappich Systemberatung
     */
    @SuppressWarnings("serial")
    public static class LegendTreeNode extends DefaultMutableTreeNode {

        private final String _nameOrText;
        private final String _info;

        /**
         * Konstruiert ein Objekt aus den gegebenen Informationen. Das userObject wird an den DefaultMutableTreeNode durchgereicht.
         *
         * @param name ein Name
         * @param info eine Info
         */
        public LegendTreeNode(String name, @Nullable String info, @Nullable Object userObjectArg) {
            super(name);
            super.setUserObject(userObjectArg);
            _nameOrText = name;
            _info = info;
        }

        /**
         * Gibt den Namen oder Text des Knoten zurück.
         *
         * @return den Namen des Knoten
         */
        public String getNameOrText() {
            return _nameOrText;
        }

        /**
         * Gibt den Infotext des Knotens zurück.
         *
         * @return den Infotext
         */
        @Nullable
        public String getInfo() {
            return _info;
        }
    }

    public static class RootNode extends LegendTreeNode {

        public RootNode(String text, String info, final View view) {
            super(text, info, view);
        }
    }

    public static class TextTreeNode extends LegendTreeNode {

        private final Color _color;
        private final int _style;

        public TextTreeNode(String text, Color color, int style, String info, @Nullable Object userObjectArg) {
            super(text, info, userObjectArg);
            _color = color;
            _style = style;
        }

        public Color getColor() {
            return _color;
        }

        public int getStyle() {
            return _style;
        }
    }
    // Die Logik ist hier wie folgt zu verstehen:
    // Im der Liste werden die Nodes gemäß ihrer Reihenfolge aufgelistet. In der Map wird die Änderung
    // des Levls vermerkt: eine -1 oder jede andere negative Zahl bedeutet, dass nach diesem Node
    // ein neuer Children-Level unterhalb des jetzigen Nodes beginnt. Eine 0 bedeutet, dass sich der
    // Level und damit auch der Parent-Node für den nächsten Node nicht verändert.
    // Eine positive Zahl n bedeutet, dass der Parent-Node des nächsten Nodes der n-te Parent des
    // aktuellen Nodes ist.
    // Beispiel:
    // Root-Node: (existiert)
    //		Layer1: 		 		-1
    //			PrimitiveForm1:		-1
    //				Property1:		0
    //				Property2:		0
    //				Property3:		+1
    //			PrimitiveForm2:		-1
    //				Property4:		0
    //				Property5:		+2
    //		Layer2:					-1
    //			PrimitiveForm6:		+1 (zurück zu Root!)

    @SuppressWarnings("NonSerializableFieldInSerializableClass")
    public static class IconTreeNode extends LegendTreeNode {
        private final Icon _icon;

        public IconTreeNode(String text, String info, Icon icon, @Nullable Object userObjectArg) {
            super(text, info, userObjectArg);
            _icon = icon;
        }

        public Icon getIcon() {
            return _icon;
        }
    }
}
