/*
 * Copyright 2010-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.gnd;

import de.bsvrz.sys.funclib.debug.Debug;
import de.kappich.pat.gnd.areaPlugin.DOTAreaPlugin;
import de.kappich.pat.gnd.asbNodePlugin.DOTAsbNodePlugin;
import de.kappich.pat.gnd.complexPlugin.DOTComplexPlugin;
import de.kappich.pat.gnd.csvPlugin.DOTCsvPlugin;
import de.kappich.pat.gnd.displayObjectToolkit.DOTManager;
import de.kappich.pat.gnd.elrPlugin.DOTElrPlugin;
import de.kappich.pat.gnd.kmPlugin.DOTKmPlugin;
import de.kappich.pat.gnd.linePlugin.DOTLinePlugin;
import de.kappich.pat.gnd.needlePlugin.DOTNeedlePlugin;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectTypePlugin;
import de.kappich.pat.gnd.pointPlugin.DOTPointPlugin;
import de.kappich.pat.gnd.rnPlugin.DOTRnPlugin;
import de.kappich.pat.gnd.statPlugin.DOTStatPlugin;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Vector;

/**
 * Diese Klasse dient zur Verwaltung externer Plugins.
 *
 * @author Kappich Systemberatung
 */
public final class PluginManager {

    /* Hier werden die externen Plugins gespeichert. Jeder Name muss der vollständige Klassenname
     * einer Klasse sein, die das Interface DisplayObjectTypePlugin implementiert. */
    private static final Set<String> _pluginNames = new HashSet<>();
    /* In dieser Map sind die Schlüssel die Rückgabewerte von DisplayObjectTypePlugin.getName() für externe
     * Implementationen von DisplayObjectTypePlugin und die Werte sind die Implementationen selber. */
    private static final Map<String, DisplayObjectTypePlugin> _pluginsMap = new HashMap<>();

    private static final Debug _debug = Debug.getLogger();

    private PluginManager() {
    }

    /**
     * Diese Methode macht die externen Plugins bekannt, indem die vollständigen Namen der Klassen, die {@link DisplayObjectTypePlugin}
     * implementieren, übergeben werden. Sie muss vor dem ersten Zugriff auf Teile dieser Plugins aufgerufen werden; der beste Moment dafür ist, bevor
     * der erste Konstruktor von {@link GenericNetDisplay} aufgerufen wird, denn sonst könnte schon die Initialisierung aus den Präferenzen scheitern;
     * man beachte, dass GenericNetDisplay eine {@link GenericNetDisplay#addPlugins gleichnamige} und ebenfalls statische Methode anbietet, die die
     * Arbeit an diese Methode delegiert.
     *
     * @param plugins die hinzuzufügenden externen Plugins
     */
    public static void addPlugins(final List<String> plugins) {
        if (plugins == null || (plugins.isEmpty())) {
            return;
        }
        final List<String> newPlugins = new ArrayList<>();
        for (String pluginName : plugins) {
            if (_pluginNames.contains(pluginName)) {
                continue;
            }
            final Class<?> pluginClass;
            try {
                pluginClass = Class.forName(pluginName);
            } catch (ClassNotFoundException ignore) {
                _debug.error("Fehler im PluginManager: die Klasse '" + pluginName + "' kann nicht instanziiert werden.");
                continue;
            }
            Class<?> dotPluginClass;
            try {
                dotPluginClass = Class.forName("de.kappich.pat.gnd.pluginInterfaces.DisplayObjectTypePlugin");
            } catch (ClassNotFoundException e) {
                _debug.error("Schwerer interner Fehler - ClassNotFoundException" + System.lineSeparator() + e.getMessage());
                throw new UnsupportedOperationException("Schwerer interner Fehler - ClassNotFoundException", e);
            }
            final Class<?>[] interfaces = pluginClass.getInterfaces();
            boolean interfaceFound = false;
            for (Class<?> iface : interfaces) {
                if (iface.equals(dotPluginClass)) {
                    interfaceFound = true;
                    break;
                }
            }
            if (interfaceFound) {
                final Object pluginObject;
                try {
                    pluginObject = pluginClass.newInstance();
                } catch (InstantiationException ignore) {
                    _debug.error("Fehler im PluginManager: es kann kein Objekt der Klasse '" + pluginName + "' instanziiert werden.");
                    continue;
                } catch (IllegalAccessException ignore) {
                    _debug.error("Fehler im PluginManager: es kann nicht auf ein Objekt der Klasse '" + pluginName + "' zugegriffen werden.");
                    continue;
                }
                final DisplayObjectTypePlugin displayObjectTypePlugin = (DisplayObjectTypePlugin) pluginObject;
                final String name = displayObjectTypePlugin.getName();
                if (name.equals("Fläche") || name.equals("Komplex") || name.equals("Linie") || name.equals("Punkt")) {
                    _debug.error("Fehler im PluginManager: ein Plugin mit dem Namen '" + name + "' kann nicht hinzugefüht werden.");
                } else if (_pluginsMap.containsKey(name)) {
                    _debug.error("Fehler im PluginManager: ein Plugin mit dem Namen '" + name + "' wurde bereits hinzugefügt.");
                } else {
                    _pluginNames.add(pluginName);
                    _pluginsMap.put(name, displayObjectTypePlugin);
                    newPlugins.add(pluginName);
                }
            } else {
                _debug.error("Fehler im PluginManager: die Klasse '" + pluginName + "' implementiert nicht das Interface DisplayObjectTypePlugin.");
            }
        }

        /* Das Folgende wird nicht über einen Listener-Mechanismus implementiert, weil sonst
         * sichergestellt werden müsste, dass der/die Listener bereits registriert sind. */
        DOTManager.pluginsAdded(newPlugins);
    }

    /**
     * Gibt die Namen aller Plugins zurück.
     *
     * @return alle Plugin-Namen
     */
    @SuppressWarnings("UseOfObsoleteCollectionType")
    public static Vector<String> getAllPluginNames(final boolean withInternalPlugins, final boolean withNeedlePlugin, final boolean withCsvAndEor) {
        final Vector<String> names = new Vector<>();
        names.add("Fläche");
        names.add("Komplex");
        names.add("Linie");
        names.add("Punkt");
        if (withCsvAndEor) {
            names.add("Erweiterte Ortsreferenzen");
            names.add("CSV");
        }
        if (withInternalPlugins) {
            names.add("ASB-Knotennummern");
            names.add("ASB-Stationierung");
            names.add("Autobahnschilder");
            names.add("Betriebskilometrierung");
        }
        if (withNeedlePlugin) {
            names.add("Nadel");
        }
        names.addAll(_pluginsMap.keySet());
        return names;
    }

    /**
     * Gibt die Namen aller geometrie-bezogenen Plugins zurück.
     *
     * @return Plugin-Namen
     */
    @SuppressWarnings("UseOfObsoleteCollectionType")
    public static Vector<String> getGeometryPluginNames() {
        final Vector<String> names = new Vector<>();
        names.add("Fläche");
        names.add("Komplex");
        names.add("Linie");
        names.add("Punkt");
        return names;
    }

    /**
     * Gibt das DisplayObjectTypePlugin-Objekt zurück, dessen getName-Implementation den übergebenen Namen zurückgibt, und {@code null}, wenn kein
     * solches Objekt existiert.
     *
     * @param name ein Plugin-Name
     *
     * @return das Plugin oder {@code null}
     */
    public static DisplayObjectTypePlugin getPlugin(final String name) {
	    return switch (name) {
            // die ersten drei sind interne Plugins, die nicht bei getAllPluginNames nach außen gegeben werden
		    case "ASB-Knotennummern" -> new DOTAsbNodePlugin();
		    case "ASB-Stationierung" -> new DOTStatPlugin();
		    case "Autobahnschilder" -> new DOTRnPlugin();
		    case "Betriebskilometrierung" -> new DOTKmPlugin();
		    case "CSV" -> new DOTCsvPlugin();
		    case "Nadel" -> new DOTNeedlePlugin();
            // die nächsten vier sind die internen Plugins, die nach außen gegeben werden
		    case "Fläche" -> new DOTAreaPlugin();
		    case "Komplex" -> new DOTComplexPlugin();
		    case "Linie" -> new DOTLinePlugin();
		    case "Punkt" -> new DOTPointPlugin();
		    case "Erweiterte Ortsreferenzen" -> new DOTElrPlugin();
            // und schließlich die externen Plugins
		    default -> _pluginsMap.get(name);
	    };
    }

    /**
     * Ist ein Plugin des übergebenen Namens bekannt.
     *
     * @param name der Name
     *
     * @return {@code true}, falls ja
     */
    public static boolean hasPlugin(final String name) {
	    return switch (name) {
		    case "Autobahnschilder", "ASB-Knotennummern", "ASB-Stationierung", "Betriebskilometrierung", "CSV", "Nadel",
		         "Fläche", "Komplex", "Linie", "Punkt", "Erweiterte Ortsreferenzen" -> true;
		    default -> _pluginsMap.containsKey(name);
	    };
    }
}
