/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.kmPlugin;

import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.kappich.pat.gnd.displayObjectToolkit.DOTCollection;
import de.kappich.pat.gnd.displayObjectToolkit.DisplayObject;
import de.kappich.pat.gnd.displayObjectToolkit.PrimitiveFormPropertyPair;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectPainter;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectType;
import de.kappich.pat.gnd.properties.DistanceRasterType;
import de.kappich.pat.gnd.utils.PointWithAngle;
import java.awt.Rectangle;
import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * {@code KmDisplayObject} ist die Implementation des Interfaces {@link DisplayObject} im Km-Plugin.
 *
 * @author Kappich Systemberatung
 */
public class KmDisplayObject implements DisplayObject {

    private final Map<Integer, Rectangle> _boundingRectangles = new HashMap<>();
    private final DOTCollection _dotCollection;
    private final String _routeNumber;
    private final String _blockNumber;
    private final long _kmValue;
    private final List<Object> _coordinates;
    private DisplayObjectPainter _painter;
    private int _defaultType;

    /**
     * Ein Konstruktor, der nur Kilometrierungs-Informationen enthält. Objekte, dei mit diesem Konstruktor erstellt wurden, sind nur eingesvhränkt
     * verwendbar.
     *
     * @param routeNumber
     * @param blockNumber
     * @param kmValue
     */
    public KmDisplayObject(final String routeNumber, final String blockNumber, final long kmValue) {
        super();
        _painter = null;
        _dotCollection = null;
        _defaultType = 0;
        if (routeNumber == null) {
            _routeNumber = "";
        } else {
            _routeNumber = routeNumber;
        }
        if (blockNumber == null) {
            _blockNumber = "";
        } else {
            _blockNumber = blockNumber;
        }
        _kmValue = kmValue;
        _coordinates = new ArrayList<>();
    }

    /**
     * Ein Konstruktor mit Painter, DOTCollection und Koordinaten. Objekte, die so konstruiert wurden, sind vollständig verwendungsfähig.
     *
     * @param painter       ein Painter, der dieses Object zeichnen kann, etwa ein {@link DOTKmPainter}
     * @param dotCollection die Kollektion, die beim Zeichnen verwendet werden soll
     * @param routeNumber   der Straßenname, zu dem dieses Objekt gehört
     * @param blockNumber   die Blocknummer des Objekts
     * @param kmValue       der Kilometrierungswert
     * @param point         die Koordinateninformation
     */
    public KmDisplayObject(DisplayObjectPainter painter, final DOTCollection dotCollection, final String routeNumber, final String blockNumber,
                           final long kmValue, final PointWithAngle point) {
        super();
        _painter = painter;
        _dotCollection = dotCollection;
        _defaultType = 0;
        if (routeNumber == null) {
            _routeNumber = "";
        } else {
            _routeNumber = routeNumber;
        }
        if (blockNumber == null) {
            _blockNumber = "";
        } else {
            _blockNumber = blockNumber;
        }
        _kmValue = kmValue;
        _coordinates = new ArrayList<>(1);
        _coordinates.add(point);
    }

    @Override
    public String getName() {
        return _routeNumber + ", " + _kmValue + " m  (Blocknummer: " + _blockNumber + ")";
    }

    @Override
    public List<Object> getCoordinates(final int type) {
        return Collections.unmodifiableList(_coordinates);
    }

//	/**
//	 * Gibt das den Text umgebende Rechteck zurück, falls dieses bereits gesetzt wurde.
//	 * Falls aber nicht, so wird ein kleines Default-Rechteck zurückgegeben.
//	 *
//	 * @return das Rechteck
//	 */
//	public Rectangle2D getTextBoundingRectangle() {
//		if ( _textBoundingRectangle==null) {
//			return new Rectangle(0,0,1,1);
//		} else {
//			return _textBoundingRectangle;
//		}
//	}
//
//	/**
//	 * Setzt das den Text umgebende Rechteck.
//	 *
//	 * @param rectangle das neue Rechteck
//	 */
//	public void setTextBoundingRectangle( Rectangle2D rectangle) {
//		_boundingRectangles.clear();
//		_textBoundingRectangle = rectangle;
//	}

    @Override
    public List<Object> getCoordinates() {
        return Collections.unmodifiableList(_coordinates);
    }

    @Override
    public List<Point2D> getReferencePoints(final Rectangle utmBounds) {
        List<Point2D> pointList = new ArrayList<>(1);
        if (!_coordinates.isEmpty()) {
            Point2D point = ((PointWithAngle) _coordinates.get(0)).getPoint();
            if (utmBounds.contains(point)) {
                pointList.add(point);
            }
        }
        return pointList;
    }

    @Override
    public DisplayObjectPainter getPainter() {
        return new DOTKmPainter();
    }

    @Override
    public void setPainter(DisplayObjectPainter painter) {
        _painter = painter;
    }

    @Override
    public DOTCollection getDOTCollection() {
        return _dotCollection;
    }

    @Override
    public Rectangle getBoundingRectangle(final int type) {
        if (!_boundingRectangles.containsKey(type)) {
            _boundingRectangles.put(type, _painter.getBoundingRectangle(this, type));
        }
        return _boundingRectangles.get(type);
    }

    @Override
    public Rectangle getBoundingRectangle() {
        return getBoundingRectangle(_defaultType);
    }

    @Override
    public void setDefaultType(final int defaultType) {
        _defaultType = defaultType;
    }

    @Override
    @Nullable
    public DisplayObjectType.DisplayObjectTypeItem getDisplayObjectTypeItem(final PrimitiveFormPropertyPair pair) {
        return null;
    }

    /*
     * Das ist die Methode, die für den MapPane.MapScaleListener implementiert werden muss. Sie kann hier
     * leer bleiben, da KmDisplayObjects davon keinen Gebrauch machen.
     */
    @Override
    public void mapScaleChanged(double scale) {
    }

    /**
     * Diese Methode prüft, ob {@code this} bei dem übergebenen {@code DistanceRasterType} gezeigt werden kann, d.h. ob der Kilometrierungswert zum
     * Raster gehört.
     *
     * @param distanceRasterType der {@code DistanceRasterType}
     *
     * @return {@code true} genau dann, wenn der Kilometrierungswert zum Raster gehört
     */
    public boolean checkType(@Nullable DistanceRasterType distanceRasterType) {
        return distanceRasterType != null && (_kmValue % distanceRasterType.getValue() == 0);
    }

    /**
     * Gibt den zu den Übergabeparametern gehörigen Text zurück.
     *
     * @param distanceRasterType der {@code DistanceRasterType}
     * @param kmFormat           das {@code KmFormat}
     *
     * @return der Text
     */
    @SuppressWarnings("SuspiciousIndentAfterControlStatement")
    @Nullable
    public String getText(@Nullable final DistanceRasterType distanceRasterType, @Nullable final KmFormat kmFormat) {
        if (distanceRasterType == null || kmFormat == null) {
            return null;
        }
        switch (kmFormat) {
            case RN_AND_KM:
                if (!_routeNumber.isEmpty()) {
                    return _routeNumber + ": " + distanceRasterType.getFormat().format((double) _kmValue / 1000.);
                } else {
                    return distanceRasterType.getFormat().format((double) _kmValue / 1000.);
                }
            case RN_BN_AND_KM:
                if (!_routeNumber.isEmpty() && !_blockNumber.isEmpty()) {
                    return _routeNumber + "/" + _blockNumber + ": " + distanceRasterType.getFormat().format((double) _kmValue / 1000.);
                } else if (!_routeNumber.isEmpty()) {
                    return _routeNumber + ": " + distanceRasterType.getFormat().format((double) _kmValue / 1000.);
                } else {
                    return distanceRasterType.getFormat().format((double) _kmValue / 1000.);
                }
            case RN_ONLY:
                return _routeNumber;
            case KM_ONLY:
            default:
                return distanceRasterType.getFormat().format((double) _kmValue / 1000.);
        }
    }

    @Override
    public String toString() {
        return "KmDisplayObject{" + "_boundingRectangles=" + _boundingRectangles + ", _defaultType=" + _defaultType + ", _painter=" + _painter +
               ", _dotCollection=" + _dotCollection + ", _routeNumber='" + _routeNumber + '\'' + ", _blockNumber='" + _blockNumber + '\'' +
               ", _kmValue=" + _kmValue + ", _coordinates=" + _coordinates + '}';
    }
}
