/*
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */
package de.kappich.pat.gnd.layerManagement;

import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.kappich.pat.gnd.documentation.GndHelp;
import de.kappich.pat.gnd.utils.view.GndFrame;
import de.kappich.pat.gnd.viewManagement.View;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Frame;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.WindowConstants;
import javax.swing.table.DefaultTableCellRenderer;

/**
 * Ein Dialog zum Hinzufügen eines Layers zu einer Ansicht.
 *
 * @author Kappich Systemberatung
 * @version $Revision$
 */
@SuppressWarnings({"serial", "OverlyLongMethod"})
public class AddLayerDialog extends GndFrame {
    private final View _view;

    /**
     * Konstruktor zum Anlegen eines neuen AddLayerDialog
     *
     * @param layerManager Layerverwaltung
     * @param view         die Ansicht, zu der der Layer hinzugefügt werden soll
     */
    public AddLayerDialog(final LayerManager layerManager, View view) {
        super("AddLayerDialog", "GND: Layer zur Ansicht hinzufügen");

        _view = view;

        setLayout(new BorderLayout());

        JButton hinzufuegenButton = new JButton("Layer hinzufügen");

        JButton helpButton = new JButton("Hilfe");

        JPanel panelButtons = new JPanel();
        panelButtons.setLayout(new BoxLayout(panelButtons, BoxLayout.Y_AXIS));
        panelButtons.setBorder(BorderFactory.createEmptyBorder(5, 10, 5, 10));

        Dimension d = new Dimension(15, 15);
        panelButtons.add(Box.createRigidArea(d));

        panelButtons.add(hinzufuegenButton);
        panelButtons.add(Box.createRigidArea(d));

        panelButtons.add(helpButton);
        panelButtons.add(Box.createRigidArea(d));

        final JTable table = new JTable(layerManager) {
            //Implement table cell tool tips.
            @Override
            @Nullable
            public String getToolTipText(@NotNull MouseEvent e) {
                String tip = null;
                java.awt.Point p = e.getPoint();
                int rowIndex = rowAtPoint(p);
                int colIndex = columnAtPoint(p);
                int realColumnIndex = convertColumnIndexToModel(colIndex);
                if(realColumnIndex == 0) {
                    LayerManager lm = (LayerManager) getModel();
                    tip = lm.getTooltipAt(rowIndex);
                }
                return tip;
            }
        };

        table.setPreferredScrollableViewportSize(new Dimension(500, 70));
        table.setFillsViewportHeight(true);

        table.addKeyListener(new KeyAdapter() {
            @Override
            public void keyPressed(KeyEvent e) {
                if(e.getKeyCode() == KeyEvent.VK_F1) {
                    GndHelp.openHelp("#addLayer");
                }
            }
        });

        table.setDefaultRenderer(Object.class, new DefaultTableCellRenderer() {
            @Override
            public Component getTableCellRendererComponent(
                final JTable table, final Object value,
                final boolean isSelected,
                final boolean hasFocus,
                final int row, final int column) {
                Component cell = super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);
                if(null != value) {// nur zur Sicherheit bei künftigen Änderungen: Macs rufen mit null auf.
	                if (value instanceof String layerName) {
                        if(LayerManager.getInstance().isChangeable(LayerManager.getInstance().getLayer(layerName))) {
                            setForeground(Color.BLACK);
                            Font font = getFont();
                            setFont(new Font(font.getName(), Font.PLAIN, font.getSize()));
                        }
                        else {
                            setForeground(Color.GRAY);
                            Font font = getFont();
                            setFont(new Font(font.getName(), Font.ITALIC, font.getSize()));
                        }
                    }
                }
                return cell;
            }
        });

        JButton closeButton = new JButton("Dialog schließen");

        JPanel closeButtonPanel = new JPanel();
        closeButtonPanel.setLayout(new BoxLayout(closeButtonPanel, BoxLayout.Y_AXIS));
        closeButtonPanel.setBorder(BorderFactory.createEmptyBorder(5, 10, 5, 10));
        closeButtonPanel.add(closeButton);


        add(panelButtons, BorderLayout.EAST);
        final JScrollPane comp = new JScrollPane(table);
        add(comp, BorderLayout.CENTER);
        add(closeButtonPanel, BorderLayout.SOUTH);

        ActionListener actionListenerHinzufuegen = e -> {
            int[] selectedRows = table.getSelectedRows();
            if(selectedRows.length == 0) {
                JOptionPane.showMessageDialog(
                    new JFrame(),
                    "Bitte wählen Sie mindestens eine Zeile aus der Liste aus!",
                    "Fehler",
                    JOptionPane.ERROR_MESSAGE
                );
                return;
            }
            setCursor(new Cursor(Cursor.WAIT_CURSOR));
            for(int i : selectedRows) {
                _view.addLayer(layerManager.getLayer(i));
            }
            setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
        };
        hinzufuegenButton.addActionListener(actionListenerHinzufuegen);


        ActionListener actionListenerHelp = e -> GndHelp.openHelp("#addLayer");
        helpButton.addActionListener(actionListenerHelp);

        ActionListener actionListenerClose = e -> {
            storePreferenceBounds();
            dispose();
        };
        closeButton.addActionListener(actionListenerClose);

        class MyWindowListener extends WindowAdapter {
            @Override
            public void windowClosing(WindowEvent e) {
                /* wenn nur noch ein einziges Fenster geöffnet ist
                 * beendet sich das Programm beim Schließen des Fensters */
                final Frame[] frames = Frame.getFrames();
                int length = frames.length - 1;
                for(Frame frame : frames) {
                    if(!frame.isVisible()) {
                        length--;
                    }
                }
                if(length == 0) {
                    setDefaultCloseOperation(WindowConstants.EXIT_ON_CLOSE);
                }
            }
        }
        addWindowListener(new MyWindowListener());

        Dimension dTable = table.getPreferredSize();
        setPositionAndSize(0, 0,
                           660, 50,
                           false,
                           500, Math.min(140 + dTable.height, 664)
        );
        setVisible(true);
    }

    @Override
    public String toString() {
        return "AddLayerDialog{" +
               "_view=" + _view +
               '}';
    }
}
