/*
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2021 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.kappich.pat.gnd.layerManagement;

import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.kappich.pat.gnd.csv.CsvFormat;
import de.kappich.pat.gnd.csv.CsvFormatManager;
import de.kappich.pat.gnd.displayObjectToolkit.DOTCollection;
import de.kappich.pat.gnd.displayObjectToolkit.DOTManager;
import de.kappich.pat.gnd.extLocRef.ReferenceHierarchy;
import de.kappich.pat.gnd.extLocRef.ReferenceHierarchyManager;
import de.kappich.pat.gnd.gnd.PluginManager;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectType;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectTypePlugin;
import java.io.File;
import java.util.Set;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;
import javax.swing.JOptionPane;

/**
 * Ein Klasse, die festgelegt, welche Objekte zu ihr gehören und wie diese dargestellt werden.<p>
 * <p>
 * Ein Layer hat folgende wesentlichen Bestandteile: seinen Namen, der als eindeutige Kennung verwendet wird, sowie einen Infotext und einen
 * Plugin-Namen. Darüberhinaus können die Pid eines SystemObjektTypes, eine EOR-Hierarchie, sowie ein CSV-Format und eine CSV-Datei angegeben sein.
 *
 * @author Kappich Systemberatung
 */
public class Layer implements Comparable<Layer> {

    private static final String INFO = "INFO";
    private static final String PLUGIN_NAME = "PLUGIN_NAME";
    private static final String CONFIGURATION_OBJECT_TYPE = "GEO_REFERENCE_TYP";
    private static final String REFERENCE_HIERARCHY_NAME = "REFERENCE_HIERARCHY";
    private static final String CSV_FORMAT_NAME = "CSV_FORMAT_NAME";
    private static final String CSV_FILE_PATH = "CSV_DATEI";
    private static final String DOT_COLLECTION = "DOT_COLLECTION";

    private String _name;
    private String _info;
    private DisplayObjectTypePlugin _plugin;
    private String _configurationObjectTypePid;
    private ReferenceHierarchy _referenceHierarchy;
    private CsvFormat _csvFormat;
    private File _csvFile;
    private String _csvInitInfo;

    private DOTCollection _dotCollection = new DOTCollection();

    /**
     * Ein Default-Konstruktor, der z.B. für die Initialisierung mit initializeFromPreferences() benötigt wird.
     */
    public Layer() {
        _name = null;
        _info = null;
        _plugin = null;
        _configurationObjectTypePid = null;
        _referenceHierarchy = null;
        _csvFormat = null;
        _csvFile = null;
        _csvInitInfo = null;
    }

    /**
     * Im folgenden Konstruktor werden dem Layer seine wesentlichen Bestandteile mitgegeben, während die DOTCollection über verschiedene Methoden
     * später bearbeitet werden kann.
     *
     * @param layerName               der Name des Layers
     * @param info                    der Infotext zum Layer
     * @param plugin                  der Name eine Attributgruppe oder {@code null}
     * @param configurationObjectType die Pid eines SystemObjectTypes
     */
    public Layer(String layerName, @Nullable String info, DisplayObjectTypePlugin plugin, @Nullable String configurationObjectType,
                 @Nullable ReferenceHierarchy referenceHierarchy, @Nullable CsvFormat csvFormat, @Nullable File csvFile) {
        _name = layerName;
        _info = info;
        _plugin = plugin;
        _configurationObjectTypePid = configurationObjectType;
        _referenceHierarchy = referenceHierarchy;
        _csvFormat = csvFormat;
        _csvFile = csvFile;
    }

    /**
     * Der Getter für den Namen.
     *
     * @return den Namen
     */
    public String getName() {
        return _name;
    }

    /**
     * Der Setter für den Namen.
     *
     * @param layerName der Name des Layers
     */
    public void setName(final String layerName) {
        _name = layerName;
    }

    /**
     * Der Getter für den Infotext.
     *
     * @return der Infotext des Layers
     */
    public String getInfo() {
        return _info;
    }

    /**
     * Der Setter für den Infotext.
     *
     * @param info der Infotext
     */
    public void setInfo(String info) {
        _info = info;
    }

    /**
     * Der Getter für das Plugin des Layers.
     *
     * @return das Plugin
     */
    public DisplayObjectTypePlugin getPlugin() {
        return _plugin;
    }

    /**
     * Der Setter für das Plugin.
     *
     * @param plugin das neue Plugin
     */
    public void setPlugin(DisplayObjectTypePlugin plugin) {
        _plugin = plugin;
    }

    /**
     * Der Getter für den Namen der Klasse, deren Objekte der Layer darstellt.
     *
     * @return der Name der Klasse
     */
    public String getConfigurationObjectType() {
        return _configurationObjectTypePid;
    }

    /**
     * Der Setter für den Namen der Klasse, deren Objekte der Layer darstellt.
     *
     * @param configurationObjectType der Name der Klasse
     */
    public void setConfigurationObjectType(String configurationObjectType) {
        _configurationObjectTypePid = configurationObjectType;
    }

    /**
     * Gibt die {@link ReferenceHierarchy EOR-Hierarchie} des Layers zurück.
     *
     * @return die EOR-Hierarchie
     */
    @Nullable
    public ReferenceHierarchy getReferenceHierarchy() {
        return _referenceHierarchy;
    }

    /**
     * Setzt die {@link ReferenceHierarchy EOR-Hierarchie} des Layers.
     *
     * @param hierarchy der neue Wert; kann {@code null} sein
     */
    public void setReferenceHierarchy(@Nullable final ReferenceHierarchy hierarchy) {
        _referenceHierarchy = hierarchy;
    }

    /**
     * Gibt das {@link CsvFormat} des Layers zurück.
     *
     * @return ein CsvFormat oder {@code null}
     */
    @Nullable
    public CsvFormat getCsvFormat() {
        return _csvFormat;
    }

    /**
     * Setzt die {@link CsvFormat} des Layers.
     *
     * @param csvFormat der neue Wert; kann {@code null} sein
     */
    public void setCsvFormat(@Nullable final CsvFormat csvFormat) {
        _csvFormat = csvFormat;
    }

    /**
     * Gibt die {@link File CSV-Datei} des Layers zurück.
     *
     * @return eine CSV-Datei oder {@code null}
     */
    @Nullable
    public File getCsvFile() {
        return _csvFile;
    }

    /**
     * Setzt die {@link File CSV-Datei} des Layers.
     *
     * @param csvFile CSV-Datei der neue Wert; kann {@code null} sein
     */
    public void setCsvFile(@Nullable final File csvFile) {
        _csvFile = csvFile;
    }

    /**
     * Gibt die CSV-Initialisierungs-Information zurück.
     *
     * @return CSV-Initialisierungs-Information
     */
    @Nullable
    public String getCsvInitInfo() {
        return _csvInitInfo;
    }

    /**
     * Setzt die CSV-Initialisierungs-Information, aber nur für CSV-Layer möglich.
     *
     * @param info die Info
     */
    public void setCsvInitInfo(String info) {
        if (_csvFile != null) {
            _csvInitInfo = info;
        } else {
            throw new IllegalAccessError("setCsvInitInfo darf nur für CSV-Layer aufgerufen werden.");
        }
    }

    /**
     * Mit dieser Methode kann man der DOTCollection des Layers einen Darstellungstypen hinzufügen, und muss dabei das Intervall für das er gelten
     * soll, angeben. Der Darstellungstyp muss von {@code null} verschieden sein, und lowerScale muss mindestens so groß wie upperScale sein (es
     * handelt sich um die Xe der 1:X-Werte von Maßstäben).
     */
    public void addDisplayObjectType(DisplayObjectType type, int lowerScale, int upperScale) {
        _dotCollection.addDisplayObjectType(type, lowerScale, upperScale);
    }

    /**
     * Durch Aufruf dieser Methode wird die DOTCollection des Layers geleert.
     */
    @SuppressWarnings("unused")
    public void clearDisplayObjectTypes() {
        _dotCollection.clear();
    }

    /**
     * Gibt die DOTCollection des Layers zurück.
     *
     * @return die DOTCollection des Layers
     */
    public DOTCollection getDotCollection() {
        return _dotCollection;
    }

    /**
     * Setzt die DOTCollection des Layers.
     *
     * @param dotCollection die neue DOTCollection des Layers
     */
    public void setDotCollection(DOTCollection dotCollection) {
        _dotCollection = dotCollection;
    }

    /**
     * Gibt einen Darstellungstypen für den übergebenen 1:X-Maßstabswert zurück, falls es in der DOTCollection einen solchen für diesen Wert gibt.
     * Gibt es mehr als einen, so ist nicht festgelegt, welchen man erhält. Gibt es keinen, so erhält man null als Rückgabewert.
     *
     * @param scale ein Maßstabswert
     *
     * @return ein zugehöriger Darstellungstyp
     */
    @Nullable
    public DisplayObjectType getDisplayObjectType(int scale) {
        return _dotCollection.getDisplayObjectType(scale);
    }

    /**
     * Speichert die Präferenzen des Layers unter dem übergebenen Knoten.
     *
     * @param prefs der Knoten, unter dem die Präferenzen gespeichert werden
     */
    public void putPreferences(Preferences prefs) {
        deletePreferences(prefs);
        Preferences objectPrefs = prefs.node(getName());
        if (_info == null) {
            objectPrefs.put(INFO, "");
        } else {
            objectPrefs.put(INFO, _info);
        }
        objectPrefs.put(PLUGIN_NAME, _plugin.getName());
        if (null != _referenceHierarchy) {
            objectPrefs.put(REFERENCE_HIERARCHY_NAME, _referenceHierarchy.getName());
        }
        if (null != _configurationObjectTypePid) {
            objectPrefs.put(CONFIGURATION_OBJECT_TYPE, _configurationObjectTypePid);
        }
        if (null != _csvFormat) {
            objectPrefs.put(CSV_FORMAT_NAME, _csvFormat.getName());
        }
        if (null != _csvFile) {
            objectPrefs.put(CSV_FILE_PATH, _csvFile.getAbsolutePath());
        }
        Preferences collectionPrefs = objectPrefs.node(DOT_COLLECTION);
        _dotCollection.putPreferences(collectionPrefs);
    }

    /**
     * Löscht die Präferenzen des Layers unter dem Knoten.
     *
     * @param prefs der Knoten, unter dem die Präferenzen gelöscht werden
     */
    public void deletePreferences(Preferences prefs) {
        Preferences objectPrefs = prefs.node(getName());
        try {
            objectPrefs.removeNode();
        } catch (BackingStoreException e) {
            JOptionPane
                .showMessageDialog(null, "Das Löschen des Layers war nicht erfolgreich." + e.toString(), "Fehlermeldung", JOptionPane.ERROR_MESSAGE);
        }
    }

    /**
     * Initialisiert den Layer aus den Präferenzen unter dem übergebenen Knoten.
     *
     * @param prefs der Knoten, unter dem die Präferenzen gesucht werden
     *
     * @return gibt {@code true} zurück, wenn die Initialisierung erfolgreich war, und {@code false}, falls nicht
     */
    public boolean initializeFromPreferences(Preferences prefs) {
        _name = prefs.name();
        _info = prefs.get(INFO, "");
        _plugin = PluginManager.getPlugin(prefs.get(PLUGIN_NAME, ""));
        _configurationObjectTypePid = prefs.get(CONFIGURATION_OBJECT_TYPE, "");
        String referenceHierachyName = prefs.get(REFERENCE_HIERARCHY_NAME, "");
        if (!referenceHierachyName.isEmpty()) {
            _referenceHierarchy = ReferenceHierarchyManager.getInstance().getReferenceHierarchy(referenceHierachyName);
        } else {
            _referenceHierarchy = null;
        }
        String csvFormatName = prefs.get(CSV_FORMAT_NAME, "");
        if (!csvFormatName.isEmpty()) {
            _csvFormat = CsvFormatManager.getInstance().getCsvFormat(csvFormatName);
        } else {
            _csvFormat = null;
        }
        String csvFileName = prefs.get(CSV_FILE_PATH, "");
        if (!csvFileName.isEmpty()) {
            _csvFile = new File(csvFileName);
        } else {
            _csvFile = null;
        }
        Preferences collectionPrefs = prefs.node(DOT_COLLECTION);
        return _dotCollection.initializeFromPreferences(collectionPrefs, DOTManager.getInstance());
    }

    /**
     * Gibt die Menge aller Namen aller Farben, die von den Darstellungstypen in der DOTCollection des Layers verwendet werden, zurück.
     *
     * @return die Namen aller benutzten Farben
     */
    public Set<String> getUsedColors() {
        return _dotCollection.getUsedColors();
    }

    @Override
    public String toString() {
        if (_plugin != null) {
            return "[Layer: " + _name + ", Info: " + _info + ", Plugin: " + _plugin.getName() + ", SystemObjektTyp: " + _configurationObjectTypePid +
                   ", EOR-Hierarchie: " + _referenceHierarchy + ", CSV-Format: " + _csvFormat + ", CSV-Datei: " + _csvFile + ", DOTCollection: " +
                   _dotCollection.toString() + "]";
        } else {
            return "[Layer: " + _name + ", Info: " + _info + ", SystemObjektTyp: " + _configurationObjectTypePid + ", EOR-Hierarchie: " +
                   _referenceHierarchy + ", CSV-Format: " + _csvFormat + ", CSV-Datei: " + _csvFile + ", DOTCollection: " +
                   _dotCollection.toString() + "]";
        }
    }

    /**
     * Diese Methode gibt eine Kopie des Layers zurück.
     *
     * @return die Kopie
     */
    public Layer getCopy() {
        final Layer copy = new Layer(_name, _info, _plugin, _configurationObjectTypePid, _referenceHierarchy, _csvFormat, _csvFile);
        copy.setDotCollection(_dotCollection.getCopy());
        return copy;
    }

    @Override
    public int compareTo(final Layer o) {
        return getName().toLowerCase().compareTo(o.getName().toLowerCase());
    }

    @Override
    public boolean equals(Object o) {
	    if (!(o instanceof Layer other)) {
            return false;
        }
        return getName().toLowerCase().equals(other.getName().toLowerCase());
    }

    @Override
    public int hashCode() {
        return getName().toLowerCase().hashCode();
    }
}
