/*
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2021 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.kappich.pat.gnd.layerManagement;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.dav.daf.main.impl.config.DafConfigurationObjectType;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.bsvrz.sys.funclib.kappich.filechooser.AwtFileChooser;
import de.kappich.pat.gnd.csv.CsvFormat;
import de.kappich.pat.gnd.csv.CsvFormatManager;
import de.kappich.pat.gnd.displayObjectToolkit.DOTCollection;
import de.kappich.pat.gnd.displayObjectToolkit.DOTManager;
import de.kappich.pat.gnd.displayObjectToolkit.DOTSubscriptionData;
import de.kappich.pat.gnd.documentation.GndHelp;
import de.kappich.pat.gnd.elrPlugin.DOTElrPlugin;
import de.kappich.pat.gnd.extLocRef.ReferenceHierarchy;
import de.kappich.pat.gnd.extLocRef.ReferenceHierarchyManager;
import de.kappich.pat.gnd.gnd.PluginManager;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectType;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectTypePlugin;
import de.kappich.pat.gnd.utils.SpringUtilities;
import de.kappich.pat.gnd.utils.view.GndFrame;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.text.Collator;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import javax.swing.BorderFactory;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.SpringLayout;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.TableModelListener;
import javax.swing.filechooser.FileNameExtensionFilter;
import javax.swing.table.TableColumn;
import javax.swing.table.TableModel;

/**
 * Der Dialog zur Definition und Bearbeitung von Layern.
 *
 * @author Kappich Systemberatung
 */
@SuppressWarnings({"serial", "OverlyLongMethod"})
public class LayerDefinitionDialog extends GndFrame
    implements ReferenceHierarchyManager.RhmChangeListener, CsvFormatManager.CsvFormatManagerChangeListener {

    private static final Debug _debug = Debug.getLogger();
    private static final int _lowerHeightBound = 400;
    private static final int _upperHeightBound = Toolkit.getDefaultToolkit().getScreenSize().height;
    private final DataModel _configuration;
    private final JTextField _nameTextField = new JTextField();
    private final JTextField _infoTextField = new JTextField();
    private final JComboBox<String> _pluginComboBox;
    private final JComboBox<Object> _configurationObjectTypesComboBox;
    private final JComboBox<String> _referenceHierarchyComboBox;
    private final JComboBox<String> _csvFormatComboBox;
    private final JTextField _csvFileTextField;
    private final JButton _openCsvFileButton;
    private final JTable _dotTable = new JTable();
    private final List<EditableListenButton> _listeningButtons = new ArrayList<>();
    private final EditableListenButton _newDOTButton = new EditableListenButton("Neue Zeile");
    private final JButton _deleteDOTButton = new JButton("Zeile löschen");
    private Layer _scratchLayer;
    private Layer _unchangeableOriginalLayer;
    private boolean _editable;
    private boolean _nameChangeable;
    private boolean _somethingChanged;

    /**
     * Konstruktor zum Anlegen eines neuen Layereditors.
     *
     * @param connection     die Datenverteiler-Verbindung
     * @param layer          ein Layer
     * @param editable       ist der Layer verändebar?
     * @param nameChangeable ist der Name und damit die Identität änderbar?
     * @param title          der Titel des Fensters
     */
    public LayerDefinitionDialog(final ClientDavInterface connection, final Layer layer, final boolean editable, final boolean nameChangeable,
                                 final String title) {
        super("LayerDefinitionDialog", title);
        _configuration = connection.getDataModel();
        if (layer != null) {
            _scratchLayer = layer.getCopy();
            _unchangeableOriginalLayer = _scratchLayer.getCopy();
        } else {
            String err = "Ein LayerDefinitionDialog kann nicht mit einem Null-Layer konstruiert werden.";
            _debug.error(err);
            throw new IllegalArgumentException(err);
        }
        _editable = editable;
        _nameChangeable = nameChangeable;

        setLayout(new BorderLayout());

        Dimension labelSize = new Dimension(100, 20);

        // Oberer Teil mit Name, Info usw.

        JLabel nameLabel = new JLabel("Name: ");
        nameLabel.setPreferredSize(labelSize);
        _nameTextField.setText(_scratchLayer.getName());
        _nameTextField.setEditable(nameChangeable);

        JLabel infoLabel = new JLabel("Info: ");
        infoLabel.setPreferredSize(labelSize);
        _infoTextField.setText(_scratchLayer.getInfo());

        JLabel pluginLabel = new JLabel("Plugin: ");
        pluginLabel.setPreferredSize(labelSize);
        _pluginComboBox = new JComboBox<>(PluginManager.getAllPluginNames(true, false, true));
        _pluginComboBox.setPreferredSize(new Dimension(200, 25));

        JLabel typeLabel = new JLabel("Typ: ");
        typeLabel.setPreferredSize(labelSize);
        _configurationObjectTypesComboBox = new JComboBox<>(getConfigurationObjectTypes());
        _configurationObjectTypesComboBox.setPreferredSize(new Dimension(200, 25));

        JLabel hierarchyLabel = new JLabel("EOR-Hierachie: ");
        _referenceHierarchyComboBox =
            new JComboBox<>(new DefaultComboBoxModel<>(ReferenceHierarchyManager.getInstance(connection).getReferenceHierarchyNames(false)));
        _referenceHierarchyComboBox.setSelectedIndex(-1);
        _referenceHierarchyComboBox.setEnabled(false);
        _referenceHierarchyComboBox.setPreferredSize(new Dimension(200, 25));

        JLabel csvFormatLabel = new JLabel("CSV-Format: ");
        _csvFormatComboBox = new JComboBox<>(CsvFormatManager.getInstance().getCsvFormatNames());

        JLabel csvFileLabel = new JLabel("CSV-Datei: ");
        _csvFileTextField = new JTextField();
        _openCsvFileButton = new JButton("Auswählen");
        JPanel csvFilePanel = new JPanel();
        csvFilePanel.setLayout(new SpringLayout());
        csvFilePanel.add(_csvFileTextField);
        csvFilePanel.add(_openCsvFileButton);
        SpringUtilities.makeCompactGrid(csvFilePanel, 2, 5, 5);

        updateComboBoxes();

        JPanel upperPanel = new JPanel();
        upperPanel.setLayout(new SpringLayout());
        upperPanel.add(nameLabel);
        upperPanel.add(_nameTextField);
        upperPanel.add(infoLabel);
        upperPanel.add(_infoTextField);
        upperPanel.add(pluginLabel);
        upperPanel.add(_pluginComboBox);
        upperPanel.add(typeLabel);
        upperPanel.add(_configurationObjectTypesComboBox);
        upperPanel.add(hierarchyLabel);
        upperPanel.add(_referenceHierarchyComboBox);
        upperPanel.add(csvFormatLabel);
        upperPanel.add(_csvFormatComboBox);
        upperPanel.add(csvFileLabel);
        upperPanel.add(csvFilePanel);

        upperPanel.setBorder(BorderFactory.createEmptyBorder(5, 10, 5, 10));
        SpringUtilities.makeCompactGrid(upperPanel, 2, 5, 5);
        add(upperPanel, BorderLayout.NORTH);

        initComponents();
        addPluginListener();

        // Mittelteil für die DOTs

        setTableProperties(_scratchLayer.getDotCollection());
        addListSelectionListener();

        addDOTButtonListener();

        JPanel dotButtonsPanel = new JPanel();
        dotButtonsPanel.setLayout(new SpringLayout());

        dotButtonsPanel.add(_newDOTButton);
        dotButtonsPanel.add(_deleteDOTButton);

        dotButtonsPanel.setBorder(BorderFactory.createEmptyBorder(5, 10, 5, 10));
        SpringUtilities.makeCompactGrid(dotButtonsPanel, 1, 5, 20);

        JPanel centerPanel = new JPanel();
        centerPanel.setLayout(new SpringLayout());
        centerPanel.setBorder(BorderFactory.createMatteBorder(2, 2, 2, 2, Color.BLACK));
        centerPanel.add(new JScrollPane(_dotTable));
        centerPanel.add(dotButtonsPanel);
        SpringUtilities.makeCompactGrid(centerPanel, 2, 20, 5);

        add(centerPanel, BorderLayout.CENTER);

        // Untere Teil mit Buttons

        EditableListenButton saveButton = new EditableListenButton("Layer speichern");
        JButton cancelButton = new JButton("Dialog schließen");
        JButton helpButton = new JButton("Hilfe");
        addButtonListener(saveButton, cancelButton, helpButton);

        JPanel buttonsPanel = new JPanel();
        buttonsPanel.setLayout(new SpringLayout());

        buttonsPanel.add(saveButton);
        buttonsPanel.add(cancelButton);
        buttonsPanel.add(helpButton);

        buttonsPanel.setBorder(BorderFactory.createEmptyBorder(5, 10, 5, 10));
        SpringUtilities.makeCompactGrid(buttonsPanel, 3, 20, 5);
        add(buttonsPanel, BorderLayout.SOUTH);

        addChangeListeners();
        addFrameListener();
        ReferenceHierarchyManager.getInstance(connection).addRhmChangeListener(this);
        CsvFormatManager.getInstance().addChangeListener(this);

        setEditable(_editable, _nameChangeable);
        setPositionAndSize();
        setVisible(true);
        toFront();
    }

    private static List<SystemObjectType> getConfigurationObjectTypes(SystemObjectType systemObjectType) {
        return systemObjectType.getSubTypes();
    }

    public final void setPositionAndSize() {
        Dimension d = _dotTable.getPreferredSize();
        setPositionAndSize(690, 202 + (int) d.getHeight(), 660, 50, false, 690,
                           Math.min(Math.max(202 + (int) d.getHeight(), _lowerHeightBound), _upperHeightBound));
    }

    private void addChangeListeners() {

        _nameTextField.getDocument().addDocumentListener(new DocumentListener() {
            @Override
            public void changedUpdate(DocumentEvent e) {
                _somethingChanged = true;
            }

            @Override
            public void removeUpdate(DocumentEvent e) {
                _somethingChanged = true;
            }

            @Override
            public void insertUpdate(DocumentEvent e) {
                _somethingChanged = true;
            }
        });

        _infoTextField.getDocument().addDocumentListener(new DocumentListener() {
            @Override
            public void changedUpdate(DocumentEvent e) {
                _somethingChanged = true;
            }

            @Override
            public void removeUpdate(DocumentEvent e) {
                _somethingChanged = true;
            }

            @Override
            public void insertUpdate(DocumentEvent e) {
                _somethingChanged = true;
            }
        });

        ActionListener systemObjectTypeListener = e -> _somethingChanged = true;
        _configurationObjectTypesComboBox.addActionListener(systemObjectTypeListener);

        TableModelListener tableModelListener = e -> _somethingChanged = true;
        _dotTable.getModel().addTableModelListener(tableModelListener);
    }

    private Object[] getConfigurationObjectTypes() {
        final SystemObjectType startType = _configuration.getType("typ.konfigurationsObjekt");
        final List<SystemObjectType> configurationObjectTypes = new ArrayList<>();
        configurationObjectTypes.addAll(startType.getSubTypes());
        for (int i = 0; i < configurationObjectTypes.size(); i++) {
            configurationObjectTypes.addAll(getConfigurationObjectTypes(configurationObjectTypes.get(i)));
        }
        // configurationObjectTypes enthält manche Einträge zweimal (Mehrfachvererbung!! Deshalb jetzt
        // alles eindeutig machen und schön sortieren.
        final Set<SystemObjectType> systemObjectTypeSet = new HashSet<>();
        systemObjectTypeSet.addAll(configurationObjectTypes);
        Object[] array = systemObjectTypeSet.toArray();
        Collator collator = Collator.getInstance(Locale.GERMAN);
        collator.setStrength(Collator.SECONDARY);// a == A, a < Ä
        Arrays.sort(array, (o1, o2) -> {
            SystemObjectType t1 = (SystemObjectType) o1;
            SystemObjectType t2 = (SystemObjectType) o2;
            return collator.compare(t1.getNameOrPidOrId(), t2.getNameOrPidOrId());
        });
        return array;
    }

    private void setTableProperties(DOTCollection dotCollection) {
        _dotTable.setModel(dotCollection);
        _dotTable.setRowHeight(25);
        //_dotTable.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
        int vColIndex = 0;
        TableColumn col = _dotTable.getColumnModel().getColumn(vColIndex);
        col.setPreferredWidth(327);
        _deleteDOTButton.setEnabled(false);
    }

    /**
     * Setzt die Felder des Layereditors mit den Informationen des übergebenen Layers und aktiviert die Veränderbarkeit gemäß der zwei boolschen
     * Werte.
     *
     * @param layer          ein Layer
     * @param editable       ist der Layer veränderbar?
     * @param nameChangeable ist der Name und damit die Identität des Layers änderbar?
     */
    public void setLayer(Layer layer, boolean editable, boolean nameChangeable) {
        _scratchLayer = layer;
        _unchangeableOriginalLayer = _scratchLayer.getCopy();
        _editable = editable;
        _nameTextField.setText(_scratchLayer.getName());
        _infoTextField.setText(_scratchLayer.getInfo());
        setTableProperties(_scratchLayer.getDotCollection());
        updateComboBoxes();
        setEditable(editable, nameChangeable);
        for (EditableListenButton elButton : _listeningButtons) {
            elButton.layersEditingStateChanged(_editable);
        }
        _somethingChanged = false;
    }

    /**
     * Setzt den Wert der internen Variable, die darüber entscheidet, ob die Informationen des angezeigten Layers veränderbar sind, und macht
     * Textfelder veränderbar oder nicht, aktiviert bzw. deaktiviert Knöpfe usw.
     *
     * @param editable       ist der Layer veränderbar?
     * @param nameChangeable ist der Name und damit die Identität des Layers änderbar?
     */
    public final void setEditable(boolean editable, boolean nameChangeable) {
        _editable = editable;
        _nameChangeable = nameChangeable;
        for (EditableListenButton elButton : _listeningButtons) {
            elButton.layersEditingStateChanged(_editable);
        }
        _nameTextField.setEditable(_nameChangeable);
        _infoTextField.setEditable(_editable);
        _configurationObjectTypesComboBox.setEnabled(_editable);
        _pluginComboBox.setEnabled(_editable);
        _dotTable.setRowSelectionAllowed(_editable);
        _newDOTButton.setEnabled(_editable);
    }

    private void updateComboBoxes() {
        final String systemObjectType = _scratchLayer.getConfigurationObjectType();
        if (systemObjectType != null && !systemObjectType.isEmpty()) {
            final SystemObject object = _configuration.getObject(systemObjectType);
            if (object != null) {
                _configurationObjectTypesComboBox.setSelectedItem(object);
            }
        }
        if (_scratchLayer != null) {
            if (_scratchLayer.getPlugin() != null) {
                _pluginComboBox.setSelectedItem(_scratchLayer.getPlugin().getName());
            } else {
                _pluginComboBox.setSelectedIndex(-1);
            }
            if (_scratchLayer.getReferenceHierarchy() != null) {
                _referenceHierarchyComboBox.setSelectedItem(_scratchLayer.getReferenceHierarchy().getName());
                _referenceHierarchyComboBox.setEnabled(true);
            } else {
                _referenceHierarchyComboBox.setSelectedIndex(-1);
                _referenceHierarchyComboBox.setEnabled(false);
            }
            if (_scratchLayer.getPlugin() != null && _scratchLayer.getPlugin().getName().equals("CSV")) {
                _csvFormatComboBox.setSelectedItem(_scratchLayer.getCsvFormat());
                _csvFormatComboBox.setEnabled(true);
                if (_scratchLayer.getCsvFile() != null) {
                    _csvFileTextField.setText(_scratchLayer.getCsvFile().getAbsolutePath());
                }
                _csvFileTextField.setEnabled(true);
                _openCsvFileButton.setEnabled(true);
            } else {
                _csvFormatComboBox.setSelectedIndex(-1);
                _csvFormatComboBox.setEnabled(false);
                _csvFileTextField.setText("");
                _csvFileTextField.setEnabled(false);
                _openCsvFileButton.setEnabled(false);
            }
        }
    }

    private void updateComponents() {
        String s = (String) _pluginComboBox.getSelectedItem();
        if (null == s || s.isEmpty()) {
            _configurationObjectTypesComboBox.setEnabled(true);
            _referenceHierarchyComboBox.setEnabled(false);
            _referenceHierarchyComboBox.setSelectedIndex(-1);
            _csvFormatComboBox.setEnabled(false);
            _csvFormatComboBox.setSelectedIndex(-1);
            _csvFileTextField.setEnabled(false);
            _csvFileTextField.setText("");
            _openCsvFileButton.setEnabled(false);
        } else {
            switch (s) {
                case "Erweiterte Ortsreferenzen":
                    _configurationObjectTypesComboBox.setEnabled(true);
                    _referenceHierarchyComboBox.setEnabled(true);
                    _csvFormatComboBox.setEnabled(false);
                    _csvFormatComboBox.setSelectedIndex(-1);
                    _csvFileTextField.setEnabled(false);
                    _csvFileTextField.setText("");
                    _openCsvFileButton.setEnabled(false);
                    break;
                case "CSV":
                    _configurationObjectTypesComboBox.setEnabled(false);
                    _configurationObjectTypesComboBox.setSelectedIndex(-1);
                    _referenceHierarchyComboBox.setEnabled(false);
                    _referenceHierarchyComboBox.setSelectedIndex(-1);
                    _csvFormatComboBox.setEnabled(true);
                    _csvFileTextField.setEnabled(true);
                    _openCsvFileButton.setEnabled(true);
                    break;
                case "ASB-Knotennummern":
                case "ASB-Stationierung":
                case "Betriebskilometrierung":
                    _configurationObjectTypesComboBox.setEnabled(false);
                    _configurationObjectTypesComboBox.setSelectedIndex(-1);
                    _referenceHierarchyComboBox.setEnabled(false);
                    _referenceHierarchyComboBox.setSelectedIndex(-1);
                    _csvFormatComboBox.setEnabled(false);
                    _csvFormatComboBox.setSelectedIndex(-1);
                    _csvFileTextField.setEnabled(false);
                    _csvFileTextField.setText("");
                    _openCsvFileButton.setEnabled(false);
                    break;
                default:
                    _configurationObjectTypesComboBox.setEnabled(true);
                    _referenceHierarchyComboBox.setEnabled(false);
                    _referenceHierarchyComboBox.setSelectedIndex(-1);
                    _csvFormatComboBox.setEnabled(false);
                    _csvFormatComboBox.setSelectedIndex(-1);
                    _csvFileTextField.setEnabled(false);
                    _csvFileTextField.setText("");
                    _openCsvFileButton.setEnabled(false);
                    break;
            }
        }
    }

    private void initComponents() {
        updateComponents();

        _openCsvFileButton.addActionListener(e -> {
            final JFileChooser fileChooser;
            if (System.getProperty("os.name").toLowerCase().startsWith("mac")) {
                fileChooser = AwtFileChooser.createFileChooser();
            } else {
                fileChooser = new JFileChooser();
            }
            fileChooser.setFileFilter(new FileNameExtensionFilter("CSV-Datei", "csv", "CSV"));
            int returnVal = fileChooser.showOpenDialog(LayerDefinitionDialog.this.getFrame());
            if (returnVal == JFileChooser.APPROVE_OPTION) {
                File file = fileChooser.getSelectedFile();
                _csvFileTextField.setText(file.getAbsolutePath());
            }
        });
    }

    private void addPluginListener() {
        _pluginComboBox.addActionListener(e -> updateComponents());
    }

    @SuppressWarnings("OverlyLongMethod")
    private void addDOTButtonListener() {

        final class DOTDialog extends JDialog {

            private final JComboBox<Object> _comboBox;

            private final JTextField _fromTextField;

            private final JTextField _toTextField;

            private DOTDialog(final DisplayObjectTypePlugin plugin, @Nullable final ReferenceHierarchy referenceHierarchy) {
                super(LayerDefinitionDialog.this.getFrame(), true);

                JLabel dotLabel = new JLabel("Darstellungstyp: ");
                Object[] array = DOTManager.getInstance().getDOTNames();

                DOTElrPlugin elrPlugin = new DOTElrPlugin();
                DisplayObjectType pluginsDot = null;
                if (!elrPlugin.getName().equals(plugin.getName())) {
                    pluginsDot = plugin.getDisplayObjectType();
                } else if (null != referenceHierarchy) {
                    // Für das EOR-Plugin wird der Geometrie-Typ der EOR-Hierarchy herangezogen:
                    String geometryType = referenceHierarchy.getGeometryType();
                    DisplayObjectTypePlugin geometryPlugin = PluginManager.getPlugin(geometryType);
                    pluginsDot = geometryPlugin.getDisplayObjectType();
                }
                List<String> allowedDOTNames = new ArrayList<>();
                for (Object o : array) {
                    String dotName = (String) o;
                    DisplayObjectType dot = DOTManager.getInstance().getDisplayObjectType(dotName);
                    if (dot.getClass().isInstance(pluginsDot)) {
                        allowedDOTNames.add(dotName);
                    }
                }
                allowedDOTNames.sort(Comparator.comparing(String::toLowerCase));
                _comboBox = new JComboBox<>(allowedDOTNames.toArray());

                JLabel fromLabel = new JLabel("Von 1: ");
                _fromTextField = new JTextField();
                JLabel toLabel = new JLabel("Bis 1: ");
                _toTextField = new JTextField();

                int selectedRow = _dotTable.getSelectedRow();
                if (selectedRow == -1) {
                    if (_dotTable.getModel().getRowCount() > 0) {
                        selectedRow = 0;
                    }
                }
                if (selectedRow > -1) {
                    final DOTCollection dotCollection = _scratchLayer.getDotCollection();
                    _comboBox.setSelectedItem(dotCollection.getValueAt(selectedRow, 0));
                    final Integer fromValueAt = (Integer) dotCollection.getValueAt(selectedRow, 1);
                    if (null != fromValueAt) {
                        _fromTextField.setText(fromValueAt.toString());
                    }
                    final Integer toValueAt = (Integer) dotCollection.getValueAt(selectedRow, 2);
                    if (null != toValueAt) {
                        _toTextField.setText(toValueAt.toString());
                    }
                } else {
                    _fromTextField.setText("2147483647");
                    _toTextField.setText("1");
                }

                JPanel panel = new JPanel();
                panel.setLayout(new SpringLayout());
                panel.add(dotLabel);
                panel.add(_comboBox);
                panel.add(fromLabel);
                panel.add(_fromTextField);
                panel.add(toLabel);
                panel.add(_toTextField);
                SpringUtilities.makeCompactGrid(panel, 2, 20, 5);

                EditableListenButton saveButton = new EditableListenButton("Daten übernehmen");
                JButton cancelButton = new JButton("Dialog schließen");

                JPanel buttonsPanel = new JPanel();
                buttonsPanel.setLayout(new SpringLayout());

                buttonsPanel.add(saveButton);
                buttonsPanel.add(cancelButton);

                buttonsPanel.setBorder(BorderFactory.createEmptyBorder(5, 10, 5, 10));
                SpringUtilities.makeCompactGrid(buttonsPanel, 2, 20, 5);
                addButtonListeners(saveButton, cancelButton);
                addDialogListener();

                setTitle("GND: Darstellungsfestlegung eines Layers");
                setLayout(new BorderLayout());
                add(panel, BorderLayout.NORTH);
                add(buttonsPanel, BorderLayout.SOUTH);

                pack();
                setLocationRelativeTo(LayerDefinitionDialog.this.getFrame());
            }

            private void runDialog() {
                setVisible(true);
            }

            private void addButtonListeners(JButton saveButton, JButton cancelButton) {
                ActionListener actionListenerSave = e -> {
                    final Object item = _comboBox.getSelectedItem();
                    if (item != null && (item instanceof String)) {
                        DisplayObjectType type = DOTManager.getInstance().getDisplayObjectType((String) item);
	                    int lowerScale = Integer.parseInt(_fromTextField.getText());
	                    int upperScale = Integer.parseInt(_toTextField.getText());
                        if (lowerScale < upperScale) {
                            JOptionPane.showMessageDialog(new JFrame(), "Der Von-Wert darf nicht kleiner als der Bis-Wert sein!", "Fehler",
                                                          JOptionPane.ERROR_MESSAGE);
                            return;
                        }
                        _scratchLayer.getDotCollection().addDisplayObjectType(type, lowerScale, upperScale);
                    }
                };
                saveButton.addActionListener(actionListenerSave);

                ActionListener actionListenerCancel = e -> {
                    storePreferenceBounds();
                    dispose();
                };
                cancelButton.addActionListener(actionListenerCancel);
            }

            private void addDialogListener() {
                class DialogListener extends WindowAdapter {
                    @Override
                    public void windowClosing(WindowEvent e) {
                        storePreferenceBounds();
                    }
                }
                addWindowListener(new DialogListener());
            }

            @Override
            public String toString() {
                return "DOTDialog{}";
            }
        }

        ActionListener actionListenerNewDOT = e -> {
            String pluginName = (String) _pluginComboBox.getSelectedItem();
            if (null == pluginName || pluginName.isEmpty()) {
                JOptionPane.showMessageDialog(new JFrame(), "Bitte wählen Sie ein Plugin aus!", "Fehler", JOptionPane.ERROR_MESSAGE);
                return;
            }
            DOTElrPlugin elrPlugin = new DOTElrPlugin();
            String referenceHierarchyName = (String) _referenceHierarchyComboBox.getSelectedItem();
            ReferenceHierarchy referenceHierarchy = null;
            if (pluginName.equals(elrPlugin.getName())) {
                if (null == referenceHierarchyName || referenceHierarchyName.isEmpty()) {
                    JOptionPane.showMessageDialog(new JFrame(), "Bitte wählen Sie eine EOR-Hierarchie aus!", "Fehler", JOptionPane.ERROR_MESSAGE);
                    return;
                }
                referenceHierarchy = ReferenceHierarchyManager.getInstance().getReferenceHierarchy(referenceHierarchyName);
                if (null == referenceHierarchy) {  // sollte nicht passieren können
                    JOptionPane.showMessageDialog(new JFrame(), "Die gewählte EOR-Hierarchie konnte nicht gefunden werden!", "Fehler",
                                                  JOptionPane.ERROR_MESSAGE);
                    return;
                }
            }
            DisplayObjectTypePlugin plugin = PluginManager.getPlugin(pluginName);
            if (null == plugin) {  // sollte nicht passieren können
                JOptionPane.showMessageDialog(new JFrame(), "Das gewählte Plugin konnte nicht gefunden werden!", "Fehler", JOptionPane.ERROR_MESSAGE);
                return;
            }
            DOTDialog dotDialog = new DOTDialog(plugin, referenceHierarchy);
            dotDialog.runDialog();
        };
        _newDOTButton.addActionListener(actionListenerNewDOT);

        ActionListener actionListenerDeleteDOT = e -> {
            final int[] selectedRows = _dotTable.getSelectedRows();
            if (selectedRows.length == 0) {
                JOptionPane.showMessageDialog(new JFrame(), "Bitte wählen Sie mindestens eine Zeile aus der Liste aus!", "Fehler",
                                              JOptionPane.ERROR_MESSAGE);
                return;
            }
            for (int i = selectedRows.length - 1; i >= 0; i--) {
                final int selectedRow = selectedRows[i];
                final TableModel model = _dotTable.getModel();
                DOTCollection dotCollection = (DOTCollection) model;
                final String dotName = (String) dotCollection.getValueAt(selectedRow, 0);
                if (null != dotName) {
                    DisplayObjectType type = DOTManager.getInstance().getDisplayObjectType(dotName);
                    if (type != null) {
                        Object lowerValue = dotCollection.getValueAt(selectedRow, 1);
                        Object upperValue = dotCollection.getValueAt(selectedRow, 2);
                        if (null != lowerValue && null != upperValue) {
                            _scratchLayer.getDotCollection().removeDisplayObjectType(type, (int) lowerValue, (int) upperValue);
                        }
                    }
                }
            }
        };
        _deleteDOTButton.addActionListener(actionListenerDeleteDOT);
    }

    private void addButtonListener(final JButton saveButton, final JButton cancelButton, final JButton helpButton) {

        ActionListener actionListenerSave = e -> {
            DOTCollection collection = (DOTCollection) _dotTable.getModel();
            if (collection.areIntervalsDisjoint()) {
                saveLayer();
            } else {
                JOptionPane.showMessageDialog(this.getFrame(), "Die Intervalle sind nicht überschneidungsfrei!", "Fehler", JOptionPane.ERROR_MESSAGE);
            }
        };
        saveButton.addActionListener(actionListenerSave);

        ActionListener actionListenerCancel = e -> {
            if (_editable && _somethingChanged) {
                if (!saveChangesWanted()) {    // Es wird nicht gespeichert: Änderungen rückgängig machen!
                    setLayer(_unchangeableOriginalLayer.getCopy(), _editable, _nameChangeable);
                }
            }
            setVisible(false);
            storePreferenceBounds();
            dispose();
        };
        cancelButton.addActionListener(actionListenerCancel);

        ActionListener actionListenerHelp = e -> GndHelp.openHelp("#layerDefinitionDialog");
        helpButton.addActionListener(actionListenerHelp);
    }

    private void saveLayer() {
        if (!_editable) {    // Sollte nie passieren, da der Button disabled ist.
            JOptionPane.showMessageDialog(this.getFrame(), "Dieser Layer ist nicht veränderbar!", "Fehler", JOptionPane.ERROR_MESSAGE);
            return;
        }
        String layerName = _nameTextField.getText();
        if (layerName.isEmpty()) {
            JOptionPane.showMessageDialog(this.getFrame(), "Bitte geben Sie einen Namen ein!", "Fehler", JOptionPane.ERROR_MESSAGE);
            return;
        } else if (layerName.length() >
                   40) {    // Präferenzen erlauben 80 Zeichen als Node-Name, doch damit der Name gut in die Legende passt, ist hier 40 gewählt.
            JOptionPane
                .showMessageDialog(this.getFrame(), "Bitte verwenden Sie einen Namen mit höchstens 40 Zeichen!", "Fehler", JOptionPane.ERROR_MESSAGE);
            return;
        }

        String infoText = _infoTextField.getText();
        if (infoText.isEmpty()) {
            infoText = null;
        }

        final String selectedPluginName = (String) _pluginComboBox.getSelectedItem();
        if (selectedPluginName == null || selectedPluginName.isEmpty()) {
            JOptionPane.showMessageDialog(this.getFrame(), "Bitte wählen Sie ein Plugin aus!", "Fehler", JOptionPane.ERROR_MESSAGE);
            return;
        }
        DisplayObjectTypePlugin selectedPlugin = PluginManager.getPlugin(selectedPluginName);

        final DafConfigurationObjectType selectedObjectType = (DafConfigurationObjectType) _configurationObjectTypesComboBox.getSelectedItem();
        if (selectedObjectType == null && !selectedPlugin.getName().equals("CSV")) {
            JOptionPane.showMessageDialog(this.getFrame(), "Bitte wählen Sie einen Typ aus!", "Fehler", JOptionPane.ERROR_MESSAGE);
            return;
        }

        final DOTCollection dotCollection = (DOTCollection) _dotTable.getModel();
        if (dotCollection.isEmpty()) {
            JOptionPane
                .showMessageDialog(this.getFrame(), "Bitte geben Sie mindestens einen Darstellungstypen an!", "Fehler", JOptionPane.ERROR_MESSAGE);
            return;
        }

        DOTElrPlugin elrPlugin = new DOTElrPlugin();
        ReferenceHierarchy referenceHierarchy = null;
        if (!elrPlugin.getName().equals(selectedPluginName)) {   // 'normale' Plugins
            for (DisplayObjectType dot : dotCollection.values()) {
                final DisplayObjectTypePlugin dotPlugin = dot.getDisplayObjectTypePlugin();
                if (null != selectedObjectType) {
                    if (!dotPlugin.isSystemObjectTypeSupported(_configuration, selectedObjectType)) {
                        JOptionPane.showMessageDialog(this.getFrame(),
                                                      "Der Darstellungstyp '" + dot.getName() + "' passt nicht zu '" + selectedObjectType.getName() +
                                                      "'!", "Fehler", JOptionPane.ERROR_MESSAGE);
                        return;
                    }
                }
            }
        } else {  // EOR-Plugin
            String referenceHierarchyName = (String) _referenceHierarchyComboBox.getSelectedItem();
            if (null == referenceHierarchyName || referenceHierarchyName.isEmpty()) {
                JOptionPane.showMessageDialog(this.getFrame(), "Bitte wählen Sie eine EOR-Hierachie aus!", "Fehler", JOptionPane.ERROR_MESSAGE);
                return;
            }
            referenceHierarchy = ReferenceHierarchyManager.getInstance().getReferenceHierarchy(referenceHierarchyName);
            String geometryType = referenceHierarchy.getGeometryType();
            final DisplayObjectTypePlugin geometryPlugin = PluginManager.getPlugin(geometryType);
            for (DisplayObjectType dot : dotCollection.values()) {
                final DisplayObjectTypePlugin dotPlugin = dot.getDisplayObjectTypePlugin();
                if (!dotPlugin.getName().equals(geometryPlugin.getName())) {
                    JOptionPane
                        .showMessageDialog(this.getFrame(), "Der Darstellungstyp '" + dot.getName() + "' passt nicht zu der EOR-Hierachie!", "Fehler",
                                           JOptionPane.ERROR_MESSAGE);
                    return;
                }
            }
        }

        if (null != selectedObjectType) {
            final List<AttributeGroup> attributeGroups = selectedObjectType.getAttributeGroups();
            for (DisplayObjectType dot : dotCollection.values()) {
                final Set<DOTSubscriptionData> subscriptionData = dot.getSubscriptionData();
                for (DOTSubscriptionData singleSubscriptionData : subscriptionData) {
                    final String attributeGroupPID = singleSubscriptionData.getAttributeGroup();
                    final AttributeGroup attributeGroup = _configuration.getAttributeGroup(attributeGroupPID);
                    if (!attributeGroups.contains(attributeGroup)) {
                        JOptionPane.showMessageDialog(this.getFrame(),
                                                      "Der Objekttyp '" + selectedObjectType.getName() + "' kann nicht mit der Attributgruppe '" +
                                                      attributeGroupPID + "' des Darstellungstyps '" + dot.getName() + "' verwendet werden.",
                                                      "Fehler", JOptionPane.ERROR_MESSAGE);
                        return;
                    }
                }
            }
        }
        final Layer existingLayer = LayerManager.getInstance().getLayer(layerName);
        if (existingLayer != null) {
            if (LayerManager.getInstance().isChangeable(existingLayer)) {
                Object[] options = {"Layer überschreiben", "Speichern abbrechen"};
                int n = JOptionPane
                    .showOptionDialog(this.getFrame(), "Soll der bestehende Layer mit diesem Namen wirklich überschrieben werden?", "Layer speichern",
                                      JOptionPane.YES_NO_CANCEL_OPTION, JOptionPane.QUESTION_MESSAGE, null, options, options[1]);
                if (n != 0) {
                    return;
                }
            } else {
                JOptionPane
                    .showMessageDialog(this.getFrame(), "Der bestehende Layer darf nicht überschrieben werden!", "Fehler", JOptionPane.ERROR_MESSAGE);
                return;
            }
        } else if (LayerManager.getInstance().hasLayerToLowerCase(layerName)) {
            JOptionPane.showMessageDialog(this.getFrame(),
                                          "Es existiert bereits ein Layer, dessen Name sich nur bezüglich Klein-Groß-Schreibung unterscheidet!",
                                          "Fehler", JOptionPane.ERROR_MESSAGE);
            return;
        }

        CsvFormat csvFormat = null;
        File csvFile = null;
        if (selectedPlugin.getName().equals("CSV")) {
            String csvFormatName = (String) _csvFormatComboBox.getSelectedItem();
            if (null == csvFormatName || csvFormatName.isEmpty()) {
                JOptionPane.showMessageDialog(this.getFrame(), "Bitte geben Sie ein CSV-Format an!", "Fehler", JOptionPane.ERROR_MESSAGE);
                return;
            }
            csvFormat = CsvFormatManager.getInstance().getCsvFormat(csvFormatName);
            String csvFileName = _csvFileTextField.getText();
            if (null == csvFileName || csvFileName.isEmpty()) {
                JOptionPane.showMessageDialog(this.getFrame(), "Bitte geben Sie eine CSV-Datei an!", "Fehler", JOptionPane.ERROR_MESSAGE);
                return;
            }
            csvFile = new File(csvFileName);
        }

        String pid = null;
        if (null != selectedObjectType) {
            pid = selectedObjectType.getPid();
        }

        Layer newLayer = new Layer(layerName, infoText, selectedPlugin, pid, referenceHierarchy, csvFormat, csvFile);
        newLayer.setDotCollection(dotCollection);

        //noinspection VariableNotUsedInsideIf
        if (existingLayer != null) {
            LayerManager.getInstance().changeLayer(newLayer);
        } else {
            LayerManager.getInstance().addLayer(newLayer);
        }
        _scratchLayer = newLayer;
        _unchangeableOriginalLayer = _scratchLayer.getCopy();
        _somethingChanged = false;
        // Damit der Benutzer dasselbe Objekt nicht unter anderem Namen speichern kann:
        _nameChangeable = false;
        _nameTextField.setEditable(false);
    }

    private void addFrameListener() {
        class FrameListener extends WindowAdapter {
            @Override
            public void windowClosing(WindowEvent e) {
                if (_somethingChanged && _editable) {
                    if (saveChangesWanted()) {
                        saveLayer();
                        setVisible(false);
                        storePreferenceBounds();
                        dispose();
                    }
                }
            }
        }
        addWindowListener(new FrameListener());
    }

    private boolean saveChangesWanted() {
        Object[] options = {"Änderungen speichern", "Nicht speichern"};
        int n = JOptionPane
            .showOptionDialog(LayerDefinitionDialog.this.getFrame(), "Änderungen speichern?", "Es wurden Änderungen an dem Layer vorgenommen.",
                              JOptionPane.YES_NO_CANCEL_OPTION, JOptionPane.QUESTION_MESSAGE, null, options, options[1]);
        return n == 0;
    }

    private void addListSelectionListener() {
        final ListSelectionModel selectionModel = _dotTable.getSelectionModel();
        // vordefinierte Darstellungsobjekttypen dürfen nicht bearbeitet oder gelöscht werden
        ListSelectionListener listSelectionListener = e -> {
            final int selectedRow = _dotTable.getSelectedRow();
            if (selectedRow == -1) {
                _deleteDOTButton.setEnabled(false);
            } else {
                _deleteDOTButton.setEnabled(_editable);
            }
        };
        selectionModel.addListSelectionListener(listSelectionListener);
    }

    // Implementation des ReferenceHierarchieManager.RhmChangeListener:

    @Override
    public void referenceHierarchyAdded(final ReferenceHierarchy referenceHierarchy) {
        refreshHierarchies();
    }

    @Override
    public void referenceHierarchyChanged(final ReferenceHierarchy referenceHierarchy) {
        refreshHierarchies();
    }

    @Override
    public void referenceHierarchyRemoved(final String name) {
        refreshHierarchies();
    }

    private void refreshHierarchies() {
        Object selected = _referenceHierarchyComboBox.getSelectedItem();
        _referenceHierarchyComboBox.setModel(new DefaultComboBoxModel<>(ReferenceHierarchyManager.getInstance().getReferenceHierarchyNames(false)));
        if (null != selected) {
            _referenceHierarchyComboBox.setSelectedItem(selected);
        } else {
            _referenceHierarchyComboBox.setSelectedIndex(-1);
        }
    }

    // Implementation des CsvFormatManager.CsvFormatManagerChangeListener:

    @Override
    public void csvFormatAdded(final CsvFormat format) {
        refreshCsvFormats();
    }

    @Override
    public void csvFormatChanged(final CsvFormat format) {
        refreshCsvFormats();
    }

    @Override
    public void csvFormatRemoved(final String formatName) {
        refreshCsvFormats();
    }

    private void refreshCsvFormats() {
        Object selected = _csvFormatComboBox.getSelectedItem();
        _csvFormatComboBox.setModel(new DefaultComboBoxModel<>(CsvFormatManager.getInstance().getCsvFormatNames()));
        if (null != selected) {
            _csvFormatComboBox.setSelectedItem(selected);
        } else {
            _csvFormatComboBox.setSelectedIndex(-1);
        }
    }

    @Override
    public String toString() {
        return "LayerDefinitionDialog{}";
    }

    private interface LayersEditingStateListener {
        @SuppressWarnings("unused")
        void layersEditingStateChanged(boolean editable);
    }

    private class EditableListenButton extends JButton implements LayersEditingStateListener {

        EditableListenButton(String text) {
            super(text);
            _listeningButtons.add(this);

        }

        @Override
        public void layersEditingStateChanged(boolean editable) {
            setEnabled(editable);
        }
    }
}
