/*
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */
package de.kappich.pat.gnd.layerManagement;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.kappich.pat.gnd.displayObjectToolkit.DOTCollection;
import de.kappich.pat.gnd.documentation.GndHelp;
import de.kappich.pat.gnd.utils.SpringUtilities;
import de.kappich.pat.gnd.utils.view.GndFrame;
import de.kappich.pat.gnd.utils.view.GndTable;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Frame;
import java.awt.Rectangle;
import java.awt.Window;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.List;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.SpringLayout;
import javax.swing.WindowConstants;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableModel;


/**
 * Eine Dialog, der alle Layer auflistet und über Schaltflächen Möglichkeiten bietet, Layer zu betrachten,
 * zu bearbeiten, zu löschen oder neu einzuführen. Der LayerManagerDialog zeigt die Inhalte der
 * {@link LayerManager Layerverwaltung}.
 *
 * @author Kappich Systemberatung
 * @version $Revision$
 */
@SuppressWarnings("OverlyLongMethod")
public class LayerManagerDialog {

    private final ClientDavInterface _connection;

    private final GndFrame _frame;

    private LayerDefinitionDialog _layerDefinitionDialog = null;

    /**
     * Konstruiert einen neuen LayerManagerDialog. Das ClientDavInterface ist z.B. notwendig, um über das
     * DataModel Informationen zu den Geo-Referenz-Objekten bekommen zu können.
     *
     * @param connection die Datenverteiler-Verbindung
     */
    public LayerManagerDialog(ClientDavInterface connection) {

        _connection = connection;

        _frame = new GndFrame("LayerManagerDialog", "GND: Layer-Verwaltung");

        _frame.setLayout(new BorderLayout());

        final JButton buttonNeu = new JButton("Neuer Layer");
        buttonNeu.setActionCommand("Einen neuen Layer anlegen");

        final JButton buttonBearbeiten = new JButton("Bearbeiten");
        buttonBearbeiten.setActionCommand("Details des augewählten Layers bearbeiten");

        final JButton buttonCopy = new JButton("Kopieren");
        buttonCopy.setActionCommand("Den ausgewählten Layer kopieren und bearbeiten");

        final JButton buttonLoeschen = new JButton("Löschen");
        buttonLoeschen.setActionCommand("Den ausgewählten Layer löschen");

        final JButton buttonHilfe = new JButton("Hilfe");
        buttonHilfe.setActionCommand("Die Online-Hilfe öffnen");

        buttonBearbeiten.setEnabled(false);
        buttonCopy.setEnabled(false);
        buttonLoeschen.setEnabled(false);

        JPanel buttonsPanelEast = new JPanel();
        buttonsPanelEast.setLayout(new BoxLayout(buttonsPanelEast, BoxLayout.Y_AXIS));

        Dimension d = new Dimension(15, 15);
        buttonsPanelEast.add(Box.createRigidArea(d));

        buttonsPanelEast.add(buttonNeu);
        buttonsPanelEast.add(Box.createRigidArea(d));

        buttonsPanelEast.add(buttonBearbeiten);
        buttonsPanelEast.add(Box.createRigidArea(d));

        buttonsPanelEast.add(buttonCopy);
        buttonsPanelEast.add(Box.createRigidArea(d));

        buttonsPanelEast.add(buttonLoeschen);
        buttonsPanelEast.add(Box.createRigidArea(d));

        buttonsPanelEast.add(buttonHilfe);
        buttonsPanelEast.add(Box.createRigidArea(d));

        buttonsPanelEast.setBorder(BorderFactory.createEmptyBorder(5, 10, 5, 10));

        JPanel buttonsPanelSouth = new JPanel();
        buttonsPanelSouth.setLayout(new SpringLayout());
        buttonsPanelSouth.setBorder(BorderFactory.createEmptyBorder(5, 10, 5, 10));

        JButton schliessenButton = new JButton("Dialog schließen");
        buttonsPanelSouth.add(schliessenButton);
        SpringUtilities.makeCompactGrid(buttonsPanelSouth, 1, 20, 5);

        final GndTable table = new GndTable(LayerManager.getInstance());

        table.setFillsViewportHeight(true);
        table.setDefaultRenderer(Object.class, new LayerTableRenderer());
        // vordefinierte Darstellungstypen dürfen nicht bearbeitet oder gelöscht werden
        final ListSelectionModel selectionModel = table.getSelectionModel();

        final List<Layer> layerList = LayerManager.getInstance().getLayers();

        ListSelectionListener listSelctionListener = e -> {
            final int selectedRow = table.getSelectedRow();

            if(selectedRow == -1) {
                buttonBearbeiten.setToolTipText("Es ist keine Ansicht ausgewählt worden");
                buttonBearbeiten.setEnabled(false);
                buttonCopy.setToolTipText("Es ist keine Ansicht ausgewählt worden");
                buttonCopy.setEnabled(false);
                buttonLoeschen.setToolTipText("Es ist keine Ansicht ausgewählt worden");
                buttonLoeschen.setEnabled(false);
            }
            else {
                final boolean changeable = LayerManager.getInstance().isChangeable(layerList.get(selectedRow));
                if(!changeable) {
                    buttonBearbeiten.setText("Betrachten");
                    buttonBearbeiten.setToolTipText("Details des augewählten Layers betrachten");
                    buttonBearbeiten.setEnabled(true);
                    buttonCopy.setToolTipText("Kopie des ausgewählten Layers erstellen und bearbeiten");
                    buttonCopy.setEnabled(true);
                    buttonLoeschen.setToolTipText("Der ausgewählte Layer ist nicht löschbar");
                    buttonLoeschen.setEnabled(false);
                }
                else {
                    buttonBearbeiten.setText("Bearbeiten");
                    buttonBearbeiten.setToolTipText("Details des augewählten Layers bearbeiten");
                    buttonBearbeiten.setEnabled(true);
                    buttonCopy.setToolTipText("Kopie des ausgewählten Layers erstellen und bearbeiten");
                    buttonCopy.setEnabled(true);
                    buttonLoeschen.setToolTipText("Den ausgewählten Layer löschen");
                    buttonLoeschen.setEnabled(true);
                }
            }
        };
        selectionModel.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        selectionModel.addListSelectionListener(listSelctionListener);

        final KeyAdapter keyAdapter = new KeyAdapter() {
            @Override
            public void keyPressed(KeyEvent e) {
                if(e.getKeyCode() == KeyEvent.VK_DELETE) {
                    final int selectedRow = table.getSelectedRow();
                    if(selectedRow >= 0) {
                        final Layer layer = layerList.get(selectedRow);
                        final LayerManager instance = LayerManager.getInstance();
                        if(instance.isChangeable(layer)) {
                            Object[] options = {"Ja", "Nein"};
                            int n = JOptionPane.showOptionDialog(
                                _frame.getFrame(),
                                "Wollen Sie den Layer " + layer.getName() + " wirklich löschen?",
                                "Layer " + layer.getName() + "löschen",
                                JOptionPane.YES_NO_OPTION,
                                JOptionPane.QUESTION_MESSAGE,
                                null,
                                options,
                                options[0]
                            );
                            if(n == 0) {
                                instance.removeLayer(layer);
                            }
                        }
                    }
                }
                else {
                    char c = e.getKeyChar();
                    if(Character.isLetter(c)) {
                        TableModel model = table.getModel();
                        int index = ((LayerManager) model).getIndexOfFirstLayer(c);
                        if(index >= 0) {
                            table.getSelectionModel().setSelectionInterval(index, index);
                            table.scrollRectToVisible(new Rectangle(table.getCellRect(index, 0, true)));
                        }
                    }
                }
            }
        };
        table.addKeyListener(keyAdapter);
        buttonNeu.addKeyListener(keyAdapter);
        buttonBearbeiten.addKeyListener(keyAdapter);
        buttonCopy.addKeyListener(keyAdapter);
        buttonLoeschen.addKeyListener(keyAdapter);
        buttonHilfe.addKeyListener(keyAdapter);

        table.addMouseListener(new MouseAdapter() {
            @Override
            public void mousePressed(final MouseEvent e) {
                if(e.getClickCount() > 1) {
                    editOrInspectAction(table);
                }
            }
        });

        _frame.add(buttonsPanelEast, BorderLayout.EAST);
        _frame.add(buttonsPanelSouth, BorderLayout.SOUTH);
        _frame.add(new JScrollPane(table), BorderLayout.CENTER);

        // Neu, Bearbeiten, Kopieren, Löschen
        ActionListener actionListenerNew = e -> newAction();
        buttonNeu.addActionListener(actionListenerNew);


        ActionListener actionListenerLoeschen = e -> deleteAction(table);
        buttonLoeschen.addActionListener(actionListenerLoeschen);

        ActionListener actionListenerKopieren = e -> copyAction(table);
        buttonCopy.addActionListener(actionListenerKopieren);

        ActionListener actionListenerBearbeiten = e -> editOrInspectAction(table);
        buttonBearbeiten.addActionListener(actionListenerBearbeiten);

        ActionListener actionListenerHelp = e -> GndHelp.openHelp("#theLayerManagerDialog");
        buttonHilfe.addActionListener(actionListenerHelp);

        ActionListener actionListenerDialogSchliessen = e -> closeDialog();
        schliessenButton.addActionListener(actionListenerDialogSchliessen);

        class FrameListener extends WindowAdapter {
            @Override
            public void windowClosing(WindowEvent e) {
                /*
                 * wenn nur noch ein einziges Fenster geöffnet ist
                 * beendet sich das Programm beim Schließen des Fensters
                 */
//				final Frame[] frames = Frame.getFrames();
//				int length = frames.length - 1;
//				for(Frame frame : frames) {
//					if(!frame.isVisible()) {
//						length--;
//					}
//				}
//				if(length == 0) {
//					_frame.setDefaultCloseOperation(WindowConstants.EXIT_ON_CLOSE);
//				}
                // Die alte Implentation beendete gelegentlich die ganze Anwendung, wenn
                // nur ein Frame geschlossen wurde, etwa während einer Initialisierung einer
                // neuen Ansicht. Statt Frames versuche ich jetzt Windows.
                final Window[] windows = Window.getWindows();
                int length = windows.length - 1;
                for(Window window : windows) {
                    if(!window.isVisible()) {
                        length--;
                    }
                }
                if(length <= 0) {
                    _frame.setDefaultCloseOperation(WindowConstants.EXIT_ON_CLOSE);
                }
            }
        }
        _frame.addWindowListener(new FrameListener());

        if(!_frame.hasPreferences()) {
            table.setVisibleRowCount(Math.min(40, table.getModel().getRowCount()));
        }
        _frame.setPositionAndSize(0, 0, 860, 50, true, 0, 0);
        _frame.setVisible(true);
    }

    private void newAction() {
        Layer newLayer = new Layer();
        newLayer.setName("");
        newLayer.setInfo("");
        final String title = "GND: neuen Layer bearbeiten";
        if(_layerDefinitionDialog == null) {
            _layerDefinitionDialog = new LayerDefinitionDialog(_connection, newLayer, true, true, title);
        }
        else {
            _layerDefinitionDialog.setLayer(newLayer, true, true);
            _layerDefinitionDialog.setTitle(title);
            _layerDefinitionDialog.setPositionAndSize();
            _layerDefinitionDialog.setVisible(true);
            _layerDefinitionDialog.toFront();
        }
    }

    private void deleteAction(final JTable table) {
        if(table.getSelectedRowCount() == 0) {
            JOptionPane.showMessageDialog(
                _frame.getFrame(),
                "Bitte wählen Sie mindestens eine Zeile aus der Liste aus!",
                "Fehler",
                JOptionPane.ERROR_MESSAGE
            );
            return;
        }
        final int[] selectedRows = table.getSelectedRows();
        boolean allLayersRemoved = true;
        final LayerManager layerManager = LayerManager.getInstance();
        for(int i = selectedRows.length - 1; i >= 0; i--) {
            final Layer layer = layerManager.getLayer(selectedRows[i]);
            if(!layerManager.removeLayer(layer)) {
                allLayersRemoved = false;
            }
        }
        layerManager.fireTableDataChanged();
        if(!allLayersRemoved) {
            JOptionPane.showMessageDialog(
                _frame.getFrame(),
                "Es konnten nicht alle ausgewählten Layer gelöscht werden!",
                "Fehler",
                JOptionPane.ERROR_MESSAGE
            );
        }
    }

    private void copyAction(final JTable table) {
        if(table.getSelectedRowCount() != 1) {
            JOptionPane.showMessageDialog(
                _frame.getFrame(),
                "Bitte wählen Sie genau eine Zeile aus der Liste aus!",
                "Fehler",
                JOptionPane.ERROR_MESSAGE
            );
            return;
        }
        final int selectedRow = table.getSelectedRow();
        final Layer layer = LayerManager.getInstance().getLayer(selectedRow);
        Layer newLayer = layer.getCopy();
        newLayer.setName(layer.getName() + " (Kopie)");
        newLayer.setDotCollection((DOTCollection) layer.getDotCollection().clone());
        final String title = "GND: einen kopierten Layer bearbeiten";
        if(_layerDefinitionDialog == null) {
            _layerDefinitionDialog = new LayerDefinitionDialog(_connection, newLayer, true, true, title);
        }
        else {
            _layerDefinitionDialog.setLayer(newLayer, true, true);
            _layerDefinitionDialog.setTitle(title);
            _layerDefinitionDialog.setPositionAndSize();
            _layerDefinitionDialog.setVisible(true);
            _layerDefinitionDialog.toFront();
        }
    }

    private void editOrInspectAction(final JTable table) {
        final int selectedRow = table.getSelectedRow();
        if(selectedRow == -1) {
            JOptionPane.showMessageDialog(
                _frame.getFrame(),
                "Bitte wählen Sie mindestens eine Zeile aus der Liste aus!",
                "Fehler",
                JOptionPane.ERROR_MESSAGE
            );
            return;
        }
        final Layer layer = LayerManager.getInstance().getLayer(selectedRow);
        final boolean changeable = LayerManager.getInstance().isChangeable(layer);
        final String title;
        if(changeable) {
            title = "GND: einen Layer bearbeiten";
        }
        else {
            title = "GND: einen Layer betrachten";
        }
        if(_layerDefinitionDialog == null) {
            _layerDefinitionDialog = new LayerDefinitionDialog(_connection, layer, changeable, false, title);
        }
        else {
            _layerDefinitionDialog.setLayer(layer, changeable, false);
            _layerDefinitionDialog.setTitle(title);
            _layerDefinitionDialog.setPositionAndSize();
            _layerDefinitionDialog.setVisible(true);
            _layerDefinitionDialog.toFront();
        }
    }

    /**
     * CellRenderer für die JTable im Verwaltungsdialog
     *
     * @author Kappich Systemberatung
     * @version $Revision$
     */
    private static class LayerTableRenderer extends DefaultTableCellRenderer {

        private static final long serialVersionUID = 1L;

        @Override
        public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
            super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);

            if(null != value) {   // null passiert auf Macs
	            if (value instanceof String layerName) {
                    // Text
                    setText(layerName);
                    if(LayerManager.getInstance().hasLayerToLowerCase(layerName.toLowerCase())) {
                        Layer layer = LayerManager.getInstance().getLayer(layerName);
                        // Font und Vordergrundfarbe
                        if(LayerManager.getInstance().isChangeable(layer)) {
                            setForeground(Color.BLACK);
                            Font font = getFont();
                            setFont(new Font(font.getName(), Font.PLAIN, font.getSize()));
                        }
                        else {
                            setForeground(Color.GRAY);
                            Font font = getFont();
                            setFont(new Font(font.getName(), Font.ITALIC, font.getSize()));
                        }
                        // Tooltipp
                        if(layer.getPlugin() != null) {
                            setToolTipText(layer.getInfo() + " (Plugin: " + layer.getPlugin().getName() + ")");
                        }
                        else {
                            setToolTipText(layer.getInfo());
                        }
                    }
                    else { // sollte nicht eintreten
                        setForeground(Color.RED);
                        Font font = getFont();
                        setFont(new Font(font.getName(), Font.BOLD, font.getSize()));
                    }
                }
                else { // sollte nicht eintreten
                    setText(value.toString());
                    setForeground(Color.RED);
                    Font font = getFont();
                    setFont(new Font(font.getName(), Font.BOLD, font.getSize()));
                }
            }
            return this;
        }
    }

    /**
     * Macht die Layerverwaltungsdialog sichtbar.
     */
    public void showDialog(boolean visible) {
        _frame.setState(Frame.NORMAL);
        _frame.setVisible(visible);
    }

    /**
     * Beendet den Layerverwaltungsdialog.
     */
    public void closeDialog() {
        _frame.setVisible(false);
        _frame.storePreferenceBounds();
        _frame.dispose();
    }

    @Override
    public String toString() {
        return "LayerManagerDialog";
    }
}
