/*
 * Copyright 2011-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.notice;

import de.bsvrz.dav.daf.main.config.ObjectLookup;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.debug.Debug;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;
import java.util.regex.Pattern;

/**
 * @author Kappich Systemberatung
 */
public class NoticeManager {

    private static final Pattern DOUBLE_QUOTE = Pattern.compile("\"");
    private static final Debug _debug = Debug.getLogger();
    private final Map<SystemObject, Notice> _noticeMap = new HashMap<>();
    private final Preferences _preferences;
    private final SimpleDateFormat _simpleDateFormat = new SimpleDateFormat("dd.MM.yyyy HH:mm:ss");

    /**
     * Der Konstruktor.
     *
     * @param preferences die Präferenzen
     * @param dataModel   das Lookup
     */
    public NoticeManager(final Preferences preferences, final ObjectLookup dataModel) {
        _preferences = preferences;
        try {
            for (String s : _preferences.childrenNames()) {
                SystemObject object = dataModel.getObject(s);
                if (object == null) {
                    try {
                        long l = Long.parseLong(s);
                        object = dataModel.getObject(l);
                    } catch (NumberFormatException ignored) {
                    }
                }
                if (object == null) {
                    _debug.info("Es ist eine Notiz zu " + s + " gespeichert, das zugehörige Objekt kann aber nicht gefunden werden.");
                }
                Preferences node = _preferences.node(s);
                String notice = node.get("text", "");
                long creation = node.getLong("creation", System.currentTimeMillis());
                long change = node.getLong("change", System.currentTimeMillis());
                if (!notice.isEmpty()) {
                    _noticeMap.put(object, new Notice(notice, creation, change));
                }
            }
        } catch (BackingStoreException e) {
            _debug.warning("Fehler beim Laden der Notizen", e);
        }
    }

    /**
     * Setzt eine Notiz.
     *
     * @param systemObject eine SystemObject
     * @param text         der Text
     *
     * @return {@code true} im Erfolgsfall
     */
    public boolean setNotice(final SystemObject systemObject, final String text) {
        try {
            final String trimmedText = text.trim();
            if (trimmedText.isEmpty()) {
                _noticeMap.remove(systemObject);
                return true;
            }
            final Notice notice = getNotice(systemObject);
            return null == _noticeMap.put(systemObject, new Notice(trimmedText, notice.getCreationTime(), System.currentTimeMillis()));
        } finally {
            saveNode(systemObject);
        }
    }

    private void saveNode(final SystemObject systemObject) {
        final Notice notice = _noticeMap.get(systemObject);
        try {
            if (notice == null) {
                _preferences.node(systemObject.getPidOrId()).removeNode();
            } else {
                final Preferences node = _preferences.node(systemObject.getPidOrId());
                node.put("text", notice.getMessage());
                node.putLong("creation", notice.getCreationTime());
                node.putLong("change", notice.getChangeTime());
            }
            _preferences.flush();
        } catch (BackingStoreException e) {
            _debug.warning("Fehler beim Speichern einer Notiz", e);
        }
    }

    /**
     * Gibt eine unveränderliche Collection der SystemObjects mit Notiz zurück.
     *
     * @return die Collection
     */
    public Collection<SystemObject> getObjectsWithNotice() {
        return Collections.unmodifiableSet(_noticeMap.keySet());
    }

    /**
     * Gibt die Notiz zu einem SystemObject zurück.
     *
     * @param object das SystemObject
     *
     * @return die Notiz, falls existent, zumindest aber ein leerer String
     */
    public Notice getNotice(final SystemObject object) {
        Notice notice = _noticeMap.get(object);
        if (notice == null) {
            notice = new Notice("");
        }
        return notice;
    }

    /**
     * Gibt es zu dem SystemObject eine Notiz?
     *
     * @param systemObject das SystemObject
     *
     * @return die Antwaort auf die Frage
     */
    public boolean hasNotice(final SystemObject systemObject) {
        return _noticeMap.containsKey(systemObject);
    }

    /**
     * Exportiert die Notizen in eine Datei.
     *
     * @param file die Datei
     *
     * @throws IOException falls ein Fehler auftritt
     */
    public void exportToFile(final File file) throws IOException {
        try (FileWriter fileWriter = new FileWriter(file)) {
            fileWriter.write("Objekt;Text;Erstellt;Geändert\n");
            for (final Map.Entry<SystemObject, Notice> noticeEntry : _noticeMap.entrySet()) {
                fileWriter.write(
                    '"' + noticeEntry.getKey().getPidOrId() + "\";\"" + DOUBLE_QUOTE.matcher(noticeEntry.getValue().getMessage()).replaceAll("\"\"") +
                    "\";\"" + _simpleDateFormat.format(new Date(noticeEntry.getValue().getCreationTime())) + "\";\"" +
                    _simpleDateFormat.format(new Date(noticeEntry.getValue().getChangeTime())) + "\"\n");
            }
        }
    }

    /**
     * Löscht alle Notizen.
     */
    public void clear() {
        try {
            for (String s : _preferences.childrenNames()) {
                _preferences.node(s).removeNode();
            }
            _preferences.flush();
            _noticeMap.clear();
        } catch (BackingStoreException e) {
            _debug.warning("Fehler beim Löschen der Notizen", e);
        }
    }

    @Override
    public String toString() {
        return "NoticeManager{" + "_noticeMap=" + _noticeMap + ", _preferences=" + _preferences + '}';
    }
}
