/*
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.pluginInterfaces;

import de.bsvrz.dav.daf.main.DataState;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.kappich.pat.gnd.displayObjectToolkit.*;
import de.kappich.pat.gnd.gnd.LegendTreeNodes;
import de.kappich.pat.gnd.properties.ColorProperty;
import de.kappich.pat.gnd.properties.DistanceProperty;
import de.kappich.pat.gnd.properties.FillingProperty;
import de.kappich.pat.gnd.properties.PropertiesManager;
import de.kappich.pat.gnd.properties.Property;
import de.kappich.pat.gnd.properties.StrokeWidthProperty;
import de.kappich.pat.gnd.utils.Interval;
import de.kappich.pat.gnd.utils.view.PreferencesDeleter;
import java.awt.Color;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.TreeMap;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;
import javax.swing.JOptionPane;
import javax.swing.table.TableModel;

/**
 * Ein DefaultDisplayObjectType ist eine abstrakte Klasse, die eine teilweise Implementation von DisplayObjectType ist. Die Grundfigur im Interface
 * DisplayObjectType wird allerdings in dieser Implementation stets ignoriert. Subklassen, für die dies Verhalten ideal ist, sind DOTLine, DOTArea und
 * DOTComplex.
 *
 * @author Kappich Systemberatung
 */
public abstract class DefaultDisplayObjectType implements DisplayObjectType, DOTManager.DOTChangeListener {

    private static final String LOWER_BOUND = "LOWER_BOUND";
    private static final String UPPER_BOUND = "UPPER_BOUND";
    private static final String INFO = "INFO";

    // Die folgende Map definiert, welche Eigenschaften statisch sind. Für die beiden folgenden
    // Maps gilt: _staticPropertyValues enthält für jede statische Property einen Eintrag,
    // während _dynamicDOTItemManagers für jede (!) Property einen Eintrag enthält, denn
    // die DynamicDOTItemManager-Objekte in dieser Map sind in DOTLinePanel ja die TableModels.

    /**
     * Diese Map speichert für jede {@link Property}, ob sie (aktuell) statisch ist.
     */
    protected final Map<Property, Boolean> _isStaticMap = new HashMap<>();
    /**
     * Diese Map speichert die Werte für die statischen {@link Property Properties}.
     */
    protected final Map<Property, Object> _staticPropertyValues = new HashMap<>();
    /**
     * Diese Map speichert die Werte für die dynamischen {@link Property Properties}.
     */
    protected final Map<Property, DynamicDOTItemManager> _dynamicDOTItemManagers = new HashMap<>();
    /**
     * Der Name.
     */
    protected String _name = "";

    /**
     * Eine Information.
     */
    protected String _info = "";

    /**
     * Ein DefaultDisplayObjectType ist keine funktional vollständige Implementation von DisplayObjectType, sondern beinhaltet die Gemeinsamkeiten der
     * Implementation von DOTArea, DOTComplex, DOTLine und DOTKm.
     */
    protected DefaultDisplayObjectType() {
        initCollections(false);
        DOTManager.getInstance().addDOTChangeListener(this);
    }

    protected DefaultDisplayObjectType(final boolean staticOnly) {
        initCollections(staticOnly);
        DOTManager.getInstance().addDOTChangeListener(this);
    }

    private void initCollections(final boolean staticOnly) {
        final DisplayObjectTypePlugin displayObjectTypePlugin = getDisplayObjectTypePlugin();
        final Property[] properties = displayObjectTypePlugin.getProperties(null);
        for (Property property : properties) {
            _isStaticMap.put(property, true);
            _staticPropertyValues.put(property, property.getDefaultValue());
            if (!staticOnly) {
                _dynamicDOTItemManagers.put(property, new DynamicDOTItemManager());
            }
        }
    }

    @Override
    public String getName() {
        return _name;
    }

    @Override
    public void setName(String name) {
        _name = name;
    }

    @Override
    public String getInfo() {
        return _info;
    }

    @Override
    public void setInfo(String info) {
        _info = info;
    }

    @Override
    public Set<String> getPrimitiveFormNames() {
        return new HashSet<>();
    }

    @SuppressWarnings("VariableNotUsedInsideIf")
    @Override
    @Nullable
    public String getPrimitiveFormType(@Nullable String primitiveFormName) {
        if (primitiveFormName != null) {
            return null;
        }
        return "";
    }

    @SuppressWarnings("VariableNotUsedInsideIf")
    @Override
    @Nullable
    public String getPrimitiveFormInfo(@Nullable String primitiveFormName) {
        if (primitiveFormName != null) {
            return null;
        }
        return "";
    }

    @Override
    public void removePrimitiveForm(String primitiveFormName) {
    }

    @Nullable
    @SuppressWarnings("VariableNotUsedInsideIf")
    @Override
    public List<Property> getDynamicProperties(@Nullable String primitiveFormName) {
        if (primitiveFormName != null) {
            return null;
        }
        List<Property> dynamicPropertyList = new ArrayList<>();
        for (final Entry<Property, Boolean> dotPropertyBooleanEntry : _isStaticMap.entrySet()) {
            if (!dotPropertyBooleanEntry.getValue()) {
                dynamicPropertyList.add(dotPropertyBooleanEntry.getKey());
            }
        }
        return dynamicPropertyList;
    }

    @Override
    public boolean isPropertyStatic(@Nullable String primitiveFormName, Property property) {
        return _isStaticMap.get(property);
    }

    /*
     * Die Default-Implementation ignoriert den Grundfigurnamen und berücksichtigt, falls der übergebene
     * Boolean <code>true</code> ist, ob die Eigenschaft bisher dynamisch war, und setzt ihren Wert
     * auf den Defaultwert.
     */
    @Override
    public void setPropertyStatic(@Nullable String primitiveFormName, Property property, boolean becomesStatic) {
        Property[] pluginProperties = getDisplayObjectTypePlugin().getProperties(null);
        boolean propertyAllowed = false;
        for (Property pluginProperty : pluginProperties) {
            if (pluginProperty.equals(property)) {
                propertyAllowed = true;
                break;
            }
        }
        if (!propertyAllowed) {
            throw new IllegalArgumentException(
                "Unzulässige Eigenschaft '" + property.getName() + "' in DefaultDisplayObjectType.setPropertyStatic().");
        }
        if (becomesStatic) {
            final boolean wasDynamic = _dynamicDOTItemManagers.containsKey(property);
            if (wasDynamic) {
                _staticPropertyValues.put(property, property.getDefaultValue());
            }
        }
        _isStaticMap.put(property, becomesStatic);
    }

    @Nullable
    @Override
    public Object getValueOfStaticProperty(@Nullable String primitiveFormName, Property property) {
        if (_isStaticMap.containsKey(property)) {
            return _staticPropertyValues.get(property);
        } else {
            return property.getDefaultValue();
        }
    }

    @Override
    public void setValueOfStaticProperty(@Nullable String primitiveFormName, Property property, Object value) {
        _staticPropertyValues.put(property, value);
    }

    @Override
    public void setValueOfDynamicProperty(@Nullable String primitiveFormName, Property property, DisplayObjectTypeItem dItem,
                                          @Nullable Double lowerBound, @Nullable Double upperBound) {
        if (property == null) {
            return;
        }
        DynamicDOTItemManager dynamicDOTItemManager = _dynamicDOTItemManagers.get(property);
        if (dynamicDOTItemManager == null) {
            return;
        }
	    if (!(dItem instanceof DynamicDOTItem ldItem)) {
            return;
        }
        final Interval<Double> interval;
        if (lowerBound == null || upperBound == null) {
            interval = new Interval<>(Double.NEGATIVE_INFINITY, Double.NEGATIVE_INFINITY);
        } else {
            interval = new Interval<>(lowerBound, upperBound);
        }
        dynamicDOTItemManager.insert(interval, ldItem);
    }

    @Override
    public DisplayObjectType getCopy(String name) {
        DefaultDisplayObjectType copy = (DefaultDisplayObjectType) getDisplayObjectTypePlugin().getDisplayObjectType();
        if (name != null) {
            copy.setName(name);
        } else {
            copy.setName(_name);
        }
        copy.setInfo(_info);
        for (final Entry<Property, Boolean> dotPropertyBooleanEntry : _isStaticMap.entrySet()) {
            Boolean newBoolean = dotPropertyBooleanEntry.getValue();
            copy._isStaticMap.put(dotPropertyBooleanEntry.getKey(), newBoolean);
        }
        for (final Entry<Property, Object> dotPropertyObjectEntry : _staticPropertyValues.entrySet()) {
            Object object = dotPropertyObjectEntry.getValue();
            Object newObject;
	        if (object instanceof Color color) {
                newObject = new Color(color.getRed(), color.getBlue(), color.getGreen(), color.getAlpha());
            } else {
                newObject = object;
            }
            copy._staticPropertyValues.put(dotPropertyObjectEntry.getKey(), newObject);
        }
        for (final Entry<Property, DynamicDOTItemManager> dotPropertyDynamicDOTItemManagerEntry : _dynamicDOTItemManagers.entrySet()) {
            DynamicDOTItemManager dynamicDOTItemManager = dotPropertyDynamicDOTItemManagerEntry.getValue();
            DynamicDOTItemManager newDynamicDOTItemManager = dynamicDOTItemManager.getCopy();
            copy._dynamicDOTItemManagers.put(dotPropertyDynamicDOTItemManagerEntry.getKey(), newDynamicDOTItemManager);
        }
        return copy;
    }

    @Override
    public void putPreferences(Preferences prefs) {
        deletePreferences(prefs);
        Preferences classPrefs = prefs.node(getClass().getSimpleName());
        Preferences objectPrefs = classPrefs.node(getName());
        Preferences infoPrefs = objectPrefs.node("info");
        infoPrefs.put(INFO, getInfo());

        Preferences staticPrefs = objectPrefs.node("static");
        Preferences dynamicPrefs = objectPrefs.node("dynamic");
        for (final Entry<Property, Boolean> entry : _isStaticMap.entrySet()) {
            if (entry.getValue()) {
                // Eine statische Property wird als dynamische ohne Anmeldungsdaten weggeschrieben.
                Preferences propertyPrefs = staticPrefs.node(entry.getKey().getKey());
                final DynamicDOTItem dynamicDOTItem = new DynamicDOTItem("", "", "", "", _staticPropertyValues.get(entry.getKey()));
                dynamicDOTItem.putPreferences(propertyPrefs);
            } else {
                Preferences propertyPrefs = dynamicPrefs.node(entry.getKey().getKey());
                DynamicDOTItemManager dynamicDOTItemManager = _dynamicDOTItemManagers.get(entry.getKey());
                int i = 0;
                for (TreeMap<Interval<Double>, DynamicDOTItem> treeMap : dynamicDOTItemManager.getTreeMaps()) {
                    for (final Entry<Interval<Double>, DynamicDOTItem> intervalDynamicDOTItemEntry : treeMap.entrySet()) {
                        Preferences objectForItemPrefs = propertyPrefs.node("interval" + i);
                        objectForItemPrefs.putDouble(LOWER_BOUND, intervalDynamicDOTItemEntry.getKey().getLowerBound());
                        objectForItemPrefs.putDouble(UPPER_BOUND, intervalDynamicDOTItemEntry.getKey().getUpperBound());
                        DynamicDOTItem dynamicItem = intervalDynamicDOTItemEntry.getValue();
                        if ((dynamicItem == null) && (treeMap.size() >= 1)) {
                            // weil es mit Double.NEGATIVE_INFINITY nicht geht, das get(); das >= ist wichtig!
                            dynamicItem = treeMap.values().toArray(new DynamicDOTItem[1])[0];
                        }
                        if (null != dynamicItem) {
                            dynamicItem.putPreferences(objectForItemPrefs);
                        }
                        i++;
                    }
                }
            }
        }
    }

    /* Abstrakte Methoden */

    @SuppressWarnings({"OverlyLongMethod", "OverlyNestedMethod"})
    @Override
    public void initializeFromPreferences(Preferences prefs) {
        _name = prefs.name();
        Preferences infoPrefs = prefs.node("info");
        _info = infoPrefs.get(INFO, "");

        Preferences staticPrefs = prefs.node("static");
        String[] staticChilds;
        try {
            staticChilds = staticPrefs.childrenNames();
        } catch (BackingStoreException ignore) {
            PreferencesDeleter pd = new PreferencesDeleter("Ein benutzer-definierter Darstellungstyp konnte nicht geladen werden.", staticPrefs);
            pd.run();
            return;
        }
        for (String staticChild : staticChilds) {
            Property property = PropertiesManager.INSTANCE.getProperty(staticChild);
            Preferences propertyPrefs = staticPrefs.node(property.getKey());
            _isStaticMap.put(property, true);
            final DynamicDOTItem dynamicDOTItem;
            try {
                dynamicDOTItem = new DynamicDOTItem(propertyPrefs, staticChild);
            } catch (BackingStoreException ignore) {
                PreferencesDeleter pd =
                    new PreferencesDeleter("Ein benutzer-definierter Darstellungstyp konnte nicht geladen werden.", propertyPrefs);
                pd.run();
                return;
            }
            _staticPropertyValues.put(property, dynamicDOTItem.getPropertyValue());
        }

        Preferences dynamicPrefs = prefs.node("dynamic");
        String[] dynamicChilds;
        try {
            dynamicChilds = dynamicPrefs.childrenNames();
        } catch (BackingStoreException ignore) {
            PreferencesDeleter pd = new PreferencesDeleter("Ein benutzer-definierter Darstellungstyp konnte nicht geladen werden.", dynamicPrefs);
            pd.run();
            return;
        }
        for (String dynamicChild : dynamicChilds) {
            Property property = PropertiesManager.INSTANCE.getProperty(dynamicChild);
            Preferences propertyPrefs = dynamicPrefs.node(property.getKey());
            _isStaticMap.put(property, false);
            DynamicDOTItemManager dynamicDOTItemManager = new DynamicDOTItemManager();
            _dynamicDOTItemManagers.put(property, dynamicDOTItemManager);

            String[] intervalNames;
            try {
                intervalNames = propertyPrefs.childrenNames();
            } catch (BackingStoreException ignore) {
                PreferencesDeleter pd =
                    new PreferencesDeleter("Ein benutzer-definierter Darstellungstyp konnte nicht geladen werden.", propertyPrefs);
                pd.run();
                return;
            }
            for (String child : intervalNames) {
                if (child.startsWith("interval")) {
                    Preferences objectItemPrefs = propertyPrefs.node(child);
                    DynamicDOTItem dynamicItem = null;
                    try {
                        dynamicItem = new DynamicDOTItem(objectItemPrefs, dynamicChild);
                    } catch (BackingStoreException ignore) {
                        PreferencesDeleter pd =
                            new PreferencesDeleter("Ein benutzer-definierter Darstellungstyp konnte nicht geladen werden.", objectItemPrefs);
                        pd.run();
                        return;
                    }
                    setValueOfDynamicProperty(null, property, dynamicItem, objectItemPrefs.getDouble(LOWER_BOUND, Double.MAX_VALUE),
                                              objectItemPrefs.getDouble(UPPER_BOUND, Double.MIN_VALUE));
                }
            }
        }
    }

    @Override
    public void deletePreferences(Preferences prefs) {
        Preferences classPrefs = prefs.node(getClass().getSimpleName());
        Preferences objectPrefs = classPrefs.node(getName());
        try {
            objectPrefs.removeNode();
        } catch (BackingStoreException e) {
            JOptionPane.showMessageDialog(null, "Das Löschen eines Darstellungstypen war nicht erfolgreich. " + e.toString(), "Fehlermeldung",
                                          JOptionPane.ERROR_MESSAGE);
        }
    }

    /**
     * Gibt ein TableModel für die übergebene Eigenschaft zurück.
     *
     * @param property die Eigenschaft
     *
     * @return ein TableModel
     */
    public TableModel getTableModel(Property property) {
        return _dynamicDOTItemManagers.get(property);
    }

    /**
     * Liefert die Menge von Zeilen-Indizes der Zeilen, die mit mindestens einer anderen einen Konflikt haben. Ein Konflikt liegt dann vor, wenn ein
     * dynamischer Wert sowohl zu der einer als auch der anderen Zeile passt; eine Zeile entspricht hier einem DisplayObjectTypeItem.
     *
     * @param property die Eigenschaft
     *
     * @return eine Menge von Zeilen-Indizes
     */
    @Nullable
    public Set<Integer> getConflictingRows(Property property) {
        final DynamicDOTItemManager dynamicDOTItemManager = _dynamicDOTItemManagers.get(property);
        if (dynamicDOTItemManager == null) {
            return null;
        }
        return dynamicDOTItemManager.getConflictingRows();
    }

    /**
     * Liefert die Einträge der Legende zurück.
     *
     * @return eine Teilbaum für die Legende
     */
    @Override
    public LegendTreeNodes getLegendTreeNodes() {
        LegendTreeNodes legendTreeNodes = new LegendTreeNodes();
        final List<Property> dynamicProperties = getDynamicProperties(null);
        final Property colorProperty = ColorProperty.getInstance();
        final int size;
        if (null != dynamicProperties) {
            size = dynamicProperties.size();
        } else {
            size = 0;
        }
        if (size == 0) {    // keine dynamischen Eigenschaften
            final String colorName = (String) getValueOfStaticProperty(null, colorProperty);
            if (null != colorName) {
                LegendTreeNodes.LegendTreeNode node = new LegendTreeNodes.LegendTreeNode(colorName, null, this);
                legendTreeNodes.add(node, 0);
            }
        } else {
            final Boolean colorPropertyIsStatic = isPropertyStatic(null, colorProperty);
            if ((size == 1) && !colorPropertyIsStatic) {
                // nur die Farbe ist dynamisch
                DynamicDOTItemManager dynamicDOTItemManager = _dynamicDOTItemManagers.get(colorProperty);
                for (TreeMap<Interval<Double>, DynamicDOTItem> treeMap : dynamicDOTItemManager.getTreeMaps()) {
                    for (DisplayObjectTypeItem dotItem : treeMap.values()) {
                        LegendTreeNodes.LegendTreeNode node = new LegendTreeNodes.LegendTreeNode(dotItem.getDescription(), null, this);
                        legendTreeNodes.add(node, 0);
                    }
                }
            } else {    // alles andere wird eher 'generisch' behandelt
                LegendTreeNodes.LegendTreeNode node = null;
                int depth = 0;
                int newDepth;
                for (Property property : dynamicProperties) {
                    newDepth = 0;
                    if (node != null) {
                        legendTreeNodes.add(node, depth - newDepth);
                    }
                    node = new LegendTreeNodes.LegendTreeNode(property.toString(), null, this);
                    depth = newDepth;
                    final DynamicDOTItemManager dynamicDOTItemManager = _dynamicDOTItemManagers.get(property);
                    final int itemSize = dynamicDOTItemManager.size();
                    for (int rowIndex = 0; rowIndex < itemSize; rowIndex++) {
                        newDepth = 1;
                        legendTreeNodes.add(node, depth - newDepth);
                        String description = dynamicDOTItemManager.get(rowIndex).getItem().getDescription();
                        node = new LegendTreeNodes.LegendTreeNode(description, null, this);
                        depth = newDepth;
                    }
                }
                newDepth = 0;
                if (node != null) {
                    legendTreeNodes.add(node, depth - newDepth);
                }
            }
        }
        return legendTreeNodes;
    }

    @Override
    public Set<DOTSubscriptionData> getSubscriptionData() {
        Set<DOTSubscriptionData> sdSet = new HashSet<>();
        for (final Entry<Property, Boolean> dotPropertyBooleanEntry : _isStaticMap.entrySet()) {
            if (!dotPropertyBooleanEntry.getValue()) {
                sdSet.addAll(_dynamicDOTItemManagers.get(dotPropertyBooleanEntry.getKey()).getSubscriptionData());
            }
        }
        return sdSet;
    }

    @Override
    public List<String> getAttributeNames(@Nullable String primitiveFormName, Property property, DOTSubscriptionData subscriptionData) {
        final DynamicDOTItemManager dynamicDOTItemManager = _dynamicDOTItemManagers.get(property);
        if (dynamicDOTItemManager == null) {
            return new ArrayList<>();
        }
        final List<String> list = dynamicDOTItemManager.getAttributeNames(subscriptionData);
        if (list != null) {
            return list;
        }
        return new ArrayList<>();
    }

    private boolean hasProperty(final Property property) {
        return _isStaticMap.containsKey(property);
    }

    @Override
    @SuppressWarnings("unchecked")
    public Set<String> getUsedColors() {
        Set<String> usedColors = new HashSet<>();
        Property colorProperty = ColorProperty.getInstance();
        if (hasProperty(colorProperty)) {
            if (isPropertyStatic(null, colorProperty)) {
                final String colorName = (String) getValueOfStaticProperty(null, colorProperty);
                if (null != colorName) {
                    usedColors.add(colorName.toLowerCase());
                }
            } else {
                DynamicDOTItemManager dynamicDOTItemManager = _dynamicDOTItemManagers.get(colorProperty);
                final int size = dynamicDOTItemManager.size();
                for (int index = 0; index < size; index++) {
	                final DOTItemManager<DynamicDOTItem>.DisplayObjectTypeItemWithInterval dotItemWithInterval = dynamicDOTItemManager.get(index);
                    final String colorName = (String) dotItemWithInterval.getItem().getPropertyValue();
                    usedColors.add(colorName.toLowerCase());
                }
            }
        }
        Property fillingProperty = FillingProperty.getInstance();
        if (hasProperty(fillingProperty)) {
            if (isPropertyStatic(null, fillingProperty)) {
                final String colorName = (String) getValueOfStaticProperty(null, fillingProperty);
                if (null != colorName) {
                    usedColors.add(colorName.toLowerCase());
                }
            } else {
                DynamicDOTItemManager dynamicDOTItemManager = _dynamicDOTItemManagers.get(fillingProperty);
                final int size = dynamicDOTItemManager.size();
                for (int index = 0; index < size; index++) {
	                final DOTItemManager<DynamicDOTItem>.DisplayObjectTypeItemWithInterval dotItemWithInterval = dynamicDOTItemManager.get(index);
                    final String colorName = (String) dotItemWithInterval.getItem().getPropertyValue();
                    usedColors.add(colorName.toLowerCase());
                }
            }
        }
        return usedColors;
    }

    @Override
    @Nullable
    public DisplayObjectTypeItem getDOTItemForValue(@Nullable String primitiveFormName, Property property, DOTSubscriptionData subscriptionData,
                                                    String attributeName, double value) {
        if (_isStaticMap.get(property)) {
            return null;
        }
        final DynamicDOTItemManager dynamicDOTItemManager = _dynamicDOTItemManagers.get(property);
        if (dynamicDOTItemManager == null) {
            return null;
        }
        final List<String> list = dynamicDOTItemManager.getAttributeNames(subscriptionData);
        if (list == null) {
            return null;
        }
        if (!list.contains(attributeName)) {
            return null;
        }
        final TreeMap<Interval<Double>, DynamicDOTItem> treeMap =
            dynamicDOTItemManager.get(dynamicDOTItemManager.getKeyString(subscriptionData, attributeName));
        if (treeMap == null) {
            return null;
        }
        Interval<Double> valInterval = new Interval<>(value, value);
        final Entry<Interval<Double>, DynamicDOTItem> floorEntry = treeMap.floorEntry(valInterval);
        if (floorEntry != null) {
            final Interval<Double> floorKey = floorEntry.getKey();
            if ((floorKey.getLowerBound() <= value) && (value <= floorKey.getUpperBound())) {
                return floorEntry.getValue();
            }
        }
        final Entry<Interval<Double>, DynamicDOTItem> ceilingEntry = treeMap.ceilingEntry(valInterval);
        if (ceilingEntry != null) {
            final Interval<Double> ceilingKey = ceilingEntry.getKey();
            if ((ceilingKey.getLowerBound() <= value) && (value <= ceilingKey.getUpperBound())) {
                return ceilingEntry.getValue();
            }
        }
        return null;
    }

    @Override
    @Nullable
    public DisplayObjectTypeItem getDisplayObjectTypeItemForState(@Nullable final String primitiveFormName, final Property property,
                                                                  final DOTSubscriptionData subscriptionData, final DataState dataState) {
        if (_isStaticMap.get(property)) {
            return null;
        }
        final DynamicDOTItemManager dynamicDOTItemManager = _dynamicDOTItemManagers.get(property);
        final String keyString1;
        if (dataState.equals(DataState.NO_DATA)) {
            keyString1 = dynamicDOTItemManager.getKeyString(subscriptionData, DynamicDefinitionComponent.KEINE_DATEN_STATUS);
        } else if (dataState.equals(DataState.NO_SOURCE)) {
            keyString1 = dynamicDOTItemManager.getKeyString(subscriptionData, DynamicDefinitionComponent.KEINE_QUELLE_STATUS);
        } else if (dataState.equals(DataState.NO_RIGHTS)) {
            keyString1 = dynamicDOTItemManager.getKeyString(subscriptionData, DynamicDefinitionComponent.KEINE_RECHTE_STATUS);
        } else {
            keyString1 = null;
        }
        if (keyString1 != null) {    // einer der Substati hat gezogen ...
            final TreeMap<Interval<Double>, DynamicDOTItem> treeMap1 = dynamicDOTItemManager.get(keyString1);
            if ((treeMap1 != null) && (treeMap1.size() == 1)) { // ... und ist definiert
                return treeMap1.values().toArray(new DisplayObjectTypeItem[1])[0];
            }
        }
        // den übergreifenden Status überprüfen
        final String keyString2 = dynamicDOTItemManager.getKeyString(subscriptionData, DynamicDefinitionComponent.LEERE_DATEN_STATUS);
        final TreeMap<Interval<Double>, DynamicDOTItem> treeMap2 = dynamicDOTItemManager.get(keyString2);
        if ((treeMap2 != null) && (treeMap2.size() == 1)) { // ... er ist definiert
            return treeMap2.values().toArray(new DisplayObjectTypeItem[1])[0];
        } else {    // ... dann bleibt nur der Default
            return DynamicDOTItem.NO_DATA_ITEM;
        }
    }

    @Override
    public void displayObjectTypeAdded(DisplayObjectType displayObjectType) {
    }

    @Override
    public void displayObjectTypeChanged(DisplayObjectType displayObjectType) {
        if (displayObjectType.equals(this)) {
            return;
        }
        if (displayObjectType.getName().equals(_name)) {
            DefaultDisplayObjectType defaultDisplayObjectType = (DefaultDisplayObjectType) displayObjectType;
            _info = displayObjectType.getInfo();
            _isStaticMap.clear();
            for (Property property : defaultDisplayObjectType._isStaticMap.keySet()) {
                _isStaticMap.put(property, defaultDisplayObjectType._isStaticMap.get(property));
            }
            _staticPropertyValues.clear();
            for (Property property : defaultDisplayObjectType._staticPropertyValues.keySet()) {
                _staticPropertyValues.put(property, defaultDisplayObjectType._staticPropertyValues.get(property));
            }
            _dynamicDOTItemManagers.clear();
            for (Property property : defaultDisplayObjectType._dynamicDOTItemManagers.keySet()) {
                _dynamicDOTItemManagers.put(property, defaultDisplayObjectType._dynamicDOTItemManagers.get(property));
            }
        }
    }

    @Override
    public void displayObjectTypeRemoved(String displayObjectTypeName) {
    }

    @SuppressWarnings("NonFinalFieldReferenceInEquals")
    @Override
    public boolean equals(Object o) {
	    if (!(o instanceof DefaultDisplayObjectType d)) {
            return false;
        }
        if (!getDisplayObjectTypePlugin().getName().equals(d.getDisplayObjectTypePlugin().getName())) {
            return false;    // Subklasse stimmt nicht!
        }
        if (!_name.equals(d._name)) {
            return false;
        }
        if (!_info.equals(d._info)) {
            return false;
        }
        // Erst die Größen vergleichen ...
        if (_isStaticMap.size() != d._isStaticMap.size()) {
            return false;
        }
        if (_staticPropertyValues.size() != d._staticPropertyValues.size()) {
            return false;
        }
        if (_dynamicDOTItemManagers.size() != d._dynamicDOTItemManagers.size()) {
            return false;
        }
        // und dann die Container:
        for (final Entry<Property, Boolean> dotPropertyBooleanEntry : _isStaticMap.entrySet()) {
            final boolean b1 = dotPropertyBooleanEntry.getValue();
            final boolean b2 = d._isStaticMap.get(dotPropertyBooleanEntry.getKey());
            if (b1 != b2) {
                return false;
            }
        }
        for (final Entry<Property, Object> dotPropertyObjectEntry : _staticPropertyValues.entrySet()) {
            final Object dObject = d._staticPropertyValues.get(dotPropertyObjectEntry.getKey());
            if (dObject == null) {
                return false;
            }
            final Object object = dotPropertyObjectEntry.getValue();
            if (dotPropertyObjectEntry.getKey() == ColorProperty.getInstance()) {
                String dColor = (String) dObject;
                String color = (String) object;
                if (!dColor.equals(color)) {
                    return false;
                }
            } else if (dotPropertyObjectEntry.getKey() == DistanceProperty.getInstance()) {
                Integer dInt = (Integer) dObject;
                Integer myInt = (Integer) object;
                if (!dInt.equals(myInt)) {
                    return false;
                }
            } else if (dotPropertyObjectEntry.getKey() == StrokeWidthProperty.getInstance()) {
                Double dDouble = (Double) dObject;
                Double myDouble = (Double) object;
                if (!dDouble.equals(myDouble)) {
                    return false;
                }
            }
        }
        for (final Entry<Property, DynamicDOTItemManager> dotPropertyDynamicDOTItemManagerEntry : _dynamicDOTItemManagers.entrySet()) {
            final DynamicDOTItemManager dMan = d._dynamicDOTItemManagers.get(dotPropertyDynamicDOTItemManagerEntry.getKey());
            if (dMan == null) {
                return false;
            }
            final DynamicDOTItemManager myMan = dotPropertyDynamicDOTItemManagerEntry.getValue();
            if (!dMan.equals(myMan)) {
                return false;
            }
        }
        return true;
    }

    @SuppressWarnings("NonFinalFieldReferencedInHashCode")
    @Override
    public int hashCode() {
        return _name.hashCode();
    }

    @Override
    public String toString() {
        StringBuilder s = new StringBuilder();
        s.append("[").append(getClass().getName()).append(":").append(_name).append(" (").append(_info).append("), [statisch:");
        for (final Entry<Property, Boolean> dotPropertyBooleanEntry : _isStaticMap.entrySet()) {
            if (dotPropertyBooleanEntry.getValue()) {
                s.append(dotPropertyBooleanEntry.getKey().toString()).append("=")
                    .append(_staticPropertyValues.get(dotPropertyBooleanEntry.getKey()).toString());
            }
        }
        s.append("], [dynamisch:");

        for (final Entry<Property, Boolean> dotPropertyBooleanEntry : _isStaticMap.entrySet()) {
            if (!dotPropertyBooleanEntry.getValue()) {
                DynamicDOTItemManager dynamicDOTItemManager = _dynamicDOTItemManagers.get(dotPropertyBooleanEntry.getKey());
                for (TreeMap<Interval<Double>, DynamicDOTItem> treeMap : dynamicDOTItemManager.getTreeMaps()) {
                    for (final Entry<Interval<Double>, DynamicDOTItem> intervalDynamicDOTItemEntry : treeMap.entrySet()) {
                        s.append("[Interval:").append(intervalDynamicDOTItemEntry.getKey().getLowerBound());
                        s.append(",").append(intervalDynamicDOTItemEntry.getKey().getUpperBound()).append("]");
                        DynamicDOTItem dotfv = intervalDynamicDOTItemEntry.getValue();
                        s.append("[").append(dotfv.toString()).append("]");
                    }
                    s.append("]");
                }
            }
        }
        s.append("]]");
        return s.toString();
    }

    /**
     * Jede nicht-abstrakte Subklasse muss ihre Plugin-Selbstbeschreibung angeben können.
     *
     * @return die zugehörige Plugin-Selbstbeschreibung
     */
    @Override
    public abstract DisplayObjectTypePlugin getDisplayObjectTypePlugin();

    /*
     * Implementiert Comparable<DisplayObjectType>.
     */
    @Override
    public int compareTo(DisplayObjectType o) {
        return getName().toLowerCase().compareTo(o.getName().toLowerCase());
    }
}
