/*
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.pluginInterfaces;

import de.bsvrz.dav.daf.main.DataState;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.kappich.pat.gnd.areaPlugin.DOTArea;
import de.kappich.pat.gnd.complexPlugin.DOTComplex;
import de.kappich.pat.gnd.displayObjectToolkit.DOTSubscriptionData;
import de.kappich.pat.gnd.displayObjectToolkit.DisplayObject;
import de.kappich.pat.gnd.displayObjectToolkit.OnlineDisplayObject;
import de.kappich.pat.gnd.gnd.LegendTreeNodes;
import de.kappich.pat.gnd.linePlugin.DOTLine;
import de.kappich.pat.gnd.pointPlugin.DOTPoint;
import de.kappich.pat.gnd.properties.Property;
import java.util.List;
import java.util.Set;
import java.util.prefs.Preferences;

/**
 * Das Interface für die Darstellungstypen der Plugins.
 * <p>
 * Der Name eines Darstellungstypen wird bei der Definition vom Benutzer festgelegt. Er wird als eindeutiges Erkennungszeichen in den
 * Verwaltungsdialogen verwendet, und auch bei der Anzeige in der Legende. Außerdem hat ein Darstellungstyp eine 'Info', die als Tooltipp etwa in der
 * Legende benutzt wird.
 * <p>
 * Ein Darstellungstyp legt fest wie die {@link DisplayObject DisplayObjects} eines Layers dargestellt werden. Darstellungstypen können {@link
 * DOTPoint.PrimitiveForm Grundfiguren} besitzen, müssen es aber nicht. Beispiele für Darstellungstypen ohne Grundfiguren sind {@link DOTArea}, {@link
 * DOTComplex} und {@link DOTLine}. Bei diesen Klassen hängen alle {@link Property Eigenschaften} der Visualisierung (z.B. die Farbe) direkt am
 * Darstellungstypen. Anders verhält es sich bei der Klasse {@link DOTPoint}: dieser Darstellungstyp hat selbst keine Eigenschaften, sondern
 * ausschließlich benutzerdefinierte Grundfiguren (z.B. ein Rechteck festgelegter Größe) und nur an diesen hängen die Eigenschaften. Bei der
 * Implementation sollte der Zugriff auf Eigenschaften, die direkt am Darstellungstypen hängen, durch {@code null} als Wert für die Grundfigur
 * geschehen.
 * <p>
 * Jede Grundfigur hat einen Typ, der einerseits definiernde Größen (z.B. den Radius bei dem Typ Kreis), aber auch die möglichen
 * Visualisierungs-Eigenschaften festlegt (z.B. die Füllfarbe).
 * <p>
 * Eine Visualisierungs-Eigenschaft ist entweder statisch, d.h. unveränderbar, oder dynamisch, d.h. sie verändert sich in Abhängigkeit von
 * Online-Daten.
 *
 * @author Kappich Systemberatung
 */
public interface DisplayObjectType extends Comparable<DisplayObjectType> {

    /**
     * Getter für den Namen.
     *
     * @return der Name
     */
    String getName();

    /**
     * Setter für den Namen.
     *
     * @param name der neue Name
     */
    void setName(String name);

    /**
     * Getter für die Info.
     *
     * @return die Kurzinfo
     */
    String getInfo();

    /**
     * Setter für die Info.
     *
     * @param info die Kurzinfo
     */
    void setInfo(String info);

    /**
     * Zugriff auf alle auftretenden Grundfigurnamen.
     *
     * @return die Menge aller Grundfigurnamen
     */
    Set<String> getPrimitiveFormNames();

    /**
     * Gibt den Grundfigurtyp der Grundfigur zurück.
     *
     * @param primitiveFormName der Name einer Grundfigur
     *
     * @return der Typ der Grundfigur
     */
    String getPrimitiveFormType(String primitiveFormName);

    /**
     * Gibt die Kurzinfo zu der Grundfigur zurück.
     *
     * @param primitiveFormName der Name einer Grundfigur
     *
     * @return die Kurzinfo zu der Grundfigur
     */
    String getPrimitiveFormInfo(String primitiveFormName);

    /**
     * Löscht die entsprechende Grundfigur.
     *
     * @param primitiveFormName der Name einer Grundfigur
     */
    void removePrimitiveForm(String primitiveFormName);

    /**
     * Zugriff auf alle dynamischen Properties der Grundfigur.
     *
     * @param primitiveFormName der Name einer Grundfigur
     *
     * @return die Liste aller dynamischen Eigenschaften der Grundfigur
     */
    List<Property> getDynamicProperties(@Nullable String primitiveFormName);

    /**
     * Ist die Property zu der als Object übergebenen Grundfigur statisch, so erhält man {@code true} zurück; andernfalls ist die Eigenschaft
     * dynamisch und man erhält {@code false}.
     *
     * @param primitiveFormName der Name einer Grundfigur oder {@code null}
     * @param property          eine Eigenschaft
     *
     * @return ist die Eigenschaft statisch?
     */
    boolean isPropertyStatic(@Nullable String primitiveFormName, Property property);

    /**
     * Setzt die Eigenschaft statisch bzw dynamisch zu sein der übergebenen Eigenschaft, die gegebenenfalls zu der genannten Grundfigur gehört. Diese
     * Methode sollte so implementiert werden, dass sie beim Ändern die nicht mehr gültigen Werte der Eigenschaft nicht löscht (dadurch wird es
     * möglich, dass der Benutzer diese zwischen statisch und dynamisch hin- und herschaltet ohne seine vorherigen Einstellungen zu verlieren).
     *
     * @param primitiveFormName der Name einer Grundfigur oder {@code null}
     * @param property          eine Eigenschaft
     * @param b                 der neue Wert
     */
    void setPropertyStatic(@Nullable String primitiveFormName, Property property, boolean b);

    /**
     * Gibt den Wert der übergebenen Property zurück, die gegebenenfalls zu der genannten Grundfigur gehört.
     *
     * @param primitiveFormName der Name einer Grundfigur oder {@code null}
     * @param property          eine Eigenschaft
     *
     * @return der Wert der Eigenschaft
     */
    Object getValueOfStaticProperty(@Nullable String primitiveFormName, Property property);

    /**
     * Setzt den Wert der übergebenen Property, die gegebenenfalls zu der genannten Grundfigur gehört. Diese Methode sollte so implementiert werden,
     * dass sie auch auch dann den übergebenen Wert behält, wenn die Property aktuell nicht statisch ist (dadurch wird es möglich, dass der Benutzer
     * diese zwischen statisch und dynamisch hin- und herschaltet ohne seine vorherigen Einstellungen zu verlieren).
     *
     * @param primitiveFormName der Name einer Grundfigur oder {@code null}
     * @param property          eine Eigenschaft
     * @param value             der neue Wert
     */
    void setValueOfStaticProperty(@Nullable String primitiveFormName, Property property, Object value);

    /**
     * Setzt den Wert der übergebenen Property, die gegebenenfalls zu der genannten Grundfigur gehört, für das übergebene Intervall auf das übergebene
     * DisplayObjectTypeItem. Diese Methode sollte so implementiert werden, dass sie auch auch dann den übergebenen Wert behält, wenn die Property
     * aktuell nicht dynamisch ist (dadurch wird es möglich, dass der Benutzer diese zwischen statisch und dynamisch hin- und herschaltet ohne seine
     * vorherigen Einstellungen zu verlieren).
     *
     * @param primitiveFormName der Name einer Grundfigur oder {@code null}
     * @param property          eine Eigenschaft
     * @param dItem             ein Item
     * @param lowerBound        die untere Schranke
     * @param upperBound        die obere Schranke
     */
    void setValueOfDynamicProperty(@Nullable String primitiveFormName, Property property, DisplayObjectTypeItem dItem, @Nullable Double lowerBound,
                                   @Nullable Double upperBound);

    /**
     * Macht eine tiefe Kopie des DisplayObjectTypes und setzt den Namen um, falls der übergebene String nicht {@code null} ist. Diese Methode wird
     * beim Erstellen und Bearbeiten von Darstellungstypen verwendet: dem Bearbeitungs-Dialog wird eine tiefe Kopie übergeben und alle Änderungen
     * werden an diesem Objekt durchgeführt.
     *
     * @param name der neue Name oder {@code null}
     *
     * @return eine Kopie
     */
    DisplayObjectType getCopy(@Nullable String name);

    /**
     * Speichert die Informationen des DisplayObjectTypes unter dem übergebenen Knoten.
     *
     * @param prefs der Knoten, unter dem die Speicherung durchgeführt werden soll
     */
    void putPreferences(Preferences prefs);

    /**
     * Initialisiert den DisplayObjectType aus dem übergebenen Knoten.
     *
     * @param prefs der Knoten, unter dem die Initialisierung durchgeführt werden soll
     */
    void initializeFromPreferences(Preferences prefs);

    /**
     * Löscht den DisplayObjectType unter dem übergebenen Knoten.
     *
     * @param prefs der Knoten, unter dem die Löschung durchgeführt werden soll
     */
    void deletePreferences(Preferences prefs);

    /**
     * Gibt die Selebstbeschreibung des Plugins, zu dem dieser DisplayObjectType gehört, zurück.
     *
     * @return die Selbstbeschreibung
     */
    DisplayObjectTypePlugin getDisplayObjectTypePlugin();

    /**
     * Erzeugt den Teilbaum der Legende, für diesen Darstellungstyp.
     *
     * @return der Teilbaum der Legende
     */
    LegendTreeNodes getLegendTreeNodes();

    /**
     * Gibt alle Anmeldungen, die dieser DisplayObjectTyp benötigt, zurück.
     *
     * @return alle Anmeldungen
     */
    Set<DOTSubscriptionData> getSubscriptionData();

    /**
     * Gibt die Attributnamen, für die Werte benötigt werden, zu der übergebenen Eigenschaft und der übergebenen Anmeldung zurück.
     *
     * @param primitiveFormName der Name einer Grundfigur oder {@code null}
     * @param property          eine Eigenschaft
     * @param subscriptionData  eine Anmeldung
     *
     * @return alle Attributname
     */
    List<String> getAttributeNames(@Nullable String primitiveFormName, Property property, DOTSubscriptionData subscriptionData);

    /**
     * Gibt die Namen aller von diesem DisplayObject verwendeten Farben zurück.
     *
     * @return Die Menge aller Namen aller benutzten Farben
     */
    Set<String> getUsedColors();

    /**
     * Ist der Rückgabewert nicht {@code null}, so ist dieser {@link DisplayObjectTypeItem} für die übergebenen Daten anwendbar. Diese Methode wird
     * von einem {@link OnlineDisplayObject} aufgerufen, wenn neue Online-Daten vorliegen, die eine Änderung der Visualisierungs-Eigenschaft zur Folge
     * haben könnte. Der im Rückgabewert enthaltene Wert (z.B. eine Farbe) wird dann vom {@link DisplayObjectPainter Painter} zur Visualisierung
     * verwendet.
     *
     * @param primitiveFormName der Name einer Grundfigur oder {@code null}
     * @param property          die Eigenschaft
     * @param subscriptionData  Attributgruppe und Aspekt
     * @param attributeName     Attribut
     * @param value             Wert des Attributs
     *
     * @return das Item oder {@code null}
     */
    DisplayObjectTypeItem getDOTItemForValue(@Nullable String primitiveFormName, Property property, DOTSubscriptionData subscriptionData,
                                             String attributeName, double value);

    /**
     * Ist der Rückgabewert nicht {@code null}, so ist dieser {@link DisplayObjectTypeItem} für die übergebenen Daten anwendbar. Diese Methode wird
     * von einem {@link DisplayObject} aufgerufen, wenn zur gegebenen {@link DOTSubscriptionData Anmeldung} neue Daten geschickt wurden, die aber
     * keine Werte für die Attribute enthalten, sondern Informationen über den {@link DataState Zustand}. Der im Rückgabewert enthaltene Wert (z.B.
     * eine Farbe) wird dann vom {@link DisplayObjectPainter Painter} zur Visualisierung verwendet.
     *
     * @param primitiveFormName der Name einer Grundfigur oder {@code null}
     * @param property          die Eigenschaft
     * @param subscriptionData  Attributgruppe und Aspekt
     * @param dataState         Zustand des Datensatzes
     *
     * @return das Item oder {@code null}
     */
    DisplayObjectTypeItem getDisplayObjectTypeItemForState(@Nullable final String primitiveFormName, final Property property,
                                                           final DOTSubscriptionData subscriptionData, final DataState dataState);

    /**
     * Ein Interface für die kleinste Einheit beim Zuordnen von Anmeldedaten (Attributgruppe, Aspekt, Attribut) zu Eigenschaftswerten und deren
     * Beschreibung.
     *
     * @author Kappich Systemberatung
     */
    interface DisplayObjectTypeItem {
        /**
         * Getter der Attributgruppe.
         *
         * @return die Attributgruppe
         */
        String getAttributeGroup();

        /**
         * Getter des Aspekts.
         *
         * @return der Aspekt
         */
        String getAspect();

        /**
         * Getter für den Attributnamen.
         *
         * @return der Attributname
         */
        String getAttributeName();

        /**
         * Getter für die Beschreibung.
         *
         * @return die Beschreibung
         */
        String getDescription();

        /**
         * Getter für den aktuellen Wert der Eigenschaft, für die dieser DisplayObjectTypeItem Daten verwaltet.
         *
         * @return der Wert der Eigenscahft
         */
        Object getPropertyValue();

        /**
         * Erstellt eine tiefe Kopie des DisplayObjectTypeItems.
         *
         * @return die Kopie
         */
        DisplayObjectTypeItem getCopy();
    }
}
