/*
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.pluginInterfaces;

import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.kappich.pat.gnd.displayObjectToolkit.DOTDefinitionDialog;
import de.kappich.pat.gnd.properties.Property;

/**
 * Das Interface zur Selbstbeschreibung eines GND-Plugins. Ein Plugin besteht aus der Implementation der vier Interfaces in diesem Package. Die
 * Implementation dieses Interfaces liefert den Einstieg in diese Implementationen und weitere Meta-Informationen des Plugins.
 * <p>
 * Jede implementierende Klassen muss einen öffentlichen Konstruktor mit leerer Argumentenliste haben, damit Class.newInstance() aufgerufen werden
 * kann. Außerdem sollte dieser Konstruktor leichtgewichtig sein, damit er nicht die Performanz senkt.
 *
 * @author Kappich Systemberatung
 */
public interface DisplayObjectTypePlugin {

    /**
     * Gibt den eindeutigen Namen des Plugins zurück. Die Namen 'Fläche', 'Komplex', 'Linie' und 'Punkt' sind bereits vergeben.
     *
     * @return der eindeutige Name des Plugins
     */
    String getName();

    /**
     * Gibt den Darstellungstypen des Plugins zurück, das ist ein Implementation von {@link DisplayObjectType}.
     *
     * @return ein Objekt des Darstellungstyps dieses Plugins
     */
    DisplayObjectType getDisplayObjectType();

    /**
     * Gibt ein {@code DOTDefinitionPanel} zur Definition und Bearbeitung von Darstellungstypen dieses Plugins zurück. Dabei wird der übergeordnete
     * Frame dem Dialog bekanntgemacht und deshalb als Parameter übergeben.
     *
     * @param dotDefinitionDialog der übergeordnete Frame
     *
     * @return den Definitions- und Bearbeitungsdialog
     */
    DOTDefinitionPanel getPanel(DOTDefinitionDialog dotDefinitionDialog);

    /**
     * Gibt den Initializer zurück, der die Darstellungsobjekte initialisiert.
     *
     * @return ein Initialzer
     */
    DisplayObjectsInitializer getInitializer();

    /**
     * Gibt einen Painter zurück, der Darstellungsobjekte mit Darstellungsobjekttypen dieses Plugins, zeichnen kann.
     *
     * @return ein Objekt des Painters dieses Plugins
     */
    DisplayObjectPainter getPainter();

    /**
     * Gibt die möglichen Typen der Grundfiguren, die gezeichnet werden können, zurück, z.B. Rechteck, Kreis, Text usw.
     * <p>
     * Ein Plugin kann Grundfiguren besitzen, muss aber nicht. Siehe die Erläuterungen {@link DisplayObjectType hier}.
     *
     * @return die möglichen Grundfigurtypen des Plugins
     */
    String[] getPrimitiveFormTypes();

    /**
     * Gibt die Visualisierungs-Eigenschaften des Grundfigurtyps o zurück. Kann mit null aufgerufen werden, und gibt dann globale Eigenschaften
     * zurück. Siehe die Erläuterungen {@link DisplayObjectType hier}.
     *
     * @param o ein Grundfigurtyp oder null
     *
     * @return die Eigenschaften der Grundfigur oder des Darstellungstyps selbst
     */
    Property[] getProperties(@Nullable Object o);

    /**
     * Gibt 'typ.fläche', 'typ.linie', 'typ.punkt' oder 'typ.geoReferenzObject' zurück, je nachdem, ob das Plugin für Systemobjekte so eingeschränkt
     * werden kann (in den ersten drei Fällen) oder nicht (im letzten Fall).
     *
     * @return der Geometrietyp
     */
    String getGeometryType();

    /**
     * Prüft, ob der übergebene SystemObjectType von dem Plugin unterstützt wird.
     *
     * @param configuration    die Konfiguration
     * @param systemObjectType ein SystemObjectType
     *
     * @return {@code true} genau dann, wenn der Typ vom Plugin unterstützt wird
     */
    boolean isSystemObjectTypeSupported(DataModel configuration, SystemObjectType systemObjectType);

    /**
     * Sind innerhalb des Plugins Aneldungen auf dynamische Daten möglich?
     *
     * @return ist Dynamik möglich
     */
    boolean isDynamicsPossible();

    /**
     * Nicht alle Plugin-Implementationen wickeln Aufgaben über das MapScale-Listening ab. Wenn dies nicht benötigt wird, so können Resourcen geschont
     * werden.
     *
     * @return
     */
    boolean isMapScaleListeningNecessary();
}
