/*
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.pointPlugin;

import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.kappich.pat.gnd.pluginInterfaces.DOTDefinitionPanel;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectPainter;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectType;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectTypePlugin;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectsInitializer;
import de.kappich.pat.gnd.pointPlugin.DOTPoint.PrimitiveFormType;
import de.kappich.pat.gnd.properties.ColorProperty;
import de.kappich.pat.gnd.properties.DiameterProperty;
import de.kappich.pat.gnd.properties.FillingProperty;
import de.kappich.pat.gnd.properties.Property;
import de.kappich.pat.gnd.properties.StrokeWidthProperty;
import de.kappich.pat.gnd.properties.TextProperty;
import de.kappich.pat.gnd.properties.TextSizeProperty;
import de.kappich.pat.gnd.properties.TextStyleProperty;
import de.kappich.pat.gnd.properties.TransparencyProperty;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * Die Selbstbeschreibung des Punkte-Plugins.
 *
 * @author Kappich Systemberatung
 */
public class DOTPointPlugin implements DisplayObjectTypePlugin {

    private static Set<String> _supportedSystemObjectTypePIDs;

    /**
     * Konstruiert eine Selbstbeschreibung des Plugins.
     */
    @SuppressWarnings("RedundantNoArgConstructor")
    public DOTPointPlugin() {
        super();
    }

    private static void initSupportedSystemObjectTypePIDs(DataModel configuration) {
        final List<SystemObjectType> geoReferenceObjectTypes = new ArrayList<>();
        final SystemObjectType geoReferenceObjectType = configuration.getType("typ.punkt");
        geoReferenceObjectTypes.add(geoReferenceObjectType);
        for (int i = 0; i < geoReferenceObjectTypes.size(); i++) {
            geoReferenceObjectTypes.addAll(geoReferenceObjectTypes.get(i).getSubTypes());
        }
        _supportedSystemObjectTypePIDs = new HashSet<>();
        for (SystemObjectType systemObjectType : geoReferenceObjectTypes) {
            _supportedSystemObjectTypePIDs.add(systemObjectType.getPid());
        }
    }

    @Override
    public String getName() {
        return "Punkt";
    }

    @Override
    public DOTDefinitionPanel getPanel(final de.kappich.pat.gnd.displayObjectToolkit.DOTDefinitionDialog dotDefinitionDialog) {
        return new DOTPointPanel(dotDefinitionDialog);
    }

    @Override
    public DisplayObjectType getDisplayObjectType() {
        return new DOTPoint();
    }

    @Override
    public DisplayObjectsInitializer getInitializer() {
        return new PointInitializer();
    }

    @Override
    public DisplayObjectPainter getPainter() {
        return new DOTPointPainter();
    }

    @Override
    public String[] getPrimitiveFormTypes() {
        return new String[] {"Punkt", "Rechteck", "Kreis", "Halbkreis", "Textdarstellung"};
    }

    @Override
    @Nullable
    public Property[] getProperties(Object o) {
        if (o == null) {
            return new Property[] {};
        }
	    if (o instanceof PrimitiveFormType pft) {
            if (pft.equals(PrimitiveFormType.PUNKT)) {
                return new Property[] {DiameterProperty.getInstance(), ColorProperty.getInstance()};
            } else if (pft.equals(PrimitiveFormType.RECHTECK)) {
                return new Property[] {FillingProperty.getInstance(), StrokeWidthProperty.getInstance(), TransparencyProperty.getInstance()};
            } else if (pft.equals(PrimitiveFormType.KREIS)) {
                return new Property[] {FillingProperty.getInstance(), StrokeWidthProperty.getInstance(), TransparencyProperty.getInstance()};
            } else if (pft.equals(PrimitiveFormType.HALBKREIS)) {
                return new Property[] {FillingProperty.getInstance(), StrokeWidthProperty.getInstance(), TransparencyProperty.getInstance()};
            } else if (pft.equals(PrimitiveFormType.TEXTDARSTELLUNG)) {
                return new Property[] {ColorProperty.getInstance(),
                                       TextSizeProperty.getInstance(),
                                       TextProperty.getInstance(),
                                       TextStyleProperty.getInstance()};
            }
        }
	    if (!(o instanceof String s)) {
            return null;
        }
        if (s.equals(PrimitiveFormType.PUNKT.getName())) {
            return new Property[] {DiameterProperty.getInstance(), ColorProperty.getInstance()};
        } else if (s.equals(PrimitiveFormType.RECHTECK.getName())) {
            return new Property[] {FillingProperty.getInstance(), StrokeWidthProperty.getInstance(), TransparencyProperty.getInstance()};
        } else if (s.equals(PrimitiveFormType.KREIS.getName())) {
            return new Property[] {FillingProperty.getInstance(), StrokeWidthProperty.getInstance(), TransparencyProperty.getInstance()};
        } else if (s.equals(PrimitiveFormType.HALBKREIS.getName())) {
            return new Property[] {FillingProperty.getInstance(), StrokeWidthProperty.getInstance(), TransparencyProperty.getInstance()};
        } else if (s.equals(PrimitiveFormType.TEXTDARSTELLUNG.getName())) {
            return new Property[] {ColorProperty.getInstance(),
                                   TextSizeProperty.getInstance(),
                                   TextProperty.getInstance(),
                                   TextStyleProperty.getInstance()};
        }
        return null;
    }

    @Override
    public String getGeometryType() {
        return "typ.punkt";
    }

    @Override
    public boolean isSystemObjectTypeSupported(DataModel configuration, SystemObjectType systemObjectType) {
        if (_supportedSystemObjectTypePIDs == null) {
            initSupportedSystemObjectTypePIDs(configuration);
        }
        return _supportedSystemObjectTypePIDs.contains(systemObjectType.getPid());
    }

    @Override
    public boolean isDynamicsPossible() {
        return true;
    }

    @Override
    public boolean isMapScaleListeningNecessary() {
        return true;
    }

}
