/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.properties;

import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import java.text.DecimalFormat;
import java.util.prefs.Preferences;

/**
 * {@code DistanceRasterType} ist eine Enumeration für die verschiedenen Abstandsstufen des Kilometrierungs- und des Stationierungs-Plugins.
 *
 * @author Kappich Systemberatung
 */
public enum DistanceRasterType implements PropertyValue {

    /**
     * {@code TEN} steht für die 10-Meter-Stufe
     */
    TEN(10, new DecimalFormat("0.00")),
    /**
     * {@code FIFTY} steht für die 50-Meter-Stufe
     */
    FIFTY(50, new DecimalFormat("0.00")),
    /**
     * {@code ONE_HUNDRED} steht für die 100-Meter-Stufe
     */
    ONE_HUNDRED(100, new DecimalFormat("0.0")),
    /**
     * {@code FIVE_HUNDRED} steht für die 500-Meter-Stufe
     */
    FIVE_HUNDRED(500, new DecimalFormat("0.0")),
    /**
     * {@code ONE_THOUSAND} steht für die 1-Kilometer-Stufe
     */
    ONE_THOUSAND(1000, new DecimalFormat("0")),
    /**
     * {@code FIVE_THOUSAND} steht für die 5-Kilometer-Stufe
     */
    FIVE_THOUSAND(5000, new DecimalFormat("0")),
    /**
     * {@code TEN_THOUSAND} steht für die 10-Kilometer-Stufe
     */
    TEN_THOUSAND(10000, new DecimalFormat("0")),
    /**
     * {@code FIFTY_THOUSAND} steht für die 50-Kilometer-Stufe
     */
    FIFTY_THOUSAND(50000, new DecimalFormat("0"));

    private final int _value;
    private final DecimalFormat _format;

    DistanceRasterType(int value, DecimalFormat format) {
        _value = value;
        _format = format;
    }

    /**
     * @param value ein Wert einer Kilometrierungsstufe
     *
     * @return die zugehörige Kilometrierungsstufe oder {@code null}
     */
    @Nullable
    public static DistanceRasterType getType(int value) {
        for (DistanceRasterType type : DistanceRasterType.values()) {
            if (type.getValue() == value) {
                return type;
            }
        }
        return null;
    }

    @Override
    public void putPreferences(final Preferences prefs) {
        prefs.putInt("PROPERTY_VALUE", getValue());
    }

    /**
     * @return den Wert der Kilometrierungsstufe
     */
    public int getValue() {
        return _value;
    }

    /**
     * @return das Dezimalformat zur Darstellung des Werts der Kilometrierungsstufe
     */
    public DecimalFormat getFormat() {
        return _format;
    }

    @Override
    public String toString() {
        if (_value < 1000) {
            return _value + " m";
        } else {
            return _value / 1000 + " km";
        }
    }
}
