/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.rnPlugin;

import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.kappich.pat.gnd.displayObjectToolkit.DOTCollection;
import de.kappich.pat.gnd.displayObjectToolkit.DisplayObject;
import de.kappich.pat.gnd.displayObjectToolkit.PrimitiveFormPropertyPair;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectPainter;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectType;
import de.kappich.pat.gnd.utils.PointWithAngle;
import java.awt.Rectangle;
import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * {@code RnDisplayObject} ist die Implementation des Interfaces {@link DisplayObject} im Km-Plugin.
 *
 * @author Kappich Systemberatung
 */
public class RnDisplayObject implements DisplayObject {

    private final Map<Integer, Rectangle> _boundingRectangles = new HashMap<>();
    private final DOTCollection _dotCollection;
    private final int _routeNumber;
    private final Category _category;
    private final List<Object> _coordinates;
    private DisplayObjectPainter _painter;
    private int _defaultType;

    /**
     * Ein Konstruktor mit Painter, DOTCollection und Koordinaten. Objekte, die so konstruiert wurden, sind vollständig verwendungsfähig.
     *
     * @param painter       ein Painter, der dieses Object zeichnen kann, etwa ein {@link DOTRnPainter}
     * @param dotCollection die Kollektion, die beim Zeichnen verwendet werden soll
     * @param routeNumber   der Straßenname, zu dem dieses Objekt gehört
     * @param point         die Koordinateninformation
     */
    public RnDisplayObject(DisplayObjectPainter painter, final DOTCollection dotCollection, final int routeNumber, final Category category,
                           final PointWithAngle point) {
        super();
        _painter = painter;
        _dotCollection = dotCollection;
        _defaultType = 0;
        _routeNumber = routeNumber;
        _category = category;
        _coordinates = new ArrayList<>(1);
        _coordinates.add(point);
    }

    public int getRouteNumber() {
        return _routeNumber;
    }

    public Category getCategory() {
        return _category;
    }

    @Override
    public String getName() {
        Object o = _coordinates.get(0);
	    if (null != o && o instanceof PointWithAngle pointWithAngle) {
            Point2D point = pointWithAngle.getPoint();
            return "A" + _routeNumber + " (x=" + point.getX() + ", y=" + point.getY() + ")";
        } else {
            return "A" + _routeNumber;
        }

    }

    @Override
    public List<Object> getCoordinates(final int type) {
        return Collections.unmodifiableList(_coordinates);
    }

    @Override
    public List<Object> getCoordinates() {
        return Collections.unmodifiableList(_coordinates);
    }

    @Override
    public List<Point2D> getReferencePoints(final Rectangle utmBounds) {
        List<Point2D> pointList = new ArrayList<>(1);
        if (!_coordinates.isEmpty()) {
            Point2D point = ((PointWithAngle) _coordinates.get(0)).getPoint();
            if (utmBounds.contains(point)) {
                pointList.add(point);
            }
        }
        return pointList;
    }

    @Override
    public DisplayObjectPainter getPainter() {
        return _painter;
    }

    @Override
    public void setPainter(final DisplayObjectPainter painter) {
        _painter = painter;
    }

    @Override
    public DOTCollection getDOTCollection() {
        return _dotCollection;
    }

    @Override
    public Rectangle getBoundingRectangle(final int type) {
        if (!_boundingRectangles.containsKey(type)) {
            _boundingRectangles.put(type, _painter.getBoundingRectangle(this, type));
        }
        return _boundingRectangles.get(type);
    }

    @Override
    public Rectangle getBoundingRectangle() {
        return getBoundingRectangle(_defaultType);
    }

    @Override
    public void setDefaultType(final int defaultType) {
        _defaultType = defaultType;
    }

    @Nullable
    @Override
    public DisplayObjectType.DisplayObjectTypeItem getDisplayObjectTypeItem(final PrimitiveFormPropertyPair pair) {
        return null;
    }

    @Override
    public void mapScaleChanged(final double scale) {
    }

    @Override
    public String toString() {
        return "RnDisplayObject{" + "_boundingRectangles=" + _boundingRectangles + ", _painter=" + _painter + ", _dotCollection=" + _dotCollection +
               ", _routeNumber=" + _routeNumber + ", _category=" + _category + ", _coordinates=" + _coordinates + ", _defaultType=" + _defaultType +
               '}';
    }

    /**
     * Die Kategorie eines {@code RnDisplayObjects} bestimmt, wann er gezeichnet wird. Im {@link DOTRnPainter} wird diese Kategorie mit der des
     * aktiven {@link DOTRn}-Objekts verglichen: ist sie kleiner oder gleich deren Wert, so wird das Objekt gezeichnet.
     */
    public enum Category {
        NULL(0), ONE(1), TWO(2), THREE(3), FOUR(4);

        private final int _ordinal; // ein eigener Ordinalwert, weil die Section 8.9 der JLS nicht aussagekräftig

        Category(int i) {
            _ordinal = i;
        }

        /**
         * Der Getter für den Ordinal-Wert.
         *
         * @return
         */
        public int getOrdinal() {
            return _ordinal;
        }
    }
}
