/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.selection;

import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Component;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.awt.geom.AffineTransform;
import java.awt.geom.NoninvertibleTransformException;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import javax.swing.JPanel;

/**
 * Die Klasse SelectionPanel ist eine von JPanel abgeleitete Klasse, die ein Rechteck für eine Rechteckselektion verwaltet. Dabei übernimmt
 * SelectionPanel die Visualisierung des Rechtecks; die Rechteckselektion wird hier nicht durchgeführt.
 * <p>
 * SelectionPanel ist vom Charackter her eine lokale Klasse von MapPane. Dort wird für die Rechteckselektion ein Objekt dieser Klasse instanziiert und
 * mit Hilfe des Rechtecks die Selektion durchgeführt. Das ist bislang die einzige Verendung.
 *
 * @author Kappich Systemberatung
 */
public class SelectionPanel extends JPanel implements ComponentListener {

    private static final long serialVersionUID = 1L;
    private static final boolean LOG = false;
    private final Component _parent;
    /*
     * Das von SelektionPanel verwaltete Rechteck wird in zwei Varianten vorgehalten;
     * einmal als Basis-Rechteck, welches unabhängig von den aktuellen Transformationsparametern
     * von MapPane ist, und einmal in transformierter Form wie es beim Zeichnen und zur
     * Selektion benötigt wird.
     */
    private Rectangle2D _baseRectangle;
    private Rectangle2D _transformedRectangle;

    public SelectionPanel(Component parent) {
        super();
        setOpaque(false);
        _parent = parent;
    }

    /**
     * @return gibt das Rechteck in transformierter Form zurück
     */
    public Rectangle2D getRectangle() {
        return _transformedRectangle;
    }

    /**
     * Die Parameter dieser Methode sind das Rechteck aus dem MapPane und dessen aktuelle Transformationsparameter.
     *
     * @param rectangle
     * @param scale
     * @param translateX
     * @param translateY
     */
    public void setRectangle(@Nullable final Rectangle2D.Double rectangle, double scale, double translateX, double translateY) {
        if (LOG) {
            //noinspection UseOfSystemOutOrSystemErr
            System.out.println("setRectangle:" + rectangle + "   scale=" + scale + "   tX=" + translateX + "   tY=" + translateY);
        }
        setSize(_parent.getSize());
        if (rectangle != null) {
            /* Es wird eine neutrale Version des Rechtecks berechnet und in _baseRectangle gespeichert,
             * welches den Werten scale = 1 und translateX=translateY=0 entspricht. */
            AffineTransform affineTransform = new AffineTransform();
            affineTransform.translate(translateX, translateY);
            affineTransform.scale(scale, scale);
            try {
                affineTransform.invert();
            } catch (NoninvertibleTransformException e) {
                e.printStackTrace();
            }
            Point2D p1 = new Point2D.Double(rectangle.getX(), rectangle.getY());
            affineTransform.transform(p1, p1);
            Point2D p2 = new Point2D.Double(rectangle.getX() + rectangle.getWidth(), rectangle.getY() + rectangle.getHeight());
            affineTransform.transform(p2, p2);
            _baseRectangle = new Rectangle2D.Double(p1.getX(), p1.getY(), p2.getX() - p1.getX(), p2.getY() - p1.getY());
            /* _transformedRectangle enthält das aktuell transformierte Rechteck */
            _transformedRectangle = rectangle;
            if (LOG) {
                //noinspection UseOfSystemOutOrSystemErr
                System.out.println("   base rectangle: " + _baseRectangle);
                //noinspection UseOfSystemOutOrSystemErr
                System.out.println("   transform rectangle: " + _transformedRectangle);
            }
        } else {
            _baseRectangle = null;
            _transformedRectangle = null;
        }
    }

    /**
     * Diese Methode ruft das MapPane auf, wenn sich Transformationsparameter ändern.
     *
     * @param scale
     * @param translateX
     * @param translateY
     */
    public void transformRectangle(double scale, double translateX, double translateY) {
        if (_baseRectangle != null) {
            AffineTransform affineTransform = new AffineTransform();
            affineTransform.translate(translateX, translateY);
            affineTransform.scale(scale, scale);
            Point2D p1 = new Point2D.Double(_baseRectangle.getX(), _baseRectangle.getY());
            affineTransform.transform(p1, p1);
            Point2D p2 = new Point2D.Double(_baseRectangle.getX() + _baseRectangle.getWidth(), _baseRectangle.getY() + _baseRectangle.getHeight());
            affineTransform.transform(p2, p2);
            _transformedRectangle = new Rectangle2D.Double(p1.getX(), p1.getY(), p2.getX() - p1.getX(), p2.getY() - p1.getY());
            if (LOG) {
                //noinspection UseOfSystemOutOrSystemErr
                System.out.println("transformRectangle:");
                //noinspection UseOfSystemOutOrSystemErr
                System.out.println("   base rectangle: " + _baseRectangle);
                //noinspection UseOfSystemOutOrSystemErr
                System.out.println("   transform rectangle: " + _transformedRectangle);
            }
        }
    }

    @Override
    protected void paintComponent(Graphics g) {
        super.paintComponent(g);
        Graphics2D g2D = (Graphics2D) g;

        g2D.setColor(Color.black);
        g2D.setStroke(new BasicStroke(1));
        g2D.setFont(new Font("Default", Font.PLAIN, 10));

        if (_transformedRectangle != null && !_transformedRectangle.isEmpty()) {
            g2D.drawRect((int) _transformedRectangle.getX(), (int) _transformedRectangle.getY(), (int) _transformedRectangle.getWidth(),
                         (int) _transformedRectangle.getHeight());
        }
    }

    /*
     * Implementation des ComponentListeners
     */

    @Override
    public void componentHidden(ComponentEvent e) {
    }

    @Override
    public void componentMoved(ComponentEvent e) {
    }

    @Override
    public void componentResized(ComponentEvent e) {
        setSize(e.getComponent().getSize());
    }

    @Override
    public void componentShown(ComponentEvent e) {
    }

    @Override
    public String toString() {
        return "SelectionPanel{" + "_parent=" + _parent + ", _baseRectangle=" + _baseRectangle + ", _transformedRectangle=" + _transformedRectangle +
               '}';
    }
}
