/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.statPlugin;

import de.kappich.pat.gnd.colorManagement.ColorManager;
import de.kappich.pat.gnd.displayObjectToolkit.DOTManager;
import de.kappich.pat.gnd.displayObjectToolkit.DynamicDOTItem;
import de.kappich.pat.gnd.gnd.LegendTreeNodes;
import de.kappich.pat.gnd.pluginInterfaces.DefaultDisplayObjectType;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectType;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectTypePlugin;
import de.kappich.pat.gnd.properties.ColorProperty;
import de.kappich.pat.gnd.properties.DistanceRasterProperty;
import de.kappich.pat.gnd.properties.DistanceRasterType;
import de.kappich.pat.gnd.properties.Property;
import de.kappich.pat.gnd.properties.StatFormatProperty;
import de.kappich.pat.gnd.properties.TextStyleProperty;
import java.awt.Color;
import java.util.Map;
import java.util.prefs.Preferences;

/**
 * {@code DOTStat} ist die Implementation von {@link DisplayObjectType} im Stationierungs-Plugin. {@code DOTStat} erweitert {@code
 * DefaultDisplayObjectType}.
 *
 * @author Kappich Systemberatung
 */
public class DOTStat extends DefaultDisplayObjectType {

    private static final String INFO = "INFO";
    private static final String JOIN_BY_LINE = "JOIN_BY_LINE";
    private static final String TRANSLATION_FACTOR = "TRANSLATION_FACTOR";

    private Double _translationFactor;
    private boolean _joinByLine;

    /**
     * Ein Defaultkonstruktor für {@code DOTStat}. Eine spezifische Verwendung ohne Namen und Info ist nicht sinnvoll.
     */
    public DOTStat() {
        super(true);
        _name = "";
        _info = "";
        _translationFactor = 0.;
        _joinByLine = false;
        DOTManager.getInstance().addDOTChangeListener(this);
    }

    /**
     * Konstruktor mit vorgegebenen Namen und Info.
     *
     * @param name der unter allen DisplayObjectTypes eindeutige Name
     * @param info eine Kurzinformation, die z.B. als Tooltipp verwendet wird
     */
    public DOTStat(String name, String info) {
        super(true);
        _name = name;
        _info = info;
        _translationFactor = 0.;
        _joinByLine = false;
        DOTManager.getInstance().addDOTChangeListener(this);
    }

    @Override
    public DisplayObjectTypePlugin getDisplayObjectTypePlugin() {
        return new DOTStatPlugin();
    }

    /**
     * Der Getter für den Verschiebungsfaktor bzw. -länge.
     *
     * @return der Verschiebungsfaktor
     */
    public Double getTranslationFactor() {
        return _translationFactor;
    }

    /**
     * Der Setter für den Verschiebungsfaktor bzw. -länge.
     *
     * @param translationFactor der Verschiebungsfaktor
     */
    public void setTranslationFactor(Double translationFactor) {
        _translationFactor = translationFactor;
    }

    /**
     * Gibt {@code true} zurück, wenn die Lage in der Karte mit dem verschobenen Objekt durch eine Verbindungslinie verbunden werden soll.
     *
     * @return soll eine Verbindungslinie gezeichnet werden
     */
    public boolean isJoinedByLine() {
        return _joinByLine;
    }

    /**
     * Setzt die interne Variable, die bestimmt, ob die Lage in der Karte mit dem verschobenen Objekt durch eine Verbindungslinie verbunden werden
     * soll.
     *
     * @param joinByLine legt fest, ob eine Verbindungslinie gezeichnet werden soll
     */
    public void setJoinByLine(boolean joinByLine) {
        _joinByLine = joinByLine;
    }

    @Override
    public DisplayObjectType getCopy(String name) {
        String s = name;
        if (null == s) {
            s = getName();
        }
        DOTStat dotStat = (DOTStat) super.getCopy(s);
        dotStat.setJoinByLine(isJoinedByLine());
        dotStat.setTranslationFactor(getTranslationFactor());
        return dotStat;
    }

    @Override
    public String toString() {
        return "dotStat{" + "_translationFactor=" + _translationFactor + ", _joinByLine=" + _joinByLine + '}';
    }

    @Override
    public void initializeFromPreferences(Preferences prefs) {
        super.initializeFromPreferences(prefs);
        Preferences joinByLinePrefs = prefs.node("joinByLine");
        _joinByLine = joinByLinePrefs.getBoolean(JOIN_BY_LINE, false);
        Preferences translationFactorPrefs = prefs.node("translationFactor");
        _translationFactor = translationFactorPrefs.getDouble(TRANSLATION_FACTOR, 0.0);
    }

    @Override
    public void putPreferences(Preferences prefs) {
        deletePreferences(prefs);
        Preferences classPrefs = prefs.node(getClass().getSimpleName());
        Preferences objectPrefs = classPrefs.node(getName());
        Preferences infoPrefs = objectPrefs.node("info");
        infoPrefs.put(INFO, getInfo());
        Preferences translationFactorPrefs = objectPrefs.node("translationFactor");
        translationFactorPrefs.putDouble(TRANSLATION_FACTOR, _translationFactor);
        Preferences joinByLinePrefs = objectPrefs.node("joinByLine");
        joinByLinePrefs.putBoolean(JOIN_BY_LINE, _joinByLine);

        Preferences staticPrefs = objectPrefs.node("static");
        for (final Map.Entry<Property, Boolean> entry : _isStaticMap.entrySet()) {
            if (entry.getValue()) {
                // Eine statische Property wird als dynamische ohne Anmeldungsdaten weggeschrieben.
                Preferences propertyPrefs = staticPrefs.node(entry.getKey().getKey());
                final DynamicDOTItem dynamicDOTItem = new DynamicDOTItem("", "", "", "", _staticPropertyValues.get(entry.getKey()));
                dynamicDOTItem.putPreferences(propertyPrefs);
            }
        }
    }

    @Override
    public LegendTreeNodes getLegendTreeNodes() {
        LegendTreeNodes legendTreeNodes = new LegendTreeNodes();

        boolean textNode = false;
        final String colorName = (String) getValueOfStaticProperty(null, ColorProperty.getInstance());
        final Object styleObject = getValueOfStaticProperty(null, TextStyleProperty.getInstance());
        final Object statFormatObject = getValueOfStaticProperty(null, StatFormatProperty.getInstance());
        final Object distanceRasterObject = getValueOfStaticProperty(null, DistanceRasterProperty.getInstance());
        if (null != styleObject) {
            final int style;
            if (styleObject instanceof Integer) {
                style = (Integer) styleObject;
            } else {
                style = ((TextStyleProperty.Styles) styleObject).getIntValue();
            }
            if (null != colorName && (0 <= style) && (2 >= style)) {
                final Color color = ColorManager.getInstance().getColor(colorName);
                if (null != color) {
                    StatDisplayObject displayObject = new StatDisplayObject(1550L);
                    String text = displayObject.getText((DistanceRasterType) distanceRasterObject, (StatFormat) statFormatObject);
                    if (null == text) {
                        text = "1,5";
                    }
                    LegendTreeNodes.TextTreeNode node = new LegendTreeNodes.TextTreeNode(text, color, style, "ASB-Stationierungswert", null);
                    legendTreeNodes.add(node, 0);
                    textNode = true;
                }
            }
        }
        if (!textNode && null != colorName) {
            LegendTreeNodes.LegendTreeNode node = new LegendTreeNodes.LegendTreeNode(colorName, null, null);
            legendTreeNodes.add(node, 0);
        }
        return legendTreeNodes;
    }
}
