/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.statPlugin;

import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.kappich.pat.gnd.colorManagement.ColorManager;
import de.kappich.pat.gnd.displayObjectToolkit.DisplayObject;
import de.kappich.pat.gnd.gnd.MapPane;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectPainter;
import de.kappich.pat.gnd.properties.ColorProperty;
import de.kappich.pat.gnd.properties.DistanceRasterProperty;
import de.kappich.pat.gnd.properties.DistanceRasterType;
import de.kappich.pat.gnd.properties.DottingProperty;
import de.kappich.pat.gnd.properties.StatFormatProperty;
import de.kappich.pat.gnd.properties.TextSizeProperty;
import de.kappich.pat.gnd.properties.TextStyleProperty;
import de.kappich.pat.gnd.utils.PointWithAngle;
import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.Shape;
import java.awt.font.FontRenderContext;
import java.awt.geom.AffineTransform;
import java.awt.geom.Line2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.List;

/**
 * Der {@code DOTStatPainter} ist die Implemenation des {@link DisplayObjectPainter} im Stationierungs-Plugin.
 *
 * @author Kappich Systemberatung
 */
public class DOTStatPainter implements DisplayObjectPainter {

    private static final AffineTransform IDENTITY = new AffineTransform();

    @Nullable
    private static Line2D.Double getStroke(final StatDisplayObject statDisplayObject, final DOTStat dotStat) {
        List<Object> coor = statDisplayObject.getCoordinates(0);
	    if (coor.isEmpty() || !(coor.get(0) instanceof PointWithAngle pointWithAngle)) {
            return null;
        }
        AffineTransform angleTransformation = getAngleTransformation(dotStat.getTranslationFactor(), pointWithAngle.getAngle());
        final Point2D point = pointWithAngle.getPoint();
        final Double xAsDouble = point.getX();
        final Double yAsDouble = point.getY();

        Point2D.Double translation1 = new Point2D.Double(xAsDouble, yAsDouble);
        final Point2D.Double translation2 = new Point2D.Double();
        angleTransformation.transform(translation1, translation2);
        final Double x = translation2.getX();
        final Double y = translation2.getY();
        Line2D.Double line = new Line2D.Double(xAsDouble.intValue(), yAsDouble.intValue(), x.intValue(), y.intValue());
        stretchLineAtTheEnd(line, 0.33);
        return line;
    }

    /*
     * Gibt true zurück, wenn die übergebene Linie mindestens eines der Rechtecke, die von
     * Shape.getBounds() für dieübergebenen Shapes zurückgegeben werden, schneidet.
     */
    private static boolean lineIntersectsWithShapes(final Line2D line, final List<Shape> shapes) {
        for (Shape shape : shapes) {
            final Rectangle boundingRectangle = shape.getBounds();
            if (line.intersects(boundingRectangle)) {
                return true;
            }
        }
        return false;
    }

    /*
     * Verkürzt oder verlängert die übergebene Linie an ihrem Ende um den übegebenen Stretchfaktor.
     * Beispiel: ein Strechtfaktor von 0,95 führt zu einer Linie, an deren Ende 5 % abgeschnitten wurden.
     */
    private static void stretchLineAtTheEnd(final Line2D line, double stretchFactor) {
        final Point2D p1 = line.getP1();
        final Point2D p2 = line.getP2();
        line.setLine(p1.getX(), p1.getY(), p1.getX() + stretchFactor * (p2.getX() - p1.getX()), p1.getY() + stretchFactor * (p2.getY() - p1.getY()));
    }

    /*
     * Liefert eine affine Transformation zurück, die eine Translation der Länge translationFactor
     * in Richtung des Winkels angle, der zur x-Achse berechnet wird, durchführt.
     */
    private static AffineTransform getAngleTransformation(final Double translationFactor, @Nullable final Double angle) {
        AffineTransform angleTransformation = new AffineTransform();
        if (null == angle || angle.isNaN()) {
            return angleTransformation;
        }
        angleTransformation.translate(-translationFactor * Math.sin(angle), translationFactor * Math.cos(angle));
        return angleTransformation;
    }

    @SuppressWarnings("OverlyLongMethod")
    @Override
    public void paintDisplayObject(final MapPane mapPane, final Graphics2D g2D, final DisplayObject displayObject, final boolean selected) {
        //g2D.setRenderingHint( RenderingHints.KEY_FRACTIONALMETRICS, RenderingHints.VALUE_FRACTIONALMETRICS_ON);
	    if (displayObject instanceof StatDisplayObject statDisplayObject) {
            final DOTStat dotStat = (DOTStat) statDisplayObject.getDOTCollection().getDisplayObjectType(mapPane.getMapScale().intValue());
            if (dotStat == null) {
                return;
            }
            if (statDisplayObject.checkType((DistanceRasterType) dotStat.getValueOfStaticProperty(null, DistanceRasterProperty.getInstance()))) {
                List<Object> coor = statDisplayObject.getCoordinates(0);
	            if (coor.isEmpty() || !(coor.get(0) instanceof PointWithAngle pointWithAngle)) {
                    return;
                }
                AffineTransform angleTransformation = getAngleTransformation(dotStat.getTranslationFactor(), pointWithAngle.getAngle());
                final Point2D point = pointWithAngle.getPoint();
                final Double xAsDouble = point.getX();
                final Double yAsDouble = point.getY();
                // Farbe
                final Color color = ColorManager.getInstance().getColor((String) dotStat.getValueOfStaticProperty(null, ColorProperty.getInstance()));
                g2D.setColor(color);
                // Font
                final Object styleObject = dotStat.getValueOfStaticProperty(null, TextStyleProperty.getInstance());
                if (null == styleObject) {
                    return;
                }
                final int style;
                if (styleObject instanceof Integer) {
                    style = (Integer) styleObject;
                } else {
                    style = ((TextStyleProperty.Styles) styleObject).getIntValue();
                }
                final Integer textSize = (Integer) dotStat.getValueOfStaticProperty(null, TextSizeProperty.getInstance());
                Font font;
                if (null != textSize) {
                    font = new Font(Font.DIALOG, style, textSize);
                } else {
                    font = new Font(Font.DIALOG, style, 12);
                }
                g2D.setFont(font);
                // Text
                String text = statDisplayObject
                    .getText((DistanceRasterType) dotStat.getValueOfStaticProperty(null, DistanceRasterProperty.getInstance()),
                             (StatFormat) dotStat.getValueOfStaticProperty(null, StatFormatProperty.getInstance()));
                if (text == null) {
                    text = " ";
                }
                // Transformation
                Point2D.Double translation1 = new Point2D.Double(xAsDouble, yAsDouble);
                final Point2D.Double translation2 = new Point2D.Double();
                angleTransformation.transform(translation1, translation2);
                Rectangle2D textBoundingRectangle;

                AffineTransform at = g2D.getTransform();
                g2D.setTransform(IDENTITY);
                FontRenderContext fontRenderContext = g2D.getFontRenderContext();
                textBoundingRectangle = font.getStringBounds(text, fontRenderContext);
                // Dieses Rechteck ist zu hoch. Hier noch andere Versuche, die auch das hohe Rechteck liefern:
//				FontMetrics fontMetrics = g2D.getFontMetrics();
//				textBoundingRectangle = fontMetrics.getStringBounds(text, g2D);
//				LineMetrics lineMetrics = fontMetrics.getLineMetrics(text,g2D);
//				textBoundingRectangle.setRect(textBoundingRectangle.getX(), textBoundingRectangle.getY(),
//				                              textBoundingRectangle.getWidth(), lineMetrics.getHeight());
                // Deshalb jetzt dieses Verfahren:
                textBoundingRectangle.setRect(textBoundingRectangle.getX(), textBoundingRectangle.getY() + textBoundingRectangle.getHeight() / 3,
                                              textBoundingRectangle.getWidth(),
                                              textBoundingRectangle.getHeight() - textBoundingRectangle.getHeight() / 3);
                final Point2D.Double deviceTextPoint = new Point2D.Double();
                at.transform(translation2, deviceTextPoint);
                // devicePoint ist nun der Punkt in Device-Koordinaten, an den der Text verschoben
                // wird. Da der Text aber in ein Rechteck geschrieben wird, dessen linke untere Ecke
                // angegeben wird, so müssen diese Koordinaten noch berechnet werden:
                final Double textX = deviceTextPoint.getX() - textBoundingRectangle.getWidth() / 2;
                final Double textY = deviceTextPoint.getY() + textBoundingRectangle.getHeight() / 2;
                g2D.drawString(text, textX.floatValue(), textY.floatValue());

                if (dotStat.isJoinedByLine()) {
                    // für die Verbindungslinie fehlt noch der Punkt auf der Linie in Device-Koordinaten
                    final Point2D.Double deviceLinePoint = new Point2D.Double();
                    at.transform(point, deviceLinePoint);
                    Line2D.Double line = new Line2D.Double(deviceLinePoint, deviceTextPoint);
                    Rectangle2D.Double newRectangle =
                        new Rectangle2D.Double(textX, textY - textBoundingRectangle.getHeight(), textBoundingRectangle.getWidth(),
                                               textBoundingRectangle.getHeight());
                    List<Shape> shapes = new ArrayList<>();
                    shapes.add(newRectangle);
                    int counter = 0;
                    while (lineIntersectsWithShapes(line, shapes) && counter < 10) {
                        stretchLineAtTheEnd(line, 0.8);
                        counter++;
                    }
                    g2D.setStroke(new BasicStroke(1.f));
//					/*TEST*/
//					RenderingHints rh = g2D.getRenderingHints();
//					rh.add(new RenderingHints(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY));
//					g2D.setRenderingHints(rh);
//					/*TEST*/
                    g2D.draw(line);
                }
                g2D.setTransform(at);
            } else if (statDisplayObject.checkType((DistanceRasterType) dotStat.getValueOfStaticProperty(null, DottingProperty.getInstance()))) {
                // es muss gestrichelt werden
                Line2D.Double line = getStroke(statDisplayObject, dotStat);
                if (line == null) {
                    return;
                }
                final Color color = ColorManager.getInstance().getColor((String) dotStat.getValueOfStaticProperty(null, ColorProperty.getInstance()));
                g2D.setColor(color);
                g2D.setStroke(new BasicStroke((float) Math.min(3.f, mapPane.getMapScale() / 3000.)));
                g2D.draw(line);
            }
        }
    }

    @Override
    @Nullable
    public List<Object> getCoordinates(final List<Object> coordinates, final int type) {
        return null;
    }

    @Nullable
    @Override
    public Rectangle getBoundingRectangle(final DisplayObject displayObject, final int type) {
        // Provisorische Implementation (kopiert von DOTKmPainter)
        for (Object o : displayObject.getCoordinates(type)) {  // höchstens 1 Element
	        if (o instanceof PointWithAngle pwa) {
                return new Rectangle((int) (pwa.getPoint().getX() - 250), (int) (pwa.getPoint().getY() - 150), 500, 300);
	        } else if (o instanceof Point2D.Double p) {          // kommt in diesem Plugin derzeit nicht vor
                return new Rectangle((int) (p.getX() - 250), (int) (p.getY() - 150), 500, 300);
            }
        }
        return null;
    }
}
