/*
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.utils;

import de.bsvrz.sys.funclib.kappich.annotations.Nullable;

/**
 * Eine generische Klasse zur Verwaltung von Intervallen von Zahlen.
 * <p>
 * Die Klasse {@code Interval} dient zur Verwaltung von Intervalle der Klasse {@code E}, die die Klasse {@code Number} erweitert. Jedes Intervall hat
 * eine Untergrenze und eine Obergrenze, wobei es keine festgelegte Interpretation gibt, ob diese Werte zum Intervall dazu gehören oder nicht. Daneben
 * gibt es noch einen Zähler, mit dessen Hilfe ansonsten gleiche Intervalle in einer Menge koexistieren können.
 *
 * @author Kappich Systemberatung
 */
public class Interval<E extends Number> implements Comparable<Interval<E>> {
    private final E _lowerBound;
    private final E _upperBound;
    private int _counter;    // Erlaubt es ansonsten gleiche Intervalle als unterschiedliche Schlüssel
    // in Maps etc zu verwenden, wenn man den Counter hochzählt!

    /**
     * Konstruiert ein Intervall aus den übergebenen Grenzen.
     *
     * @param lowerBound die untere Grenze
     * @param upperBound die obere Grenze
     */
    public Interval(E lowerBound, E upperBound) {
        super();
        _lowerBound = lowerBound;
        _upperBound = upperBound;
        _counter = 0;
    }

    /**
     * Der Getter der unteren Grenze.
     *
     * @return die untere Grenze
     */
    public E getLowerBound() {
        return _lowerBound;
    }

    /**
     * Der Getter der oberen Grenze.
     *
     * @return die obere Grenze
     */
    public E getUpperBound() {
        return _upperBound;
    }

    /**
     * Der Getter des Zählers.
     *
     * @return der Zählerwert
     */
    public int getCounter() {
        return _counter;
    }

    /**
     * Der Setter des Zählers.
     *
     * @param counter neue Zählerwert
     */
    public void setCounter(int counter) {
        _counter = counter;
    }

    /**
     * Gibt {@code true} zurück, wenn die beiden Intervalle im Sinne abgeschlossener Intervalle eine nicht-leere Schnittmenge haben.
     *
     * @param o ein Intervall
     *
     * @return {@code true} genau dann, wenn die beiden Intervalle im Sinne abgeschlossener Intervalle eine nicht-leere Schnittmenge haben
     */
    public boolean intersects(Interval<E> o) {
        if (_lowerBound.doubleValue() < o._lowerBound.doubleValue()) {
            return _upperBound.doubleValue() >= o._lowerBound.doubleValue();
        } else {
            return _lowerBound.doubleValue() <= o._upperBound.doubleValue();
        }
    }

    /**
     * Gibt die Schnittmenge der beiden Intervalle zurück oder {@code null}, wenn der Schnitt leer ist.
     *
     * @param o ein Intervall
     *
     * @return die Schnittmenge oder {@code null}
     */
    @SuppressWarnings("unused")
    @Nullable
    public Interval<E> intersection(Interval<E> o) {
        if (_lowerBound.doubleValue() < o._lowerBound.doubleValue()) {
            if (_upperBound.doubleValue() >= o._lowerBound.doubleValue()) {
                if (_upperBound.doubleValue() > o._upperBound.doubleValue()) {
                    return new Interval<>(o._lowerBound, o._upperBound);
                } else {
                    return new Interval<>(o._lowerBound, _upperBound);
                }
            }
            return null;
        } else {
            if (_lowerBound.doubleValue() <= o._upperBound.doubleValue()) {
                if (_upperBound.doubleValue() <= o._upperBound.doubleValue()) {
                    return new Interval<>(_lowerBound, _upperBound);
                } else {
                    return new Interval<>(_lowerBound, o._upperBound);
                }
            } else {
                return null;
            }
        }
    }

    /**
     * Die Länge eines Intervalls ist die Differenz von oberer und unterer Grenze. Achtung: da keine Überprüfung stattfindet, ob diese Grenzen
     * sinnvolle Werte enthalten oder ihre Relation stimmt, kann das Ergebnis entsprechend ausfallen.
     *
     * @return die Länge
     */
    public double length() {
        return _upperBound.doubleValue() - _lowerBound.doubleValue();
    }

    /**
     * Gibt ein neues Intervall zurück, dass dieselben unteren und oberen Schranken besitzt, und auch denselben Zählerwert.
     *
     * @return die Kopie
     */
    public Interval<E> getCopy() {
        Interval<E> newInterval = new Interval<>(getLowerBound(), getUpperBound());
        newInterval.setCounter(getCounter());
        return newInterval;
    }

    @Override
    public int compareTo(Interval<E> o) {
        int v = Double.compare(_lowerBound.doubleValue(), o._lowerBound.doubleValue());
        if (0 != v) {
            return v;
        } else {
            v = Double.compare(_upperBound.doubleValue(), o._upperBound.doubleValue());
            if (0 != v) {
                return v;
            } else {
                //noinspection CompareToUsesNonFinalVariable
                return Integer.compare(_counter, o._counter);
            }
        }
    }

    @SuppressWarnings("unchecked")
    @Override
    public boolean equals(Object o) {
        if (o == null) {
            return false;
        }
        if (this == o) {
            return true;
        }
        if (getClass() != o.getClass()) {
            return false;
        }
        Interval<E> oInterval = (Interval<E>) o;
        //noinspection NonFinalFieldReferenceInEquals
        return !(!oInterval.getLowerBound().equals(getLowerBound()) || !oInterval.getUpperBound().equals(getUpperBound()) ||
                 (_counter != oInterval._counter));
    }

    /**
     * Der Hashcode eines Intervalls hängt nur von der oberen und unteren Grenze ab, nicht aber vom Zähler.
     *
     * @return die Summe der Hashcodes der unteren und oberen Grenze
     */
    @Override
    public int hashCode() {
        return _lowerBound.hashCode() + _upperBound.hashCode();
    }

    @Override
    public String toString() {
        return "[Interval: " + _lowerBound + ", " + _upperBound + ", " + _counter + "]";
    }
}
