/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.utils.view;

import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.kappich.pat.gnd.gnd.PreferencesHandler;
import java.awt.Component;
import java.awt.Container;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.LayoutManager;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Toolkit;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.util.HashMap;
import java.util.Map;
import java.util.WeakHashMap;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;
import javax.swing.JFrame;
import javax.swing.JMenuBar;

/**
 * <p>Diese Klasse erlaubt es einen {@link JFrame} mit Zusatzfunktionalität zur Speicherung und Wiederbenutzung von Größe und Location
 * auszustatten. In allen Konstruktoren wird ein Identifier angegeben. GndFrame-Objekte mit gleichem Identifier greifen auf die gleichen Größen- und
 * Location-Angeben zurück.</p>
 *
 * @author Kappich Systemberatung
 */
public class GndFrame {

    private static final boolean LOG = false;
    private static final String USER_DEFINED = "benutzer-definiert";
    private static final String TYPE = "Typ";
    private static final String VALUE = "Wert";
    private static final String INTEGER = "Integer";
    private static final String STRING = "String";
    private static final String DOUBLE = "Double";
    private static final String LONG = "Long";
    private static final String BOOLEAN = "Boolean";
    private static final String FLOAT = "Float";

    private static final Map<Frame, GndFrame> FRAME_LOOKUP = new WeakHashMap<>();

    private final JFrame _frame;
    private final String _identifier;
    private final Map<String, Object> _userDefinedPreferences = new HashMap<>();
    private boolean _hasPreferences;
    private int _preferencesX;
    private int _preferencesY;
    private int _preferencesWidth;
    private int _preferencesHeight;

    /**
     * Ein Konstruktor mit dem Identifier. Hier wird intern ein {@link JFrame} ohne Titel konstruiert.
     *
     * @param identifier ein Identifier
     */
    public GndFrame(final String identifier) {
        _frame = new JFrame();
        _identifier = identifier;
        readPreferenceBounds();
        _frame.addWindowListener(new MyWindowListener());
        FRAME_LOOKUP.put(_frame, this);
    }

    /**
     * Ein Konstruktor mit Identifier und Titel.  Hier wird intern ein {@link JFrame} mit dem übergebenen Titel konstruiert.
     *
     * @param identifier ein Identifier
     * @param title      ein Titel des {@link JFrame}
     */
    public GndFrame(final String identifier, final String title) {
        _frame = new JFrame(title);
        _identifier = identifier;
        readPreferenceBounds();
        _frame.addWindowListener(new MyWindowListener());
        FRAME_LOOKUP.put(_frame, this);
    }

    /**
     * Ein Konstruktor mit Identifier und einem {@link JFrame}. Hier wird der übergebene JFrame benutzt.
     *
     * @param frame      ein JFrame
     * @param identifier ein Identifier
     */
    public GndFrame(final JFrame frame, final String identifier) {
        _frame = frame;
        _identifier = identifier;
        readPreferenceBounds();
        _frame.addWindowListener(new MyWindowListener());
        FRAME_LOOKUP.put(_frame, this);
    }

    public static void storePreferenceBounds(final Frame[] frames) {
        for (Frame frame : frames) {
            if (FRAME_LOOKUP.containsKey(frame)) {
                GndFrame gndFrame = FRAME_LOOKUP.get(frame);
                gndFrame.storePreferenceBounds();
            }
        }
    }

    /**
     * Löscht in den Präferenzen die Informationen zu Position und Größe zu diesem Frame (und damit zu allen Frames mit dem gleichen Identifier).
     */
    public static void removePreferenceBounds() {
        Preferences prefs = getPreferenceNode();
        try {
            prefs.removeNode();
        } catch (BackingStoreException ignored) {
            PreferencesDeleter pd =
                new PreferencesDeleter("Für ein Fenster konnten die Einstellungen der letzten Sitzungen nicht gelöscht werden.", prefs);
            pd.run();
        }
    }

    private static Preferences getPreferenceNode() {
        return PreferencesHandler.getInstance().getPreferenceStartPath().node("GNDFrame");
    }

    /**
     * @return
     */
    public JFrame getFrame() {
        return _frame;
    }

    /**
     * Liegen Bounds (x,y, width und height) in den Präferenzen vor?
     *
     * @return die Antwort
     */
    public boolean hasPreferences() {
        return _hasPreferences;
    }

    /**
     * Der x-Wert aus den Präferenzen. Die Methode darf nur benutzt werden, wenn {@code hasPreferences} den Wert {@code true} zurückgeliefert hat.
     *
     * @return x
     */
    public int getPreferencesX() {
        return _preferencesX;
    }

    /**
     * Der y-Wert aus den Präferenzen. Die Methode darf nur benutzt werden, wenn {@code hasPreferences} den Wert {@code true} zurückgeliefert hat.
     *
     * @return y
     */
    public int getPreferencesY() {
        return _preferencesY;
    }

    /**
     * Der width-Wert aus den Präferenzen. Die Methode darf nur benutzt werden, wenn {@code hasPreferences} den Wert {@code true} zurückgeliefert
     * hat.
     *
     * @return width
     */
    public int getPreferencesWidth() {
        return _preferencesWidth;
    }

    /**
     * Der height-Wert aus den Präferenzen. Die Methode darf nur benutzt werden, wenn {@code hasPreferences} den Wert {@code true} zurückgeliefert
     * hat.
     *
     * @return height
     */
    public int getPreferencesHeight() {
        return _preferencesHeight;
    }

    /**
     * Liefert ein die Bounds beschreibendes Rechteck zurück, falls diese Bounds in den Präferenzen existieren, und {@code null} sonst.
     *
     * @return ein Rechteck oder {@code null}
     */
    @Nullable
    public Rectangle getPreferenceBounds() {
        if (_hasPreferences) {
            return new Rectangle(_preferencesX, _preferencesY, _preferencesWidth, _preferencesHeight);
        } else {
            return null;
        }
    }

    /**
     * Setzt eine benutzer-definierte Präferenz.
     *
     * @param key   der Schlüssel
     * @param value der Wert
     */
    public void putUserDefinedPreference(final String key, final Object value) {
        _userDefinedPreferences.put(key, value);
    }

    /**
     * Löscht die benutzer-definierte Präferenz mit dem übergebenen Schlüsssel.
     *
     * @param key der Schlüssel
     */
    @SuppressWarnings("unused")
    public void removeUserDefinedPreference(final String key) {
        _userDefinedPreferences.remove(key);
    }

    /**
     * Gibt die benutzer-definierte Präferenz zu dem übergebenen Schlüsssel zurück, wenn eine solche existiert, sonst {@code null}.
     *
     * @param key der Schlüssel
     *
     * @return die Präferenz oder {@code null}
     */
    @Nullable
    public Object getUserdefinedPreference(final String key) {
        return _userDefinedPreferences.get(key);
    }

    /**
     * Dies ist eine Methode, mit deren Hilfe man sicherstellt, dass folgende Dinge erfüllt sind: 1. Das Fenster erscheint innerhalb des Bildschirms.
     * 2. Gibt es Präferenzen, so wird zumindest eine Minimalgröße garantiert. 3. Gibt es keine Präferenzen 3.1 so werden für die Location defaultX
     * und defaultY verwendet, 3.2 und ist pack true, so wird die Größe mittels pack() bestimmt, 3.3 und ist pack false, werden die defaultWidth und
     * defaultHeight benutzt.
     *
     * @param minWidth
     * @param minHeight
     * @param defaultX
     * @param defaultY
     * @param pack
     * @param defaultWidth
     * @param defaultHeight
     */
    public void setPositionAndSize(final int minWidth, final int minHeight, final int defaultX, final int defaultY, final boolean pack,
                                   final int defaultWidth, final int defaultHeight) {
        readPreferenceBounds();
        if (hasPreferences()) {
            /* Hier wird sichergestellt, dass diese Instanz im Screen sichtbar wird. */
            final Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
            int x = Math.max(getPreferencesX(), 0);
            int y = Math.max(getPreferencesY(), 0);
            int width = Math.max(getPreferencesWidth(), minWidth);
            int height = Math.max(getPreferencesHeight(), minHeight);

            if (x > screenSize.width - width) {  // zu weit rechts
                x = Math.max(0, screenSize.width - width);
            }
            if (y > screenSize.height - height) { // zu weit unten
                y = Math.max(0, screenSize.height - height);
            }

            _frame.setLocation(x, y);
            _frame.setSize(width, height);
        } else {
            _frame.setLocation(defaultX, defaultY);
            if (pack) {
                _frame.pack();
            } else {
                _frame.setSize(defaultWidth, defaultHeight);
            }
        }
    }

    /**
     * Liest die Bounds des Frames aus den Präferenzen.
     */
    public final void readPreferenceBounds() {
        if (null == _identifier || _identifier.isEmpty()) {
            return;
        }
        Preferences prefs = getPreferenceNode();
        try {
            if (prefs.nodeExists(_identifier)) {
                Preferences node = prefs.node(_identifier);
                _preferencesX = node.getInt("x", Integer.MIN_VALUE);
                _preferencesY = node.getInt("y", Integer.MIN_VALUE);
                _preferencesWidth = node.getInt("width", Integer.MIN_VALUE);
                _preferencesHeight = node.getInt("height", Integer.MIN_VALUE);
                _hasPreferences =
                    _preferencesX != Integer.MIN_VALUE && _preferencesY != Integer.MIN_VALUE && _preferencesWidth != Integer.MIN_VALUE &&
                    _preferencesHeight != Integer.MIN_VALUE;

                if (LOG) {
                    System.out.println("Read " + _identifier + ": " + getPreferenceBounds());
                }

                Preferences userNode = node.node(USER_DEFINED);
                String[] children = userNode.childrenNames();
                for (String child : children) {
                    Preferences childNode = userNode.node(child);
                    String type = childNode.get(TYPE, "");
                    switch (type) {
                        case STRING: {
                            String value = childNode.get(VALUE, "");
                            _userDefinedPreferences.put(child, value);
                            break;
                        }
                        case INTEGER: {
                            Integer value = childNode.getInt(VALUE, Integer.MIN_VALUE);
                            _userDefinedPreferences.put(child, value);
                            break;
                        }
                        case BOOLEAN: {
                            Boolean value = childNode.getBoolean(VALUE, true);
                            _userDefinedPreferences.put(child, value);
                            break;
                        }
                        case DOUBLE: {
                            Double value = childNode.getDouble(VALUE, Double.MIN_VALUE);
                            _userDefinedPreferences.put(child, value);
                            break;
                        }
                        case FLOAT: {
                            Float value = childNode.getFloat(VALUE, Float.MIN_VALUE);
                            _userDefinedPreferences.put(child, value);
                            break;
                        }
                        case LONG: {
                            Long value = childNode.getLong(VALUE, Long.MIN_VALUE);
                            _userDefinedPreferences.put(child, value);
                            break;
                        }
                    }
                }

            }
        } catch (BackingStoreException ignored) {
            PreferencesDeleter pd =
                new PreferencesDeleter("Für ein Fenster konnten die Einstellungen der letzten Sitzungen nicht wiederhergestellt werden.", prefs);
            pd.run();
        }
    }

    /**
     * Speichert die Bounds des Frames in den Präferenzen.
     */
    public void storePreferenceBounds() {
        if (null == _identifier || _identifier.isEmpty()) {
            return;
        }
        Preferences prefs = getPreferenceNode();
        Preferences node = prefs.node(_identifier);
        node.putInt("x", _frame.getX());
        _preferencesX = _frame.getX();
        node.putInt("y", _frame.getY());
        _preferencesY = _frame.getY();
        node.putInt("width", _frame.getWidth());
        _preferencesWidth = _frame.getWidth();
        node.putInt("height", _frame.getHeight());
        _preferencesHeight = _frame.getHeight();

        Preferences userNode = node.node(USER_DEFINED);
        for (final Map.Entry<String, Object> entry : _userDefinedPreferences.entrySet()) {
            String key = entry.getKey();
            if (null == key || key.isEmpty()) {
                continue;
            }
            Preferences keyNode = userNode.node(entry.getKey());
            Object value = entry.getValue();
            if (null == value) {
                //noinspection UnnecessaryContinue
                continue;
            } else if (value instanceof String) {
                keyNode.put(TYPE, STRING);
                keyNode.put(VALUE, (String) value);
            } else if (value instanceof Integer) {
                keyNode.put(TYPE, INTEGER);
                keyNode.putInt(VALUE, (Integer) value);
            } else if (value instanceof Double) {
                keyNode.put(TYPE, DOUBLE);
                keyNode.putDouble(VALUE, (Double) value);
            } else if (value instanceof Boolean) {
                keyNode.put(TYPE, BOOLEAN);
                keyNode.putBoolean(VALUE, (Boolean) value);
            } else if (value instanceof Float) {
                keyNode.put(TYPE, FLOAT);
                keyNode.putFloat(VALUE, (Float) value);
            } else if (value instanceof Long) {
                keyNode.put(TYPE, LONG);
                keyNode.putLong(VALUE, (Long) value);
            }
        }

        if (LOG) {
            System.out.println("Stored " + _identifier + ": " + _frame.getBounds());
        }
    }

    @Override
    public String toString() {
        return "GndFrame{" + "_identifier='" + _identifier + '\'' + ", _hasPreferences=" + _hasPreferences + ", _preferencesX=" + _preferencesX +
               ", _preferencesY=" + _preferencesY + ", _preferencesWidth=" + _preferencesWidth + ", _preferencesHeight=" + _preferencesHeight + '}';
    }

    public void add(Component component) {
        _frame.add(component);
    }

    /* *******	Und hier die Methoden, die an den JFrame delegiert werden. ******* */

    public void add(Component component, Object constraints) {
        _frame.add(component, constraints);
    }

    public void addWindowListener(WindowListener listener) {
        _frame.addWindowListener(listener);
    }

    public void dispose() {
        _frame.dispose();
    }

    public Container getContentPane() {
        return _frame.getContentPane();
    }

    public Point getLocation() {
        return _frame.getLocation();
    }

    public void pack() {
        _frame.pack();
    }

    public void remove(Component component) {
        _frame.remove(component);
    }

    public void repaint() {
        _frame.repaint();
    }

    public void requestFocus() {
        _frame.requestFocus();
    }

    public void revalidate() {
        _frame.revalidate();
    }

    public void setDefaultCloseOperation(int operation) {
        _frame.setDefaultCloseOperation(operation);
    }

    public void setJMenuBar(JMenuBar menuBar) {
        _frame.setJMenuBar(menuBar);
    }

    public void setLayout(LayoutManager manager) {
        _frame.setLayout(manager);
    }

    public void setCursor(Cursor cursor) {
        _frame.setCursor(cursor);
    }

    public void setLocation(int x, int y) {
        _frame.setLocation(x, y);
    }

    public void setLocationRelativeTo(@Nullable Component component) {
        _frame.setLocationRelativeTo(component);
    }

    public void setPreferredSize(Dimension d) {
        _frame.setPreferredSize(d);
    }

    public void setSize(int x, int y) {
        _frame.setSize(x, y);
    }

    public void setState(int state) {
        _frame.setState(state);
    }

    public void setTitle(String title) {
        _frame.setTitle(title);
    }

    public void setVisible(boolean b) {
        _frame.setVisible(b);
    }

    public void toFront() {
        _frame.toFront();
    }

    private class MyWindowListener extends WindowAdapter {
        @Override
        public void windowClosing(WindowEvent e) {
            storePreferenceBounds();
        }
    }
}
