/*
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2021 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.kappich.pat.gnd.viewManagement;

import de.bsvrz.sys.funclib.debug.Debug;
import de.kappich.pat.gnd.displayObjectToolkit.DOTManager;
import de.kappich.pat.gnd.gnd.PreferencesHandler;
import de.kappich.pat.gnd.layerManagement.Layer;
import de.kappich.pat.gnd.layerManagement.LayerManager;
import de.kappich.pat.gnd.layerManagement.LayerManager.LayerManagerChangeListener;
import de.kappich.pat.gnd.utils.view.PreferencesDeleter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;
import javax.swing.table.AbstractTableModel;

/**
 * Die Ansichtsverwaltung ist ein Singleton, das unter anderem das Interface TableModel implementiert, damit es vom {@link ViewDialog Ansichtsdialog}
 * angezeigt werden kann.
 * <p>
 * Diese Klasse ist wie auch die {@link DOTManager Darstellungstyp-} und die {@link LayerManager Layerverwaltung} als Singleton implementiert, um den
 * Programmieraufwand für die Kommunikation verschiedener Objekte dieser Klasse untereinader, der andernfalls notwendig wäre, einzusparen.
 *
 * @author Kappich Systemberatung
 */
@SuppressWarnings("serial")
public final class ViewManager extends AbstractTableModel implements LayerManagerChangeListener {

    // Statische statt lazy Initialisierung da lazy nicht thread-safe. Statische Initialisierung
    // hat allerdings den Nachteil bei Exceptions. Deshalb immer auch Debug benutzen.
    public static final ViewManager _instance = new ViewManager();

    private static final Debug _debug = Debug.getLogger();
    private static final String[] columnNames = {"Name der Ansicht"};
	private final List<View> _views = new ArrayList<>() {
        @Override
        public boolean add(View view) {
            int index = Collections.binarySearch(this, view);
            if (index < 0) {
                index = ~index;
            }
            super.add(index, view);
            return true;
        }
    };
    private final HashMap<String, View> _viewsHashMap = new HashMap<>();
    private final Set<String> _unchangeables = new HashSet<>();

    private ViewManager() {
        initDefaultViews();
        initUserDefinedViews();
        LayerManager.getInstance().addChangeListener(this);
    }

    /**
     * Die übliche getInstance-Methode eines Singletons.
     *
     * @return die Instanz
     */
    public static ViewManager getInstance() {
        return _instance;
    }

    /**
     * Mit Hilfe dieser Methode kann man den ViewManager dazu zwingen, sich erneut zu konstruieren, was etwa nach dem Importieren von Präferenzen
     * sinnvoll ist.
     */
    public static void refreshInstance() {
        getInstance()._views.clear();
        getInstance()._unchangeables.clear();
        getInstance()._viewsHashMap.clear();
        getInstance().initDefaultViews();
        getInstance().initUserDefinedViews();
    }

    /**
     * Gibt den Ausgangspunkt der Präferenzen der Ansichtsverwaltung zurück.
     *
     * @return der Knoten, unter dem die Views gespeichert werden
     */
    public static Preferences getPreferenceStartPath() {
//		return Preferences.userRoot().node("de/kappich/pat/gnd/").node(kvPid).node("View");
        return PreferencesHandler.getInstance().getPreferenceStartPath().node("View");
    }

    private static void initDefaultViewsForLines(final View defaultView) {
        String s;
        s = "Störfällzustand OLSIM 1";
        final Layer layer1 = LayerManager.getInstance().getLayer(s);
        if (layer1 == null) {
            _debug.warning("ViewManager.initDefaultViewsForLines: ein Layer namens '" + s + "' konnte nicht gefunden werden.");
        } else {
            defaultView.addLayer(layer1);
        }
        s = "Straßennetz";
        final Layer layer2 = LayerManager.getInstance().getLayer(s);
        if (layer2 == null) {
            _debug.warning("ViewManager.initDefaultViewsForLines(): ein Layer namens '" + s + "' konnte nicht gefunden werden.");
        } else {
            defaultView.addLayer(layer2);
        }
        s = "Messquerschnitte (Testlayer)";
        final Layer layer3 = LayerManager.getInstance().getLayer(s);
        if (layer3 == null) {
            _debug.warning("ViewManager.initDefaultViewsForLines(): ein Layer namens '" + s + "' konnte nicht gefunden werden.");
        }
    }

    private static void initDefaultViewsForPoints(final View defaultView) {
        String s;
        Layer layer;

        s = "Messquerschnitte";
        layer = LayerManager.getInstance().getLayer(s);
        if (layer == null) {
            _debug.warning("ViewManager.initDefaultViewsForPoints: ein Layer namens '" + s + "' konnte nicht gefunden werden.");
        } else {
            defaultView.addLayer(layer);
        }
        s = "Messquerschnitte (erweitert)";
        layer = LayerManager.getInstance().getLayer(s);
        if (layer == null) {
            _debug.warning("ViewManager.initDefaultViewsForPoints: ein Layer namens '" + s + "' konnte nicht gefunden werden.");
        }

        s = "Messquerschnitte (Testlayer)";
        layer = LayerManager.getInstance().getLayer(s);
        if (layer == null) {
            _debug.warning("ViewManager.initDefaultViewsForPoints: ein Layer namens '" + s + "' konnte nicht gefunden werden.");
        }
    }

    private static void initDefaultViewsKmView(final View defaultView) {
        String s = "Betriebskilometrierung";
        final Layer kmLayer = LayerManager.getInstance().getLayer(s);
        if (null == kmLayer) {
            _debug.warning("ViewManager.initDefaultViewsKmView: ein Layer namens '" + s + "' konnte nicht gefunden werden.");
        } else {
            defaultView.addLayer(kmLayer, Integer.MAX_VALUE, 1, false, true);
        }
    }

    private static void initDefaultViewsStatView(final View defaultView) {
        String s = "ASB-Stationierung";
        final Layer statLayer = LayerManager.getInstance().getLayer(s);
        if (null == statLayer) {
            _debug.warning("ViewManager.initDefaultViewsKmView: ein Layer namens '" + s + "' konnte nicht gefunden werden.");
        } else {
            defaultView.addLayer(statLayer, Integer.MAX_VALUE, 1, false, true);
        }
    }

    private static void initDefaultViewsAsbNodesView(final View defaultView) {
        String s = "ASB-Knotennummern";
        final Layer asbNodeLayer = LayerManager.getInstance().getLayer(s);
        if (null == asbNodeLayer) {
            _debug.warning("ViewManager.initDefaultViewsKmView: ein Layer namens '" + s + "' konnte nicht gefunden werden.");
        } else {
            defaultView.addLayer(asbNodeLayer, Integer.MAX_VALUE, 1, false, true);
        }
    }

    private static void initDefaultViewsRnView(final View defaultView) {
        String s = "Autobahnschilder";
        final Layer rnLayer = LayerManager.getInstance().getLayer(s);
        if (null == rnLayer) {
            _debug.warning("ViewManager.initDefaultViewsKmView: ein Layer namens '" + s + "' konnte nicht gefunden werden.");
        } else {
            defaultView.addLayer(rnLayer, Integer.MAX_VALUE, 1, false, true);
        }
    }

    private void initDefaultViews() {
        View defaultView = getView("Vordefinierte Ansicht 1");
        if (defaultView == null) {
            List<ViewEntry> viewEntries = new ArrayList<>();
            defaultView = new View("Vordefinierte Ansicht 1", viewEntries);
        }

        initDefaultViewsRnView(defaultView);    // Autobahnschilder zu oberst.
        initDefaultViewsKmView(defaultView);    // Betriebskilometrierung
        initDefaultViewsStatView(defaultView);  // ASB-Stationierung
        initDefaultViewsAsbNodesView(defaultView);  // ASB-Knotennummern
        initDefaultViewsForPoints(defaultView);
        initDefaultViewsForLines(defaultView);
        defaultView.setSomethingChanged(false);

        internalAdd(defaultView);
        _unchangeables.add(defaultView.getName());
    }

    /**
     * Gibt {@code true} zurück, wenn die übergebene Ansicht eine benutzer-definierte Ansicht ist, und demzufolge verändert werden kann; andernfalls
     * ist der Rückgabewert {@code false}, wenn nämlich die Ansicht im Programmkode festgelegt wurde, und deshalb unveränderbar ist.
     * <p>
     * Die aktuell eingeblendete Ansicht kann immer verändert werden, auch wenn sie im Programmkode festgelegt wurde; allerdings werden die Änderungen
     * nicht gespeichert.
     *
     * @param view eine Ansicht
     *
     * @return ist die Ansicht änderbar?
     */
    public boolean isChangeable(final View view) {
        return !_unchangeables.contains(view.getName());
    }

    /**
     * Fügt die übergebene Ansicht hinzu, wenn noch keine Ansicht mit demselben Namen existiert.
     *
     * @param view eine Ansicht
     */
    public void addView(View view) {
        if (_viewsHashMap.containsKey(view.getName())) {
            throw new IllegalArgumentException("Ein View dieses Namens existiert bereits");
        }
        // Klein-/Großschreibung nicht (mehr) signifikant:
        for (View v : _views) {
            if (view.equals(v)) {
                throw new IllegalArgumentException(
                    "Es existiert bereits eine Ansicht, deren Name sich nur bezüglich Klein-/Großschreibung unterscheidet.");
            }
        }
        _views.add(view);
        _viewsHashMap.put(view.getName(), view);
        view.putPreferences(getPreferenceStartPath());
//		final int index = _views.size()-1;
//		fireTableRowsInserted(index, index); // das ist falsch, seit _views.add sortiert!
        fireTableDataChanged();
    }

    private void internalAdd(View view) {
        _views.add(view);
        _viewsHashMap.put(view.getName(), view);
    }

    /**
     * Entfernt die übergebene Ansicht aus der Ansichtverwaltung, die den Namen von view besitzt, falls eine solche existiert und diese nicht im
     * Programmkode definiert wurde.
     *
     * @param view eine Ansicht
     *
     * @return war das Löschen erfolgreich
     */
    public boolean removeView(View view) {
        final String name = view.getName();
        if (!_unchangeables.contains(name)) {
            final int index = remove(name);
            if (index > -1) {
                view.deletePreferences(getPreferenceStartPath());
                fireTableRowsDeleted(index, index);
                return true;
            }
        }
        return false;
    }

    /**
     * Leert die Ansichtsverwaltung komplett, also inklusive der im Programmkode definierten Ansichten.
     */
    @SuppressWarnings("unused")
    public void clearViews() {
        for (View view : _views) {
            removeView(view);
        }
    }

    private void initUserDefinedViews() {
        Preferences classPrefs = getPreferenceStartPath();
        String[] views;
        try {
            views = classPrefs.childrenNames();
        } catch (BackingStoreException | IllegalStateException ignore) {
            // Ein bei Programmstart nicht-vorhandener Node löst keine Exception aus.
            PreferencesDeleter pd = new PreferencesDeleter("Die benutzer-definierten Ansichten können nicht geladen werden.", classPrefs);
            pd.run();
            return;
        }
        for (String viewName : views) {
            Preferences viewPrefs = classPrefs.node(viewName);
            View view = new View();
            view.initializeFromPreferences(viewPrefs);
            internalAdd(view);
        }
    }

    /**
     * Gibt {@code true} zurück, falls die Ansichtsverwaltung eine Ansicht mit diesem Namen besitzt, und {@code false} sonst.
     *
     * @param viewName ein Ansichtsname
     *
     * @return {@code true} genau dann, wenn ein View mit dem übergebenen Namen existiert
     */
    public boolean hasView(String viewName) {
        return _viewsHashMap.containsKey(viewName);
    }

    /**
     * Gibt {@code true} zurück, falls die Ansichtsverwaltung eine Ansicht mit esem Namen besitzt, der sich allenfalls bezüglich Klein-Groß-Schreibung
     * unterscheidet. Sonst {@code false}.
     *
     * @param viewName ein Ansichtsname
     *
     * @return {@code true} genau dann, wenn ein View mit dem übergebenen Namen existiert
     */
    public boolean hasViewToLowerCase(String viewName) {
        for (View view : _views) {
            if (viewName.toLowerCase().equals(view.getName().toLowerCase())) {
                return true;
            }
        }
        return false;
    }

    /**
     * Gibt die Ansicht der Ansichtsverwaltung zurück, falls die Ansichtsverwaltung eine Ansicht mit diesem Namen besitzt, und {@code null} sonst.
     *
     * @param viewName ein Ansichtsname
     *
     * @return die Ansicht oder {@code null}
     */
    public View getView(String viewName) {
        return _viewsHashMap.get(viewName);
    }

    /**
     * Gibt die i-te Ansicht zurück (die Zählung beginnt bei 0).
     *
     * @param i ein Index
     *
     * @return die i-te Ansicht
     */
    public View getView(int i) {
        return _views.get(i);
    }

    /**
     * Bestimmt den Index der Ansicht in der Ansichtsverwaltung.
     *
     * @param viewName der Ansichtsname
     *
     * @return der Index der Ansicht
     */
    @SuppressWarnings("unused")
    public int getIndexOfView(final String viewName) {
        if (viewName == null) {
            return -1;
        }
        int index = 0;
        for (View view : _views) {
            if (viewName.equals(view.getName())) {
                return index;
            }
            index++;
        }
        return -1;
    }

    /**
     * Gibt eine Menge aller Ansichtsnamen zurück.
     *
     * @return eine Menge aller Ansichtsnamen
     */
    public Set<String> getViewNames() {
        return _viewsHashMap.keySet();
    }

    /**
     * Gibt den Index der ersten Ansicht, deren Name mit dem übergebenen Zeichen anfängt, zurück. Gibt es eine solche Ansicht nicht, so wird -1
     * zurückgegeben.
     *
     * @param c ein Buchstabe
     *
     * @return ein gültiger Index oder -1
     */
    public int getIndexOfFirstView(final char c) {
        char s = Character.toLowerCase(c);
        for (int index = 0; index < _views.size(); ++index) {
            View view = _views.get(index);
            char t = Character.toLowerCase(view.getName().charAt(0));
            if (s == t) {
                return index;
            }
        }
        return -1;
    }

    /**
     * Gibt die Anzahl der Spalten der Tabellendarstellung zurück.
     */
    @Override
    public int getColumnCount() {
        return columnNames.length;
    }

    /**
     * Gibt die Anzahl der Zeilen der Tabellendarstellung zurück.
     */
    @Override
    public int getRowCount() {
        return _views.size();
    }

    /**
     * Gibt den Spaltennamen der entsprechenden Spalte in der Tabellendarstellung zurück.
     *
     * @param columnIndex ein Spaltenindex
     */
    @Override
    public String getColumnName(int columnIndex) {
        return columnNames[columnIndex];
    }

    /**
     * Gibt den Wert des Feldes in der Tabellendarstellung zurück.
     *
     * @param rowIndex    ein Zeilenindex
     * @param columnIndex ein Spaltenindex
     */
    @Override
    public Object getValueAt(int rowIndex, int columnIndex) {
        return _views.get(rowIndex).getName();
    }

    private int remove(final String viewName) {
        int index = 0;
        for (View view : _views) {
            if (viewName.equals(view.getName())) {
                _views.remove(index);
                _viewsHashMap.remove(viewName);
                return index;
            }
            index++;
        }
        return -1;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("[ViewManager: ");
        for (View view : _views) {
            sb.append(view.toString());
        }
        return sb.toString();
    }

    /**
     * Schreibt eine kleine Statistik der Ansichtsverwaltung auf den Standardausgabekanal.
     */
    @SuppressWarnings({"UseOfSystemOutOrSystemErr", "unused"})
    public void getStatistics() {
        StringBuilder sb = new StringBuilder();
        for (View view : _views) {
            sb.append("[").append(view.getName()).append("]");
        }
        System.out.println("Anzahl Views: " + _views.size() + "  " + sb);
        System.out.println("Anzahl unveränderbarer Views: " + _unchangeables.size());
        if (_views.size() != _viewsHashMap.size()) {
            System.out.println("Interne Strukturen kaputt!");
        }
    }

    /**
     * Gibt eine Menge mit allen Namen aller in den Ansichten verwendeten Farben zurück.
     *
     * @return eine Menge mit allen benutzten Farben
     */
    public Set<String> getUsedColors() {
        Set<String> usedColors = new HashSet<>();
        for (View view : _views) {
            usedColors.addAll(view.getUsedColors());
        }
        return usedColors;
    }

    /*
     * Gehört zur Implementation von LayerManagerChangeListener und tut nichts.
     */
    @Override
    public void layerAdded(Layer layer) {
    }

    /*
     * Gehört zur Implementation von LayerManagerChangeListener.
     */
    @Override
    public void layerDefinitionChanged(Layer layer) {
        for (View view : _views) {
            view.layerDefinitionChanged(layer);
        }
    }

    /*
     * Gehört zur Implementation von LayerManagerChangeListener.
     */
    @Override
    public void layerPropertyChanged(Layer layer) {
        for (View view : _views) {
            view.layerPropertyChanged(layer);
        }
    }

    /*
     * Gehört zur Implementation von LayerManagerChangeListener.
     */
    @Override
    public void layerRemoved(String layerName) {
        for (View view : _views) { // Löschen ohne Veränderung des something-has-changed Status.
            final boolean changed = view.hasSomethingChanged();
            view.removeLayer(layerName);
            view.setSomethingChanged(changed);
        }
    }
}
