/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.sys.funclib.kappich.
 *
 * de.bsvrz.sys.funclib.kappich is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.sys.funclib.kappich is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.sys.funclib.kappich; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.sys.funclib.kappich.filechooser;

import static java.awt.FileDialog.LOAD;
import static java.awt.FileDialog.SAVE;


import java.awt.Component;
import java.awt.FileDialog;
import java.awt.Frame;
import java.awt.HeadlessException;
import java.io.File;
import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.filechooser.FileFilter;

/**
 * Implementiert die wesentliche Funktionalität eines JFileChooser mit Hilfe der AWT FileDialog Klasse, die unter macOS die nativen
 * Dateiauswahldialoge verwenden
 *
 * @author Kappich Systemberatung, Aachen
 * @version $Revision: 0000 $
 */
public final class AwtFileChooser extends JFileChooser implements FileChooser {
    private static File _lastUsedDirectory;

//	static final boolean ON_MAC = System.getProperty("os.name").toLowerCase().startsWith("mac");

    private AwtFileChooser() {
        // Setzt das Default-Verzeichnis auf das zuletzt verwendete
        setCurrentDirectory(readDefaultDirectory());
    }

    public static AwtFileChooser createFileChooser() {
        // Platzhalter, falls später andere Implementierungen benötigt werden
        return new AwtFileChooser();
    }

//	@Override
//	public void setCurrentDirectory(final File dir) {
//		// Ignorieren, damit Standardverhalten vom FileChooser verwendet wird
//	}

    @Override
    public int showOpenDialog() {
        if (isDirectorySelectionEnabled()) {
            return chooseFolder(null);
        }
        return chooseFiles(null, LOAD);
    }

    @Override
    public int showOpenDialog(Component parent) {
        if (isDirectorySelectionEnabled()) {
            return chooseFolder(parent);
        }
        return chooseFiles(parent, LOAD);
    }

    @Override
    public int showSaveDialog() {
        if (isDirectorySelectionEnabled()) {
            return chooseFolder(null);
        }
        return chooseFiles(null, SAVE);
    }

    @Override
    public int showSaveDialog(final Component parent) throws HeadlessException {
        if (isDirectorySelectionEnabled()) {
            return chooseFolder(parent);
        }
        return chooseFiles(parent, SAVE);
    }

    private int chooseFolder(final Component parent) {
        // Property bewirkt unter macOS, dass Ordner selektiert werden können
        System.setProperty("apple.awt.fileDialogForDirectories", "true");
        try {
            return chooseFiles(parent, LOAD);
        } finally {
            System.setProperty("apple.awt.fileDialogForDirectories", "false");
        }

    }

    private int chooseFiles(final Component parent, final int mode) {
        FileDialog chooser = createChooser(parent, mode);
        try {
            transferProperties(chooser);
            chooser.setVisible(true);
            return extractSelections(chooser);
        } finally {
            disposeChooser(chooser);
        }
    }

    private FileDialog createChooser(final Component parent, final int mode) {
        return new FileDialog(getFrameOf(parent), getDialogTitle(), mode);
    }

    /**
     * Versucht das Frame-Objekt zu finden, in dem die angegebene Komponente sich befindet. Wenn die übergebene Komponente selbst ein Frame ist, dann
     * wird sie zurückgeliefert. Ansonsten wird mit einem rekursiven Aufruf versucht, das Frame der Parent-Komponente zu ermitteln und diese
     * zurückgeliefert. Wenn so kein Frame gefunden wurde, dann wird `JOptionPane.getRootFrame()` als Default zurückgeliefert.
     *
     * @param component Komponente deren Frame gesucht wird.
     *
     * @return Frame der angegeben Komponente
     */
    private Frame getFrameOf(final Component component) {
        if (component instanceof Frame) {
            return (Frame) component;
        }
        if (component != null) {
            return getFrameOf(component.getParent());
        }
        return JOptionPane.getRootFrame();
    }

    private void disposeChooser(final FileDialog chooser) {
        chooser.dispose();
    }

    /**
     * Übertragen der Einstellungen am JFileChooser (this) auf den übergebenen FileDialog
     *
     * @param chooser FileDialog auf den die Einstellungen übertragen werden sollen.
     */
    protected void transferProperties(final FileDialog chooser) {
        chooser.setMultipleMode(isMultiSelectionEnabled());

        File currentDirectory = getCurrentDirectory();
        if (currentDirectory == null) {
            chooser.setDirectory(null);
        } else {
            chooser.setDirectory(currentDirectory.getAbsolutePath());
        }
        chooser.setFilenameFilter((dir, name) -> {
            FileFilter fileFilter = getFileFilter();
            return fileFilter == null || fileFilter.accept(new File(dir, name));
        });
//		if(preselectedFile != null) {
//			chooser.setDirectory(preselectedFile.getParent());
//			chooser.setFile(preselectedFile.getName());
//		}

    }

    protected int extractSelections(final FileDialog chooser) {

        final String fileName = chooser.getFile();

        File[] selectedFiles = chooser.getFiles();
        if (fileName != null && (selectedFiles == null || selectedFiles.length == 0)) {
            // Kommt vermutlich nicht vor
            selectedFiles = new File[] {new File(chooser.getDirectory(), fileName)};
        }
        setSelectedFiles(selectedFiles);
        if ((selectedFiles != null) && (selectedFiles.length != 0)) {
            System.out.println("chooser.getDirectory() = " + chooser.getDirectory());
            System.out.println("selected Dir = " + selectedFiles[0].getParentFile());
            saveDefaultDirectory(selectedFiles[0].getParentFile());
            return APPROVE_OPTION;
        } else {
            return CANCEL_OPTION;
        }
    }

    private void saveDefaultDirectory(final File directory) {
        // Todo: Unklar ist, wie hier nach Programm und Anwendungsfall unterschieden werden kann,
        // damit man jeweils ein eigenes Verzeichnis speichert.
        // Vorerst wird nicht persistent (d.h. pro Prozess und nicht pro Programm) und
        // unabhängig vom Anwendungsfall gespeichert.
        _lastUsedDirectory = directory;
    }

    private File readDefaultDirectory() {
        return _lastUsedDirectory;
    }

}
