/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.sys.funclib.kappich.
 *
 * de.bsvrz.sys.funclib.kappich is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.sys.funclib.kappich is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.sys.funclib.kappich; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.sys.funclib.kappich.properties;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.bsvrz.sys.funclib.kappich.collections.MultiValueMap;
import de.bsvrz.sys.funclib.kappich.propertytree.PropertyTreeModel;
import java.lang.annotation.Annotation;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.*;

/**
 * @author Kappich Systemberatung
*/
public final class ObjectProperties {

    public static final String FORMAT = "%-25s";
    private static final Multimap<Class<?>, Class<?>> _hooks = HashMultimap.create();

    public static void addClassHook(Class<?> source, Class<? extends PropertyModule> propertyHandler) {
        _hooks.put(source, propertyHandler);
    }

    public static String getPropertyText(final List<?> selectedItems) {
        final StringBuilder stringBuilder = new StringBuilder();

        for (Object selectedItem : selectedItems) {
            stringBuilder.append(selectedItem.toString()).append("\n");
        }

        Map<String, ObjectPropertyCategory> annotatedProperties = getObjectProperties(selectedItems);
        ObjectPropertyCategory emptyCategoryProperties = annotatedProperties.get("");
        if (emptyCategoryProperties != null) {
            appendCategory(stringBuilder, emptyCategoryProperties);
        }

        for (Map.Entry<String, ObjectPropertyCategory> categoryEntry : annotatedProperties.entrySet()) {
            if (!categoryEntry.getKey().isEmpty()) {
                stringBuilder.append(categoryEntry.getKey()).append("\n");
                appendCategory(stringBuilder, categoryEntry.getValue());
            }
        }
        return stringBuilder.toString();
    }

    private static void appendCategory(final StringBuilder stringBuilder, final ObjectPropertyCategory propertyCategory) {
        List<Map.Entry<PropertyName, Object>> entries = new ArrayList<>(propertyCategory.getProperties().entrySet());
	    entries.sort(new Comparator<>() {
            @Override
            public int compare(final Map.Entry<PropertyName, Object> o1, final Map.Entry<PropertyName, Object> o2) {
                return Integer.compare(o1.getKey().sortKey(), o2.getKey().sortKey());
            }
        });
        for (Map.Entry<PropertyName, Object> objectEntry : entries) {
            PropertyName propertyName = objectEntry.getKey();
            stringBuilder.append(String.format(FORMAT, propertyName.name() + ":"));
            Object value = objectEntry.getValue();
//			if(value instanceof CoordinateList){
//				stringBuilder.append(((CoordinateList<?>) value).size()).append(" Punkte\n");
//			}
//			else
	        if (value instanceof Collection<?> collection) {
                Iterator<?> iterator = collection.iterator();
                if (iterator.hasNext()) {
                    while (iterator.hasNext()) {
                        final Object o = iterator.next();
                        stringBuilder.append(objectToString(propertyName, o)).append("\n");
                        if (iterator.hasNext()) {
                            stringBuilder.append(String.format(FORMAT, ""));
                        }
                    }
                } else {
                    stringBuilder.append("<Keine>\n");
                }
	        } else if (value instanceof Map<?, ?> collection) {
                Iterator<? extends Map.Entry<?, ?>> iterator = collection.entrySet().iterator();
                stringBuilder.append("\n");
                if (iterator.hasNext()) {
                    while (iterator.hasNext()) {
                        final Map.Entry<?, ?> entry = iterator.next();
                        if (iterator.hasNext()) {
                            stringBuilder.append(String.format(FORMAT, objectToString(propertyName, entry.getKey() + ":")));
                            stringBuilder.append(objectToString(propertyName, entry.getValue()));
                        }
                    }
                } else {
                    stringBuilder.append("<Keine>\n");
                }
	        } else if (value instanceof MultiValueMap<?, ?> collection) {
                Iterator<? extends Map.Entry<?, ?>> iterator = collection.entrySet().iterator();
                stringBuilder.append("\n");
                if (iterator.hasNext()) {
                    while (iterator.hasNext()) {
                        final Map.Entry<?, ?> entry = iterator.next();
                        if (iterator.hasNext()) {
                            stringBuilder.append(String.format(FORMAT, objectToString(propertyName, entry.getKey() + ":")));
                            stringBuilder.append(objectToString(propertyName, entry.getValue()));
                        }
                    }
                } else {
                    stringBuilder.append("<Keine>\n");
                }
	        } else if (value instanceof Multimap<?, ?> collection) {
                Iterator<? extends Map.Entry<?, ?>> iterator = collection.asMap().entrySet().iterator();
                stringBuilder.append("\n");
                if (iterator.hasNext()) {
                    while (iterator.hasNext()) {
                        final Map.Entry<?, ?> entry = iterator.next();
                        if (iterator.hasNext()) {
                            stringBuilder.append(String.format(FORMAT, objectToString(propertyName, entry.getKey() + ":")));
                            stringBuilder.append(objectToString(propertyName, entry.getValue()));
                        }
                    }
                } else {
                    stringBuilder.append("<Keine>\n");
                }
            } else {
                stringBuilder.append(objectToString(propertyName, value)).append("\n");
            }
        }
        stringBuilder.append("\n");
    }

    public static String objectToString(final Object o) {
        return objectToString(null, o);
    }

    public static String objectToString(@Nullable final PropertyName propertyName, final Object o) {
        if (null != propertyName) {
            Class<? extends PropertyFormatter> c = propertyName.getFormatter();
            try {
                PropertyFormatter formatter = c.newInstance();
                return formatter.format(propertyName, o);
            } catch (InstantiationException | IllegalAccessException e) {
                e.printStackTrace();
                return "";
            }
        } else {
            DefaultFormatter formatter = new DefaultFormatter();
            return formatter.format(null, o);
        }
    }

    public static Map<String, ObjectPropertyCategory> getObjectProperties(final List<?> objects) {
        if (objects.isEmpty()) {
            return Collections.emptyMap();
        }
        if (objects.size() == 1) {
            return getObjectProperties(objects.get(0));
        }
        Multimap<String, ObjectPropertyCategory> tmpBagMap = HashMultimap.create();
        for (Object object : objects) {
            Map<String, ObjectPropertyCategory> objectProperties = getObjectProperties(object);
            for (Map.Entry<String, ObjectPropertyCategory> entry : objectProperties.entrySet()) {
                tmpBagMap.put(entry.getKey(), entry.getValue());
            }
        }
        final Map<String, ObjectPropertyCategory> result = new HashMap<>();
        for (Map.Entry<String, Collection<ObjectPropertyCategory>> entry : tmpBagMap.asMap().entrySet()) {
            result.put(entry.getKey(), merge(entry.getKey(), entry.getValue()));
        }
        return result;
    }

    public static Map<String, ObjectPropertyCategory> getObjectProperties(final Object o) {
        Class<?> clazz = o.getClass();

        final Set<Class<?>> implementedClasses = new HashSet<>();

        recursiveGetClasses(implementedClasses, clazz);

        final Map<String, ObjectPropertyCategory> result = new LinkedHashMap<>();

        for (Class<?> implementedClass : implementedClasses) {
            ObjectPropertyCategory opc = getClassAnnotationProperties(o, implementedClass);
            if (opc != null) {
                addCategory(result, opc);
            }
            for (ObjectPropertyCategory category : getClassModuleProperties(o, implementedClass)) {
                addCategory(result, category);
            }
            for (ObjectPropertyCategory category : getExternalClassModuleProperties(o, implementedClass)) {
                addCategory(result, category);
            }
        }

        return result;
    }

    private static void addCategory(final Map<String, ObjectPropertyCategory> result, @NotNull final ObjectPropertyCategory opc) {
        ObjectPropertyCategory old = result.put(opc.getName(), opc);
        if (old != null) {
            result.put(opc.getName(), merge(opc.getName(), Arrays.asList(old, opc)));
        }
    }

    public static ObjectPropertyCategory merge(final String name, final Collection<ObjectPropertyCategory> objectPropertyCategories) {
        if (objectPropertyCategories.size() == 1) {
            return objectPropertyCategories.iterator().next();
        }
        if (objectPropertyCategories.size() < 2) {
            throw new IllegalArgumentException();
        }
        Multimap<PropertyName, Object> tmpBagMap = HashMultimap.create();
        for (ObjectPropertyCategory objectPropertyCategory : objectPropertyCategories) {
            for (Map.Entry<PropertyName, Object> entry : objectPropertyCategory.getProperties().entrySet()) {
                tmpBagMap.put(entry.getKey(), entry.getValue());
            }
        }
        final Map<PropertyName, Object> tmpMap = new LinkedHashMap<>();
        for (ObjectPropertyCategory objectPropertyCategory : objectPropertyCategories) {
            for (PropertyName propName : objectPropertyCategory.getProperties().keySet()) {
                if (tmpMap.containsKey(propName)) {
                    break;
                }
                Collection<Object> values = tmpBagMap.get(propName);
                if (values.size() == 1) {
                    tmpMap.put(propName, values.iterator().next());
                } else {
                    tmpMap.put(propName, PropertyTreeModel.MULTIPLE);
                }
            }
        }
        return new ObjectPropertyCategory(name, tmpMap);
    }

    private static void recursiveGetClasses(final Set<Class<?>> result, final Class<?> clazz) {
        if (clazz == null) {
            return;
        }
        result.add(clazz);
        for (Class<?> iFace : clazz.getInterfaces()) {
            recursiveGetClasses(result, iFace);
        }
        recursiveGetClasses(result, clazz.getSuperclass());
    }

    @NotNull
    private static Collection<ObjectPropertyCategory> getClassModuleProperties(final Object selectedItem, final Class<?> clazz) {
        PropertyHandler annotation = clazz.getAnnotation(PropertyHandler.class);
        if (annotation != null) {
            final List<ObjectPropertyCategory> result = new ArrayList<>();
            for (Class<?> propClass : annotation.value()) {
                addProperties(selectedItem, clazz, propClass, result);
            }
            for (String className : annotation.className()) {
                try {
                    Class<?> propClass = Class.forName(className);
                    addProperties(selectedItem, clazz, propClass, result);
                } catch (ClassNotFoundException ignored) {
                }
            }
            return result;
        }
        return Collections.emptyList();
    }

    @NotNull
    private static Collection<ObjectPropertyCategory> getExternalClassModuleProperties(final Object selectedItem, final Class<?> clazz) {
        final List<ObjectPropertyCategory> result = new ArrayList<>();
        for (Class<?> propClass : _hooks.get(clazz)) {
            addProperties(selectedItem, clazz, propClass, result);
        }
        return result;
    }

    private static void addProperties(final Object object, final Class<?> objectClass, final Class<?> propertyHandler,
                                      final List<ObjectPropertyCategory> result) {
        try {
            Constructor<?> constructor = propertyHandler.getConstructor(objectClass);
            constructor.setAccessible(true);
            Object propertyModule = constructor.newInstance(object);
            ObjectPropertyCategory properties = getClassAnnotationProperties(propertyModule, propertyModule.getClass());
	        if (propertyModule instanceof PropertyModule module) {
                LinkedHashMap<String, Object> moduleProperties = module.getProperties();
                if (!moduleProperties.isEmpty()) {
                    final Map<PropertyName, Object> tmp = new HashMap<>();
                    int i = 0;
                    for (Map.Entry<String, Object> e : moduleProperties.entrySet()) {
                        tmp.put(new ManualProperty(e.getKey(), i), e.getValue());
                        i++;
                    }
                    result.add(new ObjectPropertyCategory(module.getName(), tmp));
                }
            }
            if (properties != null) {
                result.add(properties);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Nullable
    private static ObjectPropertyCategory getClassAnnotationProperties(final Object selectedItem, final Class<?> clazz) {
        PropertyClass annotation = clazz.getAnnotation(PropertyClass.class);
        if (annotation != null) {
            final Map<PropertyName, Object> result = new LinkedHashMap<>();
            Method[] methods = clazz.getDeclaredMethods();
            Field[] fields = clazz.getDeclaredFields();
            for (int i = methods.length - 1; i >= 0; i--) {
                final Method method = methods[i];
                addMethodProperty(result, selectedItem, method);
            }
            for (int i = fields.length - 1; i >= 0; i--) {
                final Field field = fields[i];
                addFieldProperty(result, selectedItem, field);
            }
            String categoryName = annotation.value();
            return new ObjectPropertyCategory(categoryName, result);
        }
        return null;
    }

    private static void addMethodProperty(final Map<PropertyName, Object> resultList, final Object o, final Method method) {
        PropertyName displayProperty = method.getAnnotation(PropertyName.class);
        if (displayProperty == null) {
            return;
        }
        try {
            method.setAccessible(true);
            Object result = method.invoke(o);
            addResult(resultList, displayProperty, result);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private static void addFieldProperty(final Map<PropertyName, Object> resultList, final Object o, final Field field) {
        PropertyName displayProperty = field.getAnnotation(PropertyName.class);
        if (displayProperty == null) {
            return;
        }
        try {
            field.setAccessible(true);
            Object result = field.get(o);
            addResult(resultList, displayProperty, result);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private static void addResult(final Map<PropertyName, Object> resultList, final PropertyName displayProperty, final Object result) {

	    if (result instanceof Iterable<?> iterable) { //  && !(result instanceof CoordinateList)
            final List<Object> entry = new ArrayList<>();
            for (Object val : iterable) {
                entry.add(val);
            }
            resultList.put(displayProperty, entry);
        } else {
            resultList.put(displayProperty, result);
        }
    }

    /**
     * @author Kappich Systemberatung
    */
    public static class ManualProperty implements PropertyName {
        private final String _name;
        private final int _sortKey;

        public ManualProperty(final String name, final int sortKey) {
            _name = name;
            _sortKey = sortKey;
        }

        @Override
        public String name() {
            return _name;
        }

        @Override
        public int sortKey() {
            return _sortKey;
        }

        @Override
        public Class<? extends Annotation> annotationType() {
            return PropertyName.class;
        }

        @Override
        public String unit() {
            return "";
        }

        @Override
        public int power() {
            return 1;
        }

        @Override
        public double factor() {
            return 1;
        }

        @Override
        public Class<? extends PropertyFormatter> getFormatter() {
            return DefaultFormatter.class;
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
	        if (!(o instanceof ManualProperty that)) {
                return false;
            }

	        if (!Objects.equals(_name, that._name)) {
                return false;
            }

            return true;
        }

        @Override
        public int hashCode() {
            return _name != null ? _name.hashCode() : 0;
        }
    }
}
