/*
 * Copyright 2012-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.sys.funclib.kappich.
 *
 * de.bsvrz.sys.funclib.kappich is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.sys.funclib.kappich is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.sys.funclib.kappich; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.sys.funclib.kappich.propertytree;

import com.google.common.collect.Multimap;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.bsvrz.sys.funclib.kappich.collections.MultiValueMap;
import de.bsvrz.sys.funclib.kappich.properties.ObjectProperties;
import de.bsvrz.sys.funclib.kappich.properties.ObjectPropertyCategory;
import de.bsvrz.sys.funclib.kappich.properties.PropertyClass;
import de.bsvrz.sys.funclib.kappich.properties.PropertyName;
import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArrayList;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.event.TreeModelListener;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;

/**
 * Ein TreeModel, welches die Eigenschaften eines Objektes (oder ggf. mehrerer Objekte) anzeigt. Die angezeigten Eigenschaften eines Objekts werden
 * durch die {@link PropertyName}- und {@link PropertyClass}-Annotations festgelegt.
 *
 * @author Kappich Systemberatung
*/
public class PropertyTreeModel implements TreeModel {

    /**
     * Spezielles Objekt, dass den Wert "{@code <Unterschiedlich>}" symbolisiert, also bei mehreren gleichzeitig ausgewählten Objekten anzeigt, dass
     * diese sich in der betreffenden Eigenschaft unterscheiden.
     */
    public static final Object MULTIPLE = new Object();

    /**
     * Wurzel
     */
    private final TreeNode _root;

    /**
     * Sollen bei mehreren ausgewählten Objekten die Gemeinsamkeiten angezeigt werden?
     */
    private final boolean _showCommonProperties;

    private final List<TreeModelListener> _listeners = new CopyOnWriteArrayList<>();

    private Icon _multiIcon = new ImageIcon(PropertyTreeModel.class.getResource("/de/bsvrz/sys/funclib/kappich/selectionlist/multi.png"));
    private Icon _collectionIcon = new ImageIcon(PropertyTreeModel.class.getResource("/de/bsvrz/sys/funclib/kappich/selectionlist/collection.png"));
    private Icon _propertyIcon = new ImageIcon(PropertyTreeModel.class.getResource("/de/bsvrz/sys/funclib/kappich/selectionlist/property.png"));
    private Icon _objectIcon = new ImageIcon(PropertyTreeModel.class.getResource("/de/bsvrz/sys/funclib/kappich/selectionlist/object.png"));
    private Icon _groupIcon = new ImageIcon(PropertyTreeModel.class.getResource("/de/bsvrz/sys/funclib/kappich/selectionlist/group.png"));

    /**
     * Erstellt ein neues PropertyTreeModel. Sollen keine gemeinsamen Eigenschaften mehrerer Objekte angezeigt werden kann in der Regel eine Liste mit
     * einem Element (dem anzuzeigenden Objekt) übergeben werden.
     *
     * @param o Liste mit anzuzeigenden Objekten
     */
    public PropertyTreeModel(List<?> o) {
        this(o, true);
    }

    /**
     * Erstellt ein neues PropertyTreeModel. Sollen keine gemeinsamen Eigenschaften mehrerer Objekte angezeigt werden kann in der Regel eine Liste mit
     * einem Element (dem anzuzeigenden Objekt) übergeben werden.
     *
     * @param o                    Liste mit anzuzeigenden Objekten
     * @param showCommonProperties Sollen bei mehreren Objekten die gemeinsamen/Unterschiedlichen Eigenschaften separat angezeigt werden?
     */
    public PropertyTreeModel(List<?> o, final boolean showCommonProperties) {
        if (o.size() == 1) {
            _root = new ObjectNode(null, o.iterator().next());
        } else {
            _root = new ObjectNode(null, o);
        }
        _showCommonProperties = showCommonProperties;
    }

    @Override
    public Object getRoot() {
        return _root;
    }

    @Override
    public Object getChild(final Object parent, final int index) {
        return ((TreeNode) parent).getChildren().get(index);
    }

    @Override
    public int getChildCount(final Object parent) {
        return ((TreeNode) parent).getChildren().size();
    }

    @Override
    public boolean isLeaf(final Object node) {
        return getChildCount(node) == 0;
    }

    @Override
    public void valueForPathChanged(final TreePath path, final Object newValue) {
    }

    @Override
    public int getIndexOfChild(final Object parent, final Object child) {
        if (parent == null || child == null) {
            return -1;
        }
        return ((TreeNode) parent).getChildren().indexOf(child);
    }

    @Override
    public void addTreeModelListener(final TreeModelListener l) {
        _listeners.add(l);
    }

    @Override
    public void removeTreeModelListener(final TreeModelListener l) {
        _listeners.remove(l);
    }

    /**
     * Erstellt ein TreeNode für eine Kategorie
     *
     * @param result Ergebnisliste, in die der fertige Knoten eingefügtw werden soll
     * @param entry  Kategorie-Entry
     *
     * @return Falls keine Kategorie erstellt werden soll (weil die Eigenschaften in die Wurzel eingefügt werden sollen) werden sie stattdessen hier
     *     zurückgegeben.
     */
    private List<? extends TreeNode> addCatNode(final List<TreeNode> result, final Map.Entry<String, ObjectPropertyCategory> entry) {
        CategoryNode node = new CategoryNode(entry.getKey(), entry.getValue());
        if (entry.getKey().isEmpty()) {
            return node.getChildren();
        }
        result.add(node);
        return Collections.emptyList();
    }

    /**
     * Setzt das Symbol für Gemeinsame-Eigenschaften-Nodes
     *
     * @param icon
     */
    public void setMultiIcon(final Icon icon) {
        _multiIcon = icon;
    }

    /**
     * Setzt das Symbol für Collection-Nodes
     *
     * @param icon
     */
    public void setCollectionIcon(final Icon icon) {
        _collectionIcon = icon;
    }

    /**
     * Setzt das Symbol für Eigenschaften-Nodes
     *
     * @param icon
     */
    public void setPropertyIcon(final Icon icon) {
        _propertyIcon = icon;
    }

    /**
     * Setzt das Symbol für Objekt-Nodes
     *
     * @param icon
     */
    public void setObjectIcon(final Icon icon) {
        _objectIcon = icon;
    }

    /**
     * Setzt das Symbol für Kategorie-Nodes
     *
     * @param icon
     */
    public void setGroupIcon(final Icon icon) {
        _groupIcon = icon;
    }

    abstract static class TreeNode {
        private Icon _icon;

        public abstract List<? extends TreeNode> getChildren();

        public Icon getIcon() {
            return _icon;
        }

        protected void setIcon(final String icon) {
            _icon = new ImageIcon(getClass().getResource(icon));
        }

        protected void setIcon(final Icon icon) {
            _icon = icon;
        }
    }

    private class MergedObjectNode extends TreeNode {

        private final List<?> _objects;
        private List<TreeNode> _children;

        public MergedObjectNode(final Collection<?> objects) {
            _objects = new ArrayList<Object>(objects);
            setIcon(_multiIcon);
        }

        void generateChildren() {
            if (_objects != null) {
                Map<String, ObjectPropertyCategory> properties = ObjectProperties.getObjectProperties(_objects);
                final List<TreeNode> result = new ArrayList<>(properties.size());
                for (Map.Entry<String, ObjectPropertyCategory> entry : properties.entrySet()) {
                    result.addAll(0, addCatNode(result, entry));
                }
                _children = result;
            } else {
                _children = Collections.emptyList();
            }
        }

        @Override
        public List<TreeNode> getChildren() {
            if (_children == null) {
                generateChildren();
            }
            return Collections.unmodifiableList(_children);
        }

        @Override
        public String toString() {
            return "Gemeinsame Eigenschaften";
        }
    }

    public class ObjectNode extends TreeNode implements Comparable<ObjectNode> {
        final Object _object;
        private final PropertyName _propertyName;
        List<TreeNode> _children;

        public ObjectNode(@Nullable final PropertyName propertyName, Object object) {
            if (object instanceof Iterable) {
                setIcon(_collectionIcon);
            } else //noinspection VariableNotUsedInsideIf
                if (propertyName != null) {
                    setIcon(_propertyIcon);
                } else {
                    setIcon(_objectIcon);
                }

            _propertyName = propertyName;
            _object = object;
        }

        public Object getObject() {
            return _object;
        }

        void generateChildren() {
            if (_object == null) {
                _children = Collections.emptyList();
                return;
            }
            final List<TreeNode> result = new ArrayList<>();
            if (_object instanceof Iterable) {
	            if (_object instanceof Collection<?> col) {
                    if (col.size() > 1 && _showCommonProperties) {
                        result.add(new MergedObjectNode(col));
                    }
                }
                for (Object entry : (Iterable<?>) _object) {
                    result.add(new ObjectNode(null, entry));
                }
            } else if (_object.getClass().isArray()) {
                for (int i = 0; i < Array.getLength(_object); i++) {
                    result.add(new ObjectNode(null, Array.get(_object, i)));
                }
            } else if (_object instanceof Map) {
                int i = 0;
                for (Map.Entry<?, ?> entry : ((Map<?, ?>) _object).entrySet()) {
                    if (entry.getKey() instanceof String) {
                        result.add(new ObjectNode(new ObjectProperties.ManualProperty((String) entry.getKey(), i), entry.getValue()));
                        i++;
                    } else {
                        result.add(new EntryNode(entry));
                    }
                }
            } else if (_object instanceof MultiValueMap) {
                int i = 0;
                for (Map.Entry<?, ?> entry : ((MultiValueMap<?, ?>) _object).entrySet()) {
                    if (entry.getKey() instanceof String) {
                        result.add(new ObjectNode(new ObjectProperties.ManualProperty((String) entry.getKey(), i), entry.getValue()));
                        i++;
                    } else {
                        result.add(new EntryNode(entry));
                    }
                }
            } else if (_object instanceof Multimap) {
                int i = 0;
                for (Map.Entry<?, ?> entry : ((Multimap<?, ?>) _object).asMap().entrySet()) {
                    if (entry.getKey() instanceof String) {
                        result.add(new ObjectNode(new ObjectProperties.ManualProperty((String) entry.getKey(), i), entry.getValue()));
                        i++;
                    } else {
                        result.add(new EntryNode(entry));
                    }
                }
            }
            Map<String, ObjectPropertyCategory> properties = ObjectProperties.getObjectProperties(_object);
            for (Map.Entry<String, ObjectPropertyCategory> entry : properties.entrySet()) {
                result.addAll(0, addCatNode(result, entry));
            }
            _children = result;
        }

        @Override
        public List<TreeNode> getChildren() {
            if (_children == null) {
                generateChildren();
            }
            return Collections.unmodifiableList(_children);
        }

        @Override
        public String toString() {
            String s = ObjectProperties.objectToString(_propertyName, _object);
            if (_propertyName == null) {
                return s;
            }
            return _propertyName.name() + ": " + s;
        }

        @Override
        public int compareTo(final ObjectNode o) {
            int x = _propertyName.sortKey();
            int y = o._propertyName.sortKey();
            return Integer.compare(x, y);
        }
    }

    private class EntryNode extends ObjectNode {
        private final Map.Entry<?, ?> _entry;

        public EntryNode(final Map.Entry<?, ?> entry) {
            super(null, entry.getKey());
            _entry = entry;
        }

        @Override
        void generateChildren() {
            super.generateChildren();
            _children = new ArrayList<>(_children);
            _children.add(new ObjectNode(null, _entry.getValue()));
        }
    }

    private class CategoryNode extends TreeNode {
        private final String _name;
        private final ObjectPropertyCategory _category;
        private List<ObjectNode> _children;

        public CategoryNode(final String name, final ObjectPropertyCategory category) {
            setIcon(_groupIcon);
            _name = name;
            _category = category;
        }

        private void generateChildren() {
            Map<PropertyName, Object> properties = _category.getProperties();
            final List<ObjectNode> result = new ArrayList<>(properties.size());
            for (Map.Entry<PropertyName, Object> entry : properties.entrySet()) {
                result.add(new ObjectNode(entry.getKey(), entry.getValue()));
            }
            Collections.sort(result);
            _children = result;
        }

        @Override
        public List<ObjectNode> getChildren() {
            if (_children == null) {
                generateChildren();
            }
            return Collections.unmodifiableList(_children);
        }

        @Override
        public String toString() {
            return _name;
        }
    }
}
