/*
 * Copyright 2010-2020 by Kappich Systemberatung, Aachen
 * Copyright 2021 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.dav.dav.
 *
 * de.bsvrz.dav.dav is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.dav is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.dav.dav.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.dav.dav.communication.accessControl;

import de.bsvrz.dav.daf.accessControl.AccessControlManager;
import de.bsvrz.dav.daf.accessControl.UserInfo;
import de.bsvrz.dav.daf.communication.lowLevel.telegrams.BaseSubscriptionInfo;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.config.AttributeGroupUsage;
import de.bsvrz.dav.daf.main.config.ConfigurationArea;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.MutableSet;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.sys.funclib.dataSerializer.Deserializer;
import de.bsvrz.sys.funclib.dataSerializer.NoSuchVersionException;
import de.bsvrz.sys.funclib.dataSerializer.Serializer;
import de.bsvrz.sys.funclib.dataSerializer.SerializingFactory;
import de.bsvrz.sys.funclib.debug.Debug;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

/**
 * Zugriffssteuerungs-Plugin für den Datenverteiler, das Konfigurationsänderungen überwacht.
 *
 * @author Kappich Systemberatung
 * @version $Revision: 0000 $
 */
public class ConfigAccessControlPlugin implements AccessControlPlugin {

    private static final Debug _debug = Debug.getLogger();
    private AccessControlManager _accessControlManager;
    private DataModel _dataModel;

    /**
     * Verarbeitet ein übergebenes Datenobjekt
     *
     * @param data        Datenobjekt
     * @param dataHandler Lambda-Ausdruck, der entscheidet, ob Aktion erlaubt, oder nicht.
     *
     * @return Ggf. modifiziertes Datenobjekt.
     */
    private static Data handleData(final Data data, UserInfo user, final DataHandler dataHandler) {
        final byte[] queryMessage = data.getUnscaledArray("daten").getByteArray();
        try {
            // Versuchen, die Daten zu deserialisieren
            final Deserializer deserializer = SerializingFactory.createDeserializer(2, new ByteArrayInputStream(queryMessage));
            return dataHandler.handleData(deserializer);

        } catch (Exception e) {
            _debug.warning("Fehler beim Analysieren der Nachricht", e);
            // Aktion nicht erlaubt, Datenobjekt "unschädlich machen".
            return modifyData(data, user, e);
        }
    }

    /**
     * Modifiziert ein angegebenes Datenobjekt
     *
     * @param data       Datenobjekt, das eine Konfigurationsanfrage zum Ändern von Mengen beinhaltet
     * @param user       Benutzer
     * @param parameters Weitere Informationen zur verbotenen Aktion (z. B. Name der Aktion oder verursachender Fehler)
     *
     * @return modifiziertes Datenobjekt
     */
    private static Data modifyData(final Data data, final UserInfo user, final Object... parameters) {
        final Data modifiableCopy = data.createModifiableCopy();
        final ByteArrayOutputStream byteArrayStream = new ByteArrayOutputStream();
        try {
            final Serializer serializer = SerializingFactory.createSerializer(2, byteArrayStream);
            serializer.writeString(
                    "Aktion: " + data.getItem("nachrichtenTyp").asTextValue().getText()
                            + " (" + Arrays.stream(parameters).map(String::valueOf).collect(Collectors.joining(", ")) + "), Benutzer: " +
                user);
        } catch (NoSuchVersionException | IOException e) {
            _debug.warning("Fehler beim Analysieren der Nachricht", e);
        }
        modifiableCopy.getUnscaledArray("daten").set(byteArrayStream.toByteArray());
        modifiableCopy.getItem("nachrichtenTyp").asTextValue().setText("KonfigurationsänderungVerweigert");
        return modifiableCopy;
    }

    @Override
    public void initialize(final AccessControlManager accessControlManager, final ClientDavInterface clientDavInterface) {
        _accessControlManager = accessControlManager;
        _dataModel = clientDavInterface.getDataModel();
    }

    @Override
    public Collection<AttributeGroupUsage> getAttributeGroupUsagesToFilter() {
        return List.of(_dataModel.getAttributeGroupUsage("atg.konfigurationsAnfrageSchnittstelleSchreibend", "asp.anfrage"));
    }

    @Override
    public Data handleData(final long userID, final BaseSubscriptionInfo baseSubscriptionInfo, final Data data) {
        final AttributeGroupUsage usage = _dataModel.getAttributeGroupUsage(baseSubscriptionInfo.getUsageIdentification());

        // Stimmt die Attributgruppenverwendung des Datenpakets mit unserem Filter überein?
        // Diese Prüfung ist eigentlich unnötig, wird aber sinnvoll, wenn mehrere ATGUs gefiltert werden sollen
        if (usage.getAttributeGroup().getPid().equals("atg.konfigurationsAnfrageSchnittstelleSchreibend")) {

            // Benutzerrechte ermitteln
            final UserInfo userInfo = _accessControlManager.getUserPermissions(userID);

            // Data-Objekt analysieren
            final String request = data.getItem("nachrichtenTyp").asTextValue().getText();
            switch (request) {
                case "DynamischeMengeElementeÄndern":
                case "KonfigurierendeMengeElementeÄndern":

                    return handleData(data, userInfo, (deserializer) -> {
                        final MutableSet mutableSet = (MutableSet) deserializer.readObjectReference(_dataModel);

                        // Benutzerrechte für diesen Typ prüfen. Falls erlaubt originales Datenobjekt zurückgeben.
                        if (userInfo.mayModifyObjectSet(mutableSet.getConfigurationArea(), mutableSet.getObjectSetType())) {
                            return data;
                        }
                        return modifyData(data, userInfo, mutableSet);
                    });
                case "KonfigurierendenDatensatzFestlegen":
                    return handleData(data, userInfo, (deserializer) -> {
                        final AttributeGroupUsage attributeGroupUsage = (AttributeGroupUsage) deserializer.readObjectReference(_dataModel);
                        final MutableSet object = (MutableSet) deserializer.readObjectReference(_dataModel);

                        // Benutzerrechte für diesen Typ prüfen. Falls erlaubt originales Datenobjekt zurückgeben.
                        if (userInfo.mayCreateModifyRemoveObject(object.getConfigurationArea(), object.getType())) {
                            return data;
                        }
                        return modifyData(data, userInfo, object, attributeGroupUsage);
                    });
                case "ObjektAnlegen":
                case "DynamischesObjektMitKonfigurierendenDatensaetzenAnlegen":
                    return handleData(data, userInfo, (deserializer) -> {
                        if (request.equals("ObjektAnlegen")) {
                            deserializer.readBoolean(); // konfigurierend ja/nein
                        }
                        final ConfigurationArea configurationArea = (ConfigurationArea) deserializer.readObjectReference(_dataModel);
                        final String pid = deserializer.readString();
                        deserializer.readString(); // name
                        final SystemObjectType type = (SystemObjectType) deserializer.readObjectReference(_dataModel);

                        // Benutzerrechte für diesen Typ prüfen. Falls erlaubt originales Datenobjekt zurückgeben.
                        if (userInfo.mayCreateModifyRemoveObject(configurationArea, type)) {
                            return data;
                        }
                        return modifyData(data, userInfo, pid, type);
                    });
                case "ObjektKopieren":
                case "ObjektMengenBearbeiten":
                case "ObjektLöschen":
                case "ObjektWiederherstellen":
                case "ObjektNamenÄndern":
                    return handleData(data, userInfo, (deserializer) -> {
                        final SystemObject object = deserializer.readObjectReference(_dataModel);

                        // Benutzerrechte für diesen Typ prüfen. Falls erlaubt originales Datenobjekt zurückgeben.
                        if (userInfo.mayCreateModifyRemoveObject(object.getConfigurationArea(), object.getType())) {
                            return data;
                        }
                        return modifyData(data, userInfo, object);
                    });
            }
        }
        return data;
    }

    @Override
    public String toString() {
        return "ConfigAccessControlPlugin";
    }

    @FunctionalInterface
    interface DataHandler {
        Data handleData(final Deserializer deserializer) throws IOException;
    }
}
