/*
 * Copyright 2011-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.dav.
 *
 * de.bsvrz.dav.dav is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.dav is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.dav.dav.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.dav.main;

import de.bsvrz.dav.daf.communication.protocol.UserLogin;
import de.bsvrz.dav.daf.main.ClientDavConnection;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.DataAndATGUsageInformation;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.AttributeGroupUsage;
import de.bsvrz.dav.daf.main.config.ClientApplication;
import de.bsvrz.dav.daf.main.config.ConfigurationArea;
import de.bsvrz.dav.daf.main.config.ConfigurationChangeException;
import de.bsvrz.dav.daf.main.config.ConfigurationObject;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.DynamicObject;
import de.bsvrz.dav.daf.main.config.DynamicObjectType;
import de.bsvrz.dav.daf.main.config.MutableSet;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.debug.Debug;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * Klasse, die Applikationsobjekte für die verbundenen Anwendungen erstellt
 *
 * @author Kappich Systemberatung
 */
public final class ApplicationObjectManager {

    private static final Debug DEBUG = Debug.getLogger();

    private final DataModel _dataModel;

    private final MutableSet _applicationsSet;
    private final ClientDavConnection _connection;
    private final ConfigurationObject _davObject;
    private ConfigurationArea _appObjectConfigurationArea;
    private Boolean _canWriteApplicationSet; // null = unbekannt, true = schreiben erlaubt, false = schreiben verboten

    /**
     * Konstruktor
     *
     * @param connectionsManager                 HighLevelConnectionsManagerInterface
     * @param connection                         Loakle Datenverteilerverbindung
     * @param configAreaPidForApplicationObjects Pid des Konfigurationsbereichs für Applikationsobjekte wie in ServerDavParameters angegeben
     */
    public ApplicationObjectManager(final HighLevelConnectionsManagerInterface connectionsManager, final ClientDavConnection connection,
                                    final String configAreaPidForApplicationObjects) {
        _connection = connection;
        _dataModel = _connection.getDataModel();
        _davObject = connectionsManager.getDavObject();
        _applicationsSet = _davObject.getMutableSet("Applikationen");

        String defaultAreaPid = connection.getLocalConfigurationAuthority()
            .getConfigurationData(_dataModel.getAttributeGroup("atg.konfigurationsVerantwortlicherEigenschaften")).getTextArray("defaultBereich")
            .getText(0);

        final ConfigurationArea defaultConfigArea = _dataModel.getObject(defaultAreaPid, ConfigurationArea.class);

        if (configAreaPidForApplicationObjects == null || configAreaPidForApplicationObjects.isEmpty()) {
            _appObjectConfigurationArea = defaultConfigArea;
        } else {
            _appObjectConfigurationArea = _dataModel.getConfigurationArea(configAreaPidForApplicationObjects);
            if (_appObjectConfigurationArea == null) {
                DEBUG.warning(
                    "Angegebener Konfigurationsbereich für Applikationsobjekte '" + configAreaPidForApplicationObjects + "' nicht gefunden. " +
                    "Es wird der Defaultbereich der Konfiguration verwendet");
                _appObjectConfigurationArea = defaultConfigArea;
            }
            if (!_appObjectConfigurationArea.getConfigurationAuthority().equals(connection.getLocalConfigurationAuthority())) {
                DEBUG.warning(
                    "Angegebener Konfigurationsbereich für Applikationsobjekte '" + configAreaPidForApplicationObjects + "' ist nicht änderbar. " +
                    "Es wird der Defaultbereich der Konfiguration verwendet");
                _appObjectConfigurationArea = defaultConfigArea;
            }
        }
        if (_applicationsSet != null) {
            if (canNotWriteApplicationSet()) {
                return;
            }
            try {
                // Menge leeren
                List<SystemObject> elements;
                while (!(elements = _applicationsSet.getElements()).isEmpty()) {
                    _applicationsSet.remove(elements.toArray(new SystemObject[0]));
                }
            } catch (ConfigurationChangeException e) {
                showApplicationSetErrorMessage();
                e.printStackTrace();
            }
        }
    }

    /**
     * Erstellt ein Applikations-Objekt und gibt die Id zurück
     *
     * @param typePid   Pid des Typs der Applikation
     * @param name      Name der Applikation
     * @param userLogin
     *
     * @return Applikations-Id oder -1 bei Fehler
     *
     * @throws ConfigurationChangeException Fehler bei Konfigurationsänderung
     */
    public long createApplication(final String typePid, final String name, final UserLogin userLogin) throws ConfigurationChangeException {
        if (_dataModel != null) {
            final DynamicObjectType type = (DynamicObjectType) _dataModel.getType(typePid);
            if (type != null) {
                final Collection<DataAndATGUsageInformation> data = new ArrayList<>();

                AttributeGroup attributeGroup = _dataModel.getAttributeGroup("atg.applikationsEigenschaften");
                if (attributeGroup != null) {
                    data.add(createApplicationData(attributeGroup, userLogin));
                }

                DynamicObject application = _appObjectConfigurationArea.createDynamicObject(type, "", name, data);
                if (application != null) {
                    addApplicationToObjectSet(application);
                    return application.getId();
                }
            }
        }
        return -1L;
    }

    private DataAndATGUsageInformation createApplicationData(final AttributeGroup attributeGroup, final UserLogin userLogin) {
        AttributeGroupUsage attributeGroupUsage = attributeGroup.getAttributeGroupUsage("asp.eigenschaften");
        Data data = _connection.createData(attributeGroup);
        data.getReferenceValue("Datenverteiler").setSystemObject(_davObject);
        if (userLogin.isRegularUser()) {
            data.getReferenceValue("Benutzer").setSystemObject(_dataModel.getObject(userLogin.getRemoteUserId()));
        }
        return new DataAndATGUsageInformation(attributeGroupUsage, data);
    }

    private void addApplicationToObjectSet(final SystemObject applicationObject) {
        if (canNotWriteApplicationSet()) {
            return;
        }
        if (!_connection.isConnected()) {
            return;
        }

        if (_applicationsSet != null) {
            try {
                _applicationsSet.add(applicationObject);
            } catch (ConfigurationChangeException e) {
                showApplicationSetErrorMessage();
                e.printStackTrace();
            }
        }
    }

    private void removeApplicationFromObjectSet(final SystemObject applicationObject) {
        if (canNotWriteApplicationSet()) {
            return;
        }
        if (!_connection.isConnected()) {
            return;
        }

        if (_applicationsSet != null) {
            try {
                _applicationsSet.remove(applicationObject);
            } catch (ConfigurationChangeException e) {
                showApplicationSetErrorMessage();
                e.printStackTrace();
            }
        }
    }

    private boolean canNotWriteApplicationSet() {
        // Logik analog zu de.bsvrz.puk.config.configFile.datamodel.ConfigMutableSet.loadElementAccessProperties()
        if (_canWriteApplicationSet != null) {
            return !_canWriteApplicationSet;
        }

        boolean elementChangesAllowed =
            _dataModel.getConfigurationAuthority().equals(_applicationsSet.getConfigurationArea().getConfigurationAuthority());

        final AttributeGroup atg = _dataModel.getAttributeGroup("atg.dynamischeMenge");
        if (atg != null) {
            final Data data = _applicationsSet.getConfigurationData(atg);
            if (data != null) {
                String managementPid = data.getTextValue("verwaltung").getValueText();
                elementChangesAllowed = _dataModel.getConfigurationAuthority().getPid().equals(managementPid);
            }
        }

        _canWriteApplicationSet = elementChangesAllowed;
        return !_canWriteApplicationSet;
    }

    private void showApplicationSetErrorMessage() {
        if (_canWriteApplicationSet == false) {
            return;
        }
        _canWriteApplicationSet = false;
        DEBUG.warning("Beim Ändern der Menge \"Applikationen\" am Datenverteiler trat ein Problem auf. Bitte verwaltung=\"" +
                      _dataModel.getConfigurationAuthority().getPid() + "\" bei der Menge eintragen.");
    }

    /**
     * Löscht ein Applikationsobjekt
     *
     * @param applicationId Applikations-Id
     */
    public void removeApplication(final long applicationId) {
        final SystemObject applicationObject = _dataModel.getObject(applicationId);
        if (applicationObject != null && applicationObject instanceof ClientApplication) {
            try {
                applicationObject.invalidate();
                removeApplicationFromObjectSet(applicationObject);
            } catch (Exception e) {
                DEBUG.fine("Applikationsobjekt " + applicationObject + " konnte nicht nicht gelöscht werden", e);
            }
        }
    }
}
