/*
 * Copyright 2018-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.dav.
 *
 * de.bsvrz.dav.dav is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.dav is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.dav.dav.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.dav.main;

import de.bsvrz.dav.daf.communication.lowLevel.ConnectionInterface;
import de.bsvrz.dav.daf.communication.lowLevel.LowLevelCommunication;
import de.bsvrz.dav.daf.communication.lowLevel.ServerConnectionInterface;
import de.bsvrz.dav.daf.main.ConnectionException;
import de.bsvrz.dav.dav.communication.appProtocol.T_A_HighLevelCommunication;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Klasse, die eingehende App-Dav-Verbindungen verwaltet
 *
 * @author Kappich Systemberatung
 */
public class IncomingApplicationConnections extends AbstractConnections {

    private static final Debug _debug = Debug.getLogger();

    /**
     * Thread, der eingehende Verbindungen verarbeitet
     */
    private final ApplicationConnectionsSubscriber _applicationConnectionsSubscriber;

    /**
     * Die Serverkommunikationskomponente, die eingehende Verbindungen akzeptiert.
     */
    private final ServerConnectionInterface _applicationsServerConnection;

    private final HighLevelApplicationManager _applicationManager;

    /**
     * Verwaltung für alle offenen Verbindungen pro Applikation
     */
    private final ApplicationConnectionMap _connections;

    /**
     * Ist die Konfiguration verfügbar?
     */
    private boolean _configurationAvailable;

    /**
     * Konstruktor
     *
     * @param serverConnectionInterface  Die Serverkommunikationskomponente, die eingehende Verbindungen akzeptiert.
     * @param serverDavParameters        Server-Parameter
     * @param lowLevelConnectionsManager Low-Level-Verbindungsverwaltung
     * @param applicationManager         High-Level-Verwaltung für App-Dav-Verbindungen (Anmeldelisten usw.)
     * @param connectionMap              Verwaltung für alle offenen Verbindungen pro Datenverteiler
     */
    public IncomingApplicationConnections(final ServerConnectionInterface serverConnectionInterface, final ServerDavParameters serverDavParameters,
                                          final LowLevelConnectionsManagerInterface lowLevelConnectionsManager,
                                          final HighLevelApplicationManager applicationManager, final ApplicationConnectionMap connectionMap) {
        super(serverDavParameters, lowLevelConnectionsManager);
        _applicationManager = applicationManager;

        //  Netzwerkschnittstelle, die auf Verbindungen wartet
        _applicationsServerConnection = serverConnectionInterface;

        // Den Thread starten, der auf dieser Netzwerkschnittstelle auf ankommende Verbindungen wartet
        _applicationConnectionsSubscriber = new ApplicationConnectionsSubscriber();
        _connections = connectionMap;
    }

    public void start() {
        _applicationConnectionsSubscriber.start();
    }

    /**
     * Wird bei eingehender Verbindung ausgeführt
     *
     * @param connection Verbindung
     *
     * @throws ConnectionException
     */
    private T_A_HighLevelCommunication startApplicationConnection(final ConnectionInterface connection) throws ConnectionException {
        final LowLevelCommunication lowLevelCommunication = createLowLevelCommunication(connection, true);
        final ServerConnectionProperties properties = new ServerConnectionProperties(lowLevelCommunication, _serverDavParameters);

        T_A_HighLevelCommunication communication =
            new T_A_HighLevelCommunication(properties, _applicationManager, _lowLevelConnectionsManager, null, !_configurationAvailable);

        synchronized (_connections) {
            _connections.addUnsortedConnection(communication);
        }

        return communication;
    }

    /**
     * Trigger, dass die Konfiguration verfügbar ist.
     */
    public void setConfigurationAvailable() {
        _configurationAvailable = true;
    }

    /**
     * Terminiert die Server-Schnittstelle
     */
    public void close() {
        _applicationConnectionsSubscriber.interrupt();
        _applicationsServerConnection.disconnect();
    }

    @Override
    public String toString() {
        return "IncomingApplicationConnections{" + "_configurationAvailable=" + _configurationAvailable + '}';
    }

    /** Diese Subklasse startet einen Thread, der eine Applikation bei einem Datenverteiler anmeldet. */
    private class ApplicationConnectionsSubscriber extends Thread {

        protected ApplicationConnectionsSubscriber() {
            super("ApplicationConnectionsSubscriber");
        }

        @Override
        public final void run() {
            if (_applicationsServerConnection == null) {
                return;
            }
            while (!isInterrupted()) {
                final ConnectionInterface connection = _applicationsServerConnection.accept();
                final Runnable runnable = () -> {
                    if (connection == null) {
                        return;
                    }
                    try {
                        T_A_HighLevelCommunication communication = startApplicationConnection(connection);
                        _debug.fine("Neue eingehende Applikationsverbindung von " + communication.getRemoteAdress() + ":" +
                                    communication.getRemoteSubadress());
                    } catch (ConnectionException ex) {
                        _debug.warning("Fehler beim Aufbau einer eingehenden Applikationsverbindung von " + connection.getMainAdress() + ":" +
                                       connection.getSubAdressNumber(), ex);
                    }
                };
                final Thread thread = new Thread(runnable);
                thread.start();
            }
        }
    }
}
