/*
 * Copyright 2011-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.dav.
 *
 * de.bsvrz.dav.dav is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.dav is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.dav.dav.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.dav.main;

import de.bsvrz.dav.daf.communication.lowLevel.ParameterizedConnectionInterface;
import de.bsvrz.dav.daf.communication.lowLevel.ServerConnectionInterface;
import de.bsvrz.dav.daf.main.CommunicationError;
import de.bsvrz.dav.dav.communication.appProtocol.T_A_HighLevelCommunication;
import de.bsvrz.sys.funclib.debug.Debug;
import java.util.Collection;
import java.util.concurrent.TimeUnit;

/**
 * Diese Klasse verwaltet Applikations-Verbindung auf unterster Protokoll-Ebene. Es wird über das ServerConnectionInterface auf neue
 * Applikationsverbindungen gewartet und aus dieser wird eine neue T_A_HighLevelCommunication-Klasse erzeugt und gespeichert. Dabei werden
 * gegebenenfalls Verbindungsaufbau auf Protokollebene, Authentifizierung usw. durchgeführt.
 *
 * @author Kappich Systemberatung
 */
public final class LowLevelApplicationConnections {

    private static final Debug _debug = Debug.getLogger();
    /** Wiederverbindungswartezeit in Millisekunden */
    private final int _reconnectionDelay;
    private final HighLevelApplicationManager _applicationManager;
    private final LowLevelConnectionsManagerInterface _lowLevelConnectionsManager;
    private final ServerDavParameters _serverDavParameters;
    /** Hilfsklasse für die Zuordnung zwischen (aktiven) Verbindungen und konfigurierten Verbindungen */
    private final ApplicationConnectionMap _connectionMap;

    private IncomingApplicationConnections _incomingApplicationConnections;

    private OutgoingApplicationConnections _outgoingApplicationConnections;

    /**
     * Startet eine neue Klasse, die Applikations-Verbindung auf unterster Protokoll-Ebene entgegennimmt und verwaltet.
     *
     * @param lowLevelConnectionsManager Authentifizierungs-Modul
     * @param serverDavParameters        Server-Datenverteiler-Parameter
     */
    public LowLevelApplicationConnections(final HighLevelApplicationManager applicationManager,
                                          final LowLevelConnectionsManagerInterface lowLevelConnectionsManager,
                                          final ServerDavParameters serverDavParameters) {
        _applicationManager = applicationManager;
        _lowLevelConnectionsManager = lowLevelConnectionsManager;
        _serverDavParameters = serverDavParameters;
        long reconnectInterDavDelay = _serverDavParameters.getReconnectInterDavDelay();
        if (reconnectInterDavDelay < 0 || reconnectInterDavDelay > Integer.MAX_VALUE) {
            throw new IllegalArgumentException("Ungültige Wiederverbindungs-Wartezeit: " + reconnectInterDavDelay + "ms");
        }
        _reconnectionDelay = (int) reconnectInterDavDelay;
        _connectionMap = new ApplicationConnectionMap(_serverDavParameters.getActiveConnections());
    }

    /**
     * Startet den Aufbau der Dav-Dav-Verbindungen
     *
     * @param communicationProtocolClass Kommunikationsprotokoll-Klasse
     *
     * @throws InstantiationException Wenn das Kommunikationsprotokoll nicht erzeugt werden kann (benötigt öffentlichen, parameterlosen Konstruktor)
     * @throws IllegalAccessException Wenn das Kommunikationsprotokoll nicht erzeugt werden kann (benötigt öffentlichen, parameterlosen Konstruktor)
     * @throws CommunicationError     Wenn beim Aufbau der Serververbindung ein Fehler auftritt (z.B. Port bereits belegt)
     */
    public void startApplicationConnections(final Class<? extends ServerConnectionInterface> communicationProtocolClass)
        throws InstantiationException, IllegalAccessException, CommunicationError {
        //Start the listener at the application port
        final ServerConnectionInterface serverConnection;
        serverConnection = communicationProtocolClass.newInstance();
        // Falls vorhanden und möglich Parameter für das Kommunikationsinterface weitergeben
        final String communicationParameters = _serverDavParameters.getLowLevelCommunicationParameters();
	    if (!communicationParameters.isEmpty() && serverConnection instanceof ParameterizedConnectionInterface parameterizedConnection) {
            parameterizedConnection.setParameters(communicationParameters);
        }
        serverConnection.connect(_serverDavParameters.getApplicationConnectionsSubAddress());

        _incomingApplicationConnections =
            new IncomingApplicationConnections(serverConnection, _serverDavParameters, _lowLevelConnectionsManager, _applicationManager,
                                               _connectionMap);

        _outgoingApplicationConnections =
            new OutgoingApplicationConnections(serverConnection, _reconnectionDelay, _serverDavParameters, _lowLevelConnectionsManager,
                                               _applicationManager, _connectionMap);

        _incomingApplicationConnections.start();
        _outgoingApplicationConnections.start();
    }

    /**
     * Trigger, dass die Konfiguration verfügbar ist. Die Authentifizierung von Applikationen, die bisher gewartet haben, wird fortgesetzt.
     */
    public void continueAuthentication() {
        for (final T_A_HighLevelCommunication connection : getApplicationConnections()) {
            if (connection != null) {
                connection.continueAuthentication();
            }
        }
        _incomingApplicationConnections.setConfigurationAvailable();
        _outgoingApplicationConnections.setConfigurationAvailable();
    }

    /**
     * Wird aufgerufen, wenn die lokale Konfiguration erfolgreich verbunden ist, und ermöglicht der SelfClientDafConnection, mit der Initialisierung
     * fortzufahren (sodass diese dann nicht mehr auf die Konfiguration wartet)
     */
    public void localConfigurationAvailable() {
        final String davName = _lowLevelConnectionsManager.getClientDavParameters().getApplicationName();
        final String davType = _lowLevelConnectionsManager.getClientDavParameters().getApplicationTypePid();
        for (final T_A_HighLevelCommunication communication : getApplicationConnections()) {
            final String applicationTypePid = communication.getApplicationTypePid();
            final String applicationName = communication.getApplicationName();
            if ((applicationTypePid == null) || (applicationName == null)) {
                continue;
            }
            if (applicationTypePid.equals(davType) && applicationName.equals(davName)) {
                communication.continueAuthentication();
                break;
            }
        }
    }

    T_A_HighLevelCommunication getApplicationConnection(final long applicationId) {
        synchronized (_connectionMap) {
            return _connectionMap.getConnection(applicationId);
        }
    }

    /**
     * Beendet alle Applikationsverbindung und verhindert das Aufbauen von neuen Verbindungen
     *
     * @param error   Fehler Ja/Nein
     * @param message Fehlermeldung
     */
    public void close(final boolean error, final String message) {
        // Komponenten beenden, die neue Verbindungen starten
        _incomingApplicationConnections.close();
        _outgoingApplicationConnections.close();

        Collection<T_A_HighLevelCommunication> connections;
        synchronized (_connectionMap) {
            connections = _connectionMap.getAllConnections();

            // Map leeren
            _connectionMap.clear();
        }

        // Alle bisherigen Verbindungen terminieren
        for (final T_A_HighLevelCommunication applicationConnection : connections) {
            applicationConnection.terminate(error, message);
        }
    }

    /**
     * Wird aufgerufen, wenn eine Verbindung terminiert wurde
     *
     * @param applicationCommunication Terminierte verbindung
     */
    public void removeApplicationConnection(final T_A_HighLevelCommunication applicationCommunication) {
        synchronized (_connectionMap) {
            _connectionMap.removeConnection(applicationCommunication);
            ApplicationInfo info = applicationCommunication.getApplicationInfo();
            if (info != null) {
                // Ausgehende Verbindung wiederherstellen
                _outgoingApplicationConnections.scheduleApplicationConnect(info, _reconnectionDelay, TimeUnit.MILLISECONDS);
            }
        }
    }

    /**
     * Gibt alle Applikationsverbindungen zurück, die aktuell aufgebaut sind
     *
     * @return Alle aktiven Applikationen auf Low-Level-Ebene
     */
    public Collection<T_A_HighLevelCommunication> getApplicationConnections() {
        synchronized (_connectionMap) {
            return _connectionMap.getAllConnections();
        }
    }

    /**
     * Wird aufgerufen, wenn die ID einer Applikation bekannt wird, aktualisiert die entsprechenden Datenstrukturen in der {@link #_connectionMap}.
     *
     * @param communication Applikationsverbindung, deren ID bekannt geworden it (nach erfolgreichem Login)
     */
    public void updateId(final T_A_HighLevelCommunication communication) {
        synchronized (_connectionMap) {
            _connectionMap.putConnection(communication.getId(), communication);
        }
    }

    @Override
    public String toString() {
        synchronized (_connectionMap) {
            return "LowLevelApplicationConnections{" + "_connectionMap=" + _connectionMap + '}';
        }
    }
}
