/*
 * Copyright 2019-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.dav.
 *
 * de.bsvrz.dav.dav is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.dav is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.dav.dav.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.dav.util.accessControl;

import de.bsvrz.dav.daf.accessControl.internal.DafAccessControlManager;
import de.bsvrz.dav.daf.accessControl.internal.DataLoader;
import de.bsvrz.dav.daf.accessControl.internal.ExtendedUserInfo;
import de.bsvrz.dav.daf.accessControl.internal.UserInfoInternal;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.DataState;
import de.bsvrz.sys.funclib.operatingMessage.MessageGrade;
import de.bsvrz.sys.funclib.operatingMessage.MessageSender;
import de.bsvrz.sys.funclib.operatingMessage.MessageState;
import de.bsvrz.sys.funclib.operatingMessage.MessageType;
import de.bsvrz.sys.funclib.operatingMessage.OperatingMessage;
import de.bsvrz.sys.funclib.operatingMessage.PersistentOperatingMessage;

import java.util.*;

/**
 * Klasse, die im Datenmodell Abfragen nach Benutzerberechtigungen erlaubt.
 *
 * @author Kappich Systemberatung
 * @version $Revision: 0000 $
 */
public final class DavAccessControlManager extends DafAccessControlManager {

    /**
     * Interval zwischen 2 Betriebsmeldungen wegen fehlenden Parametern. Außerdem die Zeit, die mindestens vergangen sein muss, bis ein fehlender
     * Parameterdatensatz gemeldet wird. Bei der Anpassung der Zeit muss möglicherweise der Wortlaut der Betriebsmeldung geändert werden.
     */
    private static final int MessageSenderInterval = 60 * 1000;

    /**
     * Timer für Benachrichtigungen bei fehlenden parametern
     */
    private Timer _parameterTimer;

    private Map<DataState, Set<DataLoader>> _oldObjectsWithMissingParameters;

    private PersistentOperatingMessage _operatingMessage;

    /**
     * Erstellt eine neue Instanz des AccessControlManagers mit impliziter Benutzerverwaltung
     *
     * @param connection      Verbindung zum Datenverteiler
     * @param useNewDataModel Sollen die neuen Zugriffsrechte benutzt werden?
     */
    public DavAccessControlManager(final ClientDavInterface connection, final boolean useNewDataModel) {
        this(connection, true, useNewDataModel);
    }

    /**
     * Erstellt eine neue Instanz des AccessControlManagers
     *
     * @param connection                Verbindung zum Datenverteiler
     * @param useImplicitUserManagement Wenn false, werden nur Benutzer berücksichtigt, die mit addUser und removeUser in diese Klasse eingefügt
     *                                  werden.<br> Wenn true sind addUser und removeUser ohne Funktion und getUser ermittelt beliebige Benutzer,
     *                                  solange diese existieren.
     * @param useNewDataModel           Sollen die neuen Zugriffsrechte benutzt werden?
     */
    public DavAccessControlManager(final ClientDavInterface connection, final boolean useImplicitUserManagement, final boolean useNewDataModel) {
        super(useNewDataModel, connection, useImplicitUserManagement);
        createParameterTimer();
    }

    private static String formatMap(final Map<DataState, Set<DataLoader>> objectsWithMissingParameters) {
        final StringBuilder builder = new StringBuilder();
        for (final Map.Entry<DataState, Set<DataLoader>> entry : objectsWithMissingParameters.entrySet()) {
            if (entry.getValue().isEmpty()) {
                continue;
            }
            if (entry.getKey() == null) {
                builder.append("Kein Datensatz");
            } else {
                builder.append(entry.getKey());
            }
            builder.append(" (");
            builder.append(entry.getValue().size());
            if (entry.getValue().size() == 1) {
                builder.append(" Objekt):\n");
            } else {
                builder.append(" Objekte):\n");
            }
            for (final DataLoader loader : entry.getValue()) {
                builder.append("\t").append(loader.getSystemObject().getPidOrNameOrId()).append("\n");
            }
        }
        return builder.toString();
    }

    private static Map<DataState, Set<DataLoader>> getObjectsWithMissingParameters(final Collection<? extends DataLoader> values) {
        final Map<DataState, Set<DataLoader>> result = new LinkedHashMap<>();
        for (final DataLoader value : values) {
            if (value.getDataState() != DataState.DATA && value.getNoDataTime() > MessageSenderInterval) {
                result.computeIfAbsent(value.getDataState(), e -> new LinkedHashSet<>()).add(value);
            }
        }
        return result;
    }

    private void createParameterTimer() {
        _parameterTimer = new Timer("Warnung über fehlende Parameter", true);
        _parameterTimer.schedule(new TimerTask() {
            @Override
            public void run() {
                sendMessagesAboutMissingParameters();
            }
        }, MessageSenderInterval, MessageSenderInterval);
    }

    private void sendMessagesAboutMissingParameters() {
        final Map<DataState, Set<DataLoader>> objectsWithMissingParameters = new LinkedHashMap<>();
        final Collection<UserInfoInternal> values = _userInfoHashMap.values();
        final List<ExtendedUserInfo> users = new ArrayList<>(values.size());
        for (final UserInfoInternal value : values) {
            users.add((ExtendedUserInfo) value);
        }
        objectsWithMissingParameters.putAll(getObjectsWithMissingParameters(users));
        objectsWithMissingParameters.putAll(getObjectsWithMissingParameters(_authenticationClassHashMap.values()));
        objectsWithMissingParameters.putAll(getObjectsWithMissingParameters(_regionHashMap.values()));
        objectsWithMissingParameters.putAll(getObjectsWithMissingParameters(_roleHashMap.values()));
        if (_oldObjectsWithMissingParameters == null || _oldObjectsWithMissingParameters.isEmpty()) {
            if (objectsWithMissingParameters.isEmpty()) {
                return;
            }
            final String message = "Der Rechteprüfung fehlen Parameterdaten:\n" + formatMap(objectsWithMissingParameters);
            OperatingMessage operatingMessage = OperatingMessage.warning(MessageType.SYSTEM_DOMAIN, message);
            _operatingMessage = operatingMessage.newPersistentMessage("Zugriffsrechte");
        } else {
            if (objectsWithMissingParameters.isEmpty()) {
                _operatingMessage.setMessage("Alle derzeit berücksichtigten Objekte besitzen jetzt Parameter.");
                _operatingMessage.sendGoodMessage();
                _operatingMessage = null;
            } else {
                _operatingMessage.setMessage("Der Rechteprüfung fehlen Parameterdaten:\n" + formatMap(objectsWithMissingParameters));
                _operatingMessage.sendRepeatMessage();
            }
        }

        _oldObjectsWithMissingParameters = objectsWithMissingParameters;
    }

    @Override
    public String toString() {
        return "AccessControlManager{" + "_useImplicitUserManagement=" + _useImplicitUserManagement + '}';
    }

    /**
     * Löscht einen Benutzer aus der Benutzertabelle, wenn der Datenverteiler die Benutzerrechte prüfen soll. Wenn die interne Referenz eines
     * Benutzers 0 ist, dann wird die Benutzerinformation aus der Tabelle entfernt.
     *
     * @param userId BenutzerID
     */
    public void removeUser(final long userId) {
        if (_useImplicitUserManagement) {
            return;
        }
        _userMapLock.writeLock().lock();
        try {
            final UserInfoInternal user = _userInfoHashMap.get(userId);
            if (user != null) {
                user.decrementReference();
                if (user.canBeSafelyDeleted()) {
                    user.stopDataListener();
                    _userInfoHashMap.remove(userId);
                }
            }
        } finally {
            _userMapLock.writeLock().unlock();
        }
    }

    @Override
    public void close() {
        _parameterTimer.cancel();
        super.close();
    }

    @Override
    public void notifyInfiniteRecursion(final DataLoader node, final DataLoader parent, final List<DataLoader> trace) {
        super.notifyInfiniteRecursion(node, parent, trace);
        String msg = "Ungültige Rekursion in den Systemobjekten. Die problematische Vererbung wird deaktiviert bis das Problem behoben wird.\n" +
                     "Objekt referenziert sich selbst: " + node + "\n" + "Vererbungskette: " + trace;
        MessageSender.getInstance()
            .sendMessage("Zugriffsrechte", MessageType.SYSTEM_DOMAIN, "Rechteprüfung", MessageGrade.WARNING, MessageState.MESSAGE, msg);
    }
}
