/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.asbNodePlugin;

import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.kappich.pat.gnd.displayObjectToolkit.DOTCollection;
import de.kappich.pat.gnd.displayObjectToolkit.DisplayObject;
import de.kappich.pat.gnd.displayObjectToolkit.PrimitiveFormPropertyPair;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectPainter;
import de.kappich.pat.gnd.pluginInterfaces.DisplayObjectType;
import de.kappich.pat.gnd.utils.PointWithAngle;
import java.awt.Rectangle;
import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * {@code AsbNodeDisplayObject} ist die Implementation des Interfaces {@link DisplayObject} im ASB-Knoten-Plugin.
 *
 * @author Kappich Systemberatung
 */
public class AsbNodeDisplayObject implements DisplayObject {

    private final Map<Integer, Rectangle> _boundingRectangles = new HashMap<>();
    private final DOTCollection _dotCollection;
    private final String _asbNodeNumber;
    private final List<Object> _coordinates;
    private DisplayObjectPainter _painter;
    private int _defaultType;

    /**
     * Einziger Konstruktor.
     *
     * @param painter       ein DisplayObjectPainter
     * @param dotCollection eine DOTCollection
     * @param asbNodeNumber die ASB-Knotennummer
     * @param point         die Koordinaten plus Winkel
     */
    public AsbNodeDisplayObject(DisplayObjectPainter painter, final DOTCollection dotCollection, final String asbNodeNumber,
                                final PointWithAngle point) {
        super();
        _painter = painter;
        _dotCollection = dotCollection;
        _defaultType = 0;
        _asbNodeNumber = asbNodeNumber;
        _coordinates = new ArrayList<>(1);
        _coordinates.add(point);
    }

    @Override
    public String getName() {
        return _asbNodeNumber;
    }

    @Override
    public List<Object> getCoordinates(final int type) {
        return Collections.unmodifiableList(_coordinates);
    }

    @Override
    public List<Object> getCoordinates() {
        return Collections.unmodifiableList(_coordinates);
    }

    @Override
    public List<Point2D> getReferencePoints(final Rectangle utmBounds) {
        List<Point2D> pointList = new ArrayList<>(1);
        if (!_coordinates.isEmpty()) {
            Point2D point = ((PointWithAngle) _coordinates.get(0)).getPoint();
            if (utmBounds.contains(point)) {
                pointList.add(point);
            }
        }
        return pointList;
    }

    @Override
    public DisplayObjectPainter getPainter() {
        return new DOTAsbNodePainter();
    }

    @Override
    public void setPainter(DisplayObjectPainter painter) {
        _painter = painter;
    }

    @Override
    public DOTCollection getDOTCollection() {
        return _dotCollection;
    }

    @Override
    public Rectangle getBoundingRectangle(final int type) {
        if (!_boundingRectangles.containsKey(type)) {
            _boundingRectangles.put(type, _painter.getBoundingRectangle(this, type));
        }
        return _boundingRectangles.get(type);
    }

    @Override
    public Rectangle getBoundingRectangle() {
        return getBoundingRectangle(_defaultType);
    }

    @Override
    public void setDefaultType(final int defaultType) {
        _defaultType = defaultType;
    }

    @Override
    @Nullable
    public DisplayObjectType.DisplayObjectTypeItem getDisplayObjectTypeItem(final PrimitiveFormPropertyPair pair) {
        return null;
    }

    /*
     * Das ist die Methode, die für den MapPane.MapScaleListener implementiert werden muss. Sie kann hier
     * leer bleiben, da AsbNodeDisplayObjects davon keinen Gebrauch machen.
     */
    @Override
    public void mapScaleChanged(final double scale) {
    }

    /**
     * Gibt die ASB-Knotennummer als Text zurück.
     *
     * @return die ASB-Knotennummer
     */
    public String getAsbNodeNumber() {
        return _asbNodeNumber;
    }

    @Override
    public String toString() {
        return "AsbNodeDisplayObject{" + "_boundingRectangles=" + _boundingRectangles + ", _painter=" + _painter + ", _dotCollection=" +
               _dotCollection + ", _asbNodeNumber=" + _asbNodeNumber + ", _coordinates=" + _coordinates + ", _defaultType=" + _defaultType + '}';
    }
}
