/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.configBrowser;

import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.kappich.propertytree.PropertyTreeModel;
import de.bsvrz.sys.funclib.kappich.propertytree.PropertyTreeRenderer;
import de.kappich.pat.gnd.csvPlugin.CsvDisplayObject;
import de.kappich.pat.gnd.displayObjectToolkit.DisplayObject;
import de.kappich.pat.gnd.displayObjectToolkit.OnlineDisplayObject;
import de.kappich.pat.gnd.gnd.SelectionListener;
import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.StringSelection;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import javax.swing.ImageIcon;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JTree;
import javax.swing.SwingUtilities;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;

/**
 * {@code BrowserTree} ist JPanel, das es erlaubt die Informationen zu einer Menge von {@code SystemObjects} generisch in Baumform anzuzeigen.
 *
 * @author Kappich Systemberatung
 */
public class BrowserTree extends JTree implements SelectionListener {

    private static final int EXPAND_NODES_LIMIT = 7;
    @SuppressWarnings("FieldCanBeLocal")
    private static boolean LOG;

    /**
     * Die Klasse {@code BrowserTree} ist der Baum, der im Infofenster dargestellt wird.
     */
    public BrowserTree() {
        initTree();
    }

    private static List<Object> getListForSystemObjects(Collection<SystemObject> systemObjects) {
        List<Object> returnList = new ArrayList<>();
        if (systemObjects.size() > 1) {
            // zunächst sortieren nach Typ
            Map<String, List<ConfigObjectProperties>> map = new HashMap<>();
            int counter = 0;
            for (SystemObject systemObject : systemObjects) {
                String key = systemObject.getType().getName();
                if (map.containsKey(key)) {
                    map.get(key).add(ConfigObjectProperties.createConfigObjectProperties(systemObject));
                } else {
                    List<ConfigObjectProperties> newList = new ArrayList<>();
                    newList.add(ConfigObjectProperties.createConfigObjectProperties(systemObject));
                    map.put(key, newList);
                }
                ++counter;
                if (counter >= 50) {
                    break;
                }
            }
            for (final Map.Entry<String, List<ConfigObjectProperties>> stringListEntry : map.entrySet()) {
                Collections.sort(stringListEntry.getValue());
            }
            if (map.keySet().size() == 1) {
                Iterator<String> it = map.keySet().iterator();
                returnList.add(map.get(it.next()));
            } else {
                returnList.add(map);
            }
        } else if (systemObjects.size() == 1) {
            returnList.add(ConfigObjectProperties.createConfigObjectProperties(systemObjects.iterator().next()));
        }
        return returnList;
    }

    private static List<Object> getListForCsvObjects(Collection<CsvDisplayObject> csvObjects) {
        List<Object> returnList = new ArrayList<>();
        if (csvObjects.size() > 1) {
            int counter = 0;
            for (CsvDisplayObject csvObject : csvObjects) {
                returnList.add(CsvObjectProperties.createCsvObjectProperties(csvObject));
                ++counter;
                if (counter >= 50) {
                    break;
                }
            }
        } else if (csvObjects.size() == 1) {
            returnList.add(CsvObjectProperties.createCsvObjectProperties(csvObjects.iterator().next()));
        }
        return returnList;
    }

    private void initTree() {
//		setRowHeight(16); // brachte für die Performanz keinen durchschlagenden Erfolg!
        PropertyTreeModel model = new PropertyTreeModel(Collections.emptyList());
        model.setCollectionIcon(new ImageIcon(getClass().getResource("/de/kappich/pat/gnd/res/gruen.png")));
        model.setObjectIcon(new ImageIcon(getClass().getResource("/de/kappich/pat/gnd/res/blau.png")));
        model.setPropertyIcon(new ImageIcon(getClass().getResource("/de/kappich/pat/gnd/res/rot.png")));
        setModel(model);
        setRootVisible(false);
        setCellRenderer(new PropertyTreeRenderer());
        addMouseListener(new MyMouseListener());
    }

    @Override
    public void selectionChanged(final Collection<DisplayObject> objects) {
        List<Object> resultingList;
        boolean showRootNode;
        boolean expandSecondLevel;

        long startTime = System.currentTimeMillis();

        // Trennen von SystemObjects und CsvDisplayObjects:
        Collection<SystemObject> systemObjects = new ArrayList<>();
        Collection<CsvDisplayObject> csvObjects = new ArrayList<>();
        for (DisplayObject object : objects) {
            if (object instanceof OnlineDisplayObject) {
                systemObjects.add(((OnlineDisplayObject) object).getSystemObject());
            } else if (object instanceof CsvDisplayObject) {
                csvObjects.add((CsvDisplayObject) object);
            }
        }

        // Füllen der resultingList
        if (csvObjects.isEmpty()) {
            resultingList = getListForSystemObjects(systemObjects);
            expandSecondLevel = systemObjects.size() < EXPAND_NODES_LIMIT;
            showRootNode = systemObjects.size() == 1;
        } else if (systemObjects.isEmpty()) {
            resultingList = getListForCsvObjects(csvObjects);
            expandSecondLevel = resultingList.size() < EXPAND_NODES_LIMIT;
            showRootNode = csvObjects.size() == 1;
        } else { // beide Listen sind nicht leer
            resultingList = getListForSystemObjects(systemObjects);
            if (resultingList.size() < 50) {
                List<Object> otherList = getListForCsvObjects(csvObjects);
                int counter = resultingList.size();
                for (Object o : otherList) {
                    resultingList.add(o);
                    ++counter;
                    if (counter > 50) {
                        break;
                    }
                }
            }
            expandSecondLevel = resultingList.size() < EXPAND_NODES_LIMIT;
            showRootNode = csvObjects.size() == 1;
        }

        // Tree fertigstellen:
        if (LOG) {
            System.out.println("ErgebnisListe erstellen: " + (System.currentTimeMillis() - startTime));
            startTime = System.currentTimeMillis();
        }
        PropertyTreeModel model = new PropertyTreeModel(resultingList, false);
        model.setCollectionIcon(new ImageIcon(getClass().getResource("/de/kappich/pat/gnd/res/gruen.png")));
        model.setObjectIcon(new ImageIcon(getClass().getResource("/de/kappich/pat/gnd/res/blau.png")));
        model.setPropertyIcon(new ImageIcon(getClass().getResource("/de/kappich/pat/gnd/res/rot.png")));
        if (LOG) {
            System.out.println("PropertyTreeModel konstruiert: " + (System.currentTimeMillis() - startTime));
            startTime = System.currentTimeMillis();
        }
        setModel(model);
        if (LOG) {
            System.out.println("setModel ausführen: " + (System.currentTimeMillis() - startTime));
            startTime = System.currentTimeMillis();
        }
        setRootVisible(showRootNode);
        // Macht man den Root-Knoten allgemein sichtbar, so zeigt sich ein Nachteil: dieser Knoten ist
        // nicht durch einen selbst-definierten PropertyFormatter erreichbar.
        setLargeModel(true);
        if (LOG) {
            System.out.println("setRootVisible/setLargeModel: " + (System.currentTimeMillis() - startTime));
        }
        if (getRowCount() < EXPAND_NODES_LIMIT) {
            for (int i = getRowCount() - 1; i >= 0; --i) {
                expandRow(i);
            }
            if (expandSecondLevel) {
                for (int i = getRowCount() - 1; i >= 0; --i) {
                    expandRow(i);
                }
            }
        }
        repaint();
    }

    private static class MyMouseListener extends MouseAdapter {
        private final JPopupMenu _popup = new JPopupMenu();
        private final JMenuItem _copyIds = new JMenuItem("IDs kopieren");
        private final JMenuItem _copyPids = new JMenuItem("Pids kopieren");
        private final JMenuItem _copyNames = new JMenuItem("Namen kopieren");
        private final JMenuItem _asText = new JMenuItem("Als Text kopieren");

        MyMouseListener() {
            _popup.add(_copyIds);
            _popup.add(_copyPids);
            _popup.add(_copyNames);
            _popup.addSeparator();
            _popup.add(_asText);
        }

        void updateSelection(MouseEvent e) {
            final Object obj = e.getSource();
	        if (obj instanceof JTree tree) {
                TreePath path = tree.getPathForLocation(e.getX(), e.getY());
                if (!tree.isPathSelected(path)) {
                    tree.setSelectionPath(path);
                }
            }
        }

        List<SystemObject> getSelectedSystemObjects(MouseEvent e) {
            List<SystemObject> list = new ArrayList<>();

            final Object obj = e.getSource();
	        if (obj instanceof JTree tree) {
                TreeSelectionModel model = tree.getSelectionModel();
                TreePath[] paths = model.getSelectionPaths();
                for (TreePath path : paths) {
                    Object o = path.getLastPathComponent();
	                if (o instanceof PropertyTreeModel.ObjectNode objectNode) {
                        Object theObject = objectNode.getObject();
		                if (theObject instanceof ConfigObjectProperties propertyObject) {
                            list.add(propertyObject.getSystemObject());
                        }
                    }
                }
            }
            return list;
        }

        @Override
        public void mousePressed(MouseEvent e) {
            if (!SwingUtilities.isRightMouseButton(e)) {
                return;
            }
            updateSelection(e);
            List<SystemObject> systemObjects = getSelectedSystemObjects(e);
            if (systemObjects.isEmpty()) {
                _copyIds.setEnabled(false);
                _copyPids.setEnabled(false);
                _copyNames.setEnabled(false);
            } else {
                _copyIds.setEnabled(true);
                _copyPids.setEnabled(true);
                _copyNames.setEnabled(true);
                if (systemObjects.size() == 1) {
                    _copyIds.setText("ID kopieren");
                    _copyPids.setText("Pid kopieren");
                    _copyNames.setText("Name kopieren");
                } else {
                    _copyIds.setText("IDs kopieren");
                    _copyPids.setText("Pids kopieren");
                    _copyNames.setText("Namen kopieren");
                }
                setActionListenerIds(systemObjects);
                setActionListenerPids(systemObjects);
                setActionListenerNames(systemObjects);
            }
            setActionListenerText(e);
            _popup.show(e.getComponent(), e.getX(), e.getY());
        }

        private void setActionListenerText(final MouseEvent e) {
            ActionListener[] als = _asText.getActionListeners();
            for (ActionListener al : als) {
                _asText.removeActionListener(al);
            }
            _asText.addActionListener(ae -> {
                final Object obj = e.getSource();
	            if (obj instanceof JTree tree) {
                    TreeSelectionModel model = tree.getSelectionModel();
                    TreePath[] paths = model.getSelectionPaths();
                    if (null != paths && paths.length > 0) {
                        StringBuilder sb = new StringBuilder();
                        boolean firstOne = true;
                        for (TreePath path : paths) {
                            if (firstOne) {
                                firstOne = false;
                            } else {
                                sb.append(System.lineSeparator());
                            }
                            sb.append(path.getLastPathComponent().toString());
                        }
                        Clipboard cb = Toolkit.getDefaultToolkit().getSystemClipboard();
                        cb.setContents(new StringSelection(sb.toString()), null);
                    }
                }
            });
        }

        private void setActionListenerIds(List<SystemObject> systemObjects) {
            ActionListener[] als = _copyIds.getActionListeners();
            for (ActionListener al : als) {
                _copyIds.removeActionListener(al);
            }
            _copyIds.addActionListener(e -> {
                StringBuilder sb = new StringBuilder();
                boolean firstOne = true;
                for (SystemObject systemObject : systemObjects) {
                    if (firstOne) {
                        firstOne = false;
                    } else {
                        sb.append(System.lineSeparator());
                    }
                    sb.append(systemObject.getId());
                }
                Clipboard cb = Toolkit.getDefaultToolkit().getSystemClipboard();
                cb.setContents(new StringSelection(sb.toString()), null);
            });
        }

        private void setActionListenerPids(List<SystemObject> systemObjects) {
            ActionListener[] als = _copyPids.getActionListeners();
            for (ActionListener al : als) {
                _copyPids.removeActionListener(al);
            }
            _copyPids.addActionListener(e -> {
                StringBuilder sb = new StringBuilder();
                boolean firstOne = true;
                for (SystemObject systemObject : systemObjects) {
                    if (firstOne) {
                        firstOne = false;
                    } else {
                        sb.append(System.lineSeparator());
                    }
                    sb.append(systemObject.getPid());
                }
                Clipboard cb = Toolkit.getDefaultToolkit().getSystemClipboard();
                cb.setContents(new StringSelection(sb.toString()), null);
            });
        }

        private void setActionListenerNames(List<SystemObject> systemObjects) {
            ActionListener[] als = _copyNames.getActionListeners();
            for (ActionListener al : als) {
                _copyNames.removeActionListener(al);
            }
            _copyNames.addActionListener(e -> {
                StringBuilder sb = new StringBuilder();
                boolean firstOne = true;
                for (SystemObject systemObject : systemObjects) {
                    if (firstOne) {
                        firstOne = false;
                    } else {
                        sb.append(System.lineSeparator());
                    }
                    sb.append(systemObject.getName());
                }
                Clipboard cb = Toolkit.getDefaultToolkit().getSystemClipboard();
                cb.setContents(new StringSelection(sb.toString()), null);
            });
        }
    }
}
