/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.gnd.
 *
 * de.kappich.pat.gnd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.gnd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.gnd.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.gnd.configBrowser;

import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.AttributeGroupUsage;
import de.bsvrz.dav.daf.main.config.ConfigurationObject;
import de.bsvrz.dav.daf.main.config.ObjectSet;
import de.bsvrz.dav.daf.main.config.ReferenceAttributeType;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.util.HashBagMap;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.bsvrz.sys.funclib.kappich.properties.PropertyClass;
import de.bsvrz.sys.funclib.kappich.properties.PropertyName;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * {@code ConfigObjectProperties} ist eine Klasse, die rekursiv alle Informationen über ein {@code SystemObject}  zusammenstellt.
 *
 * @author Kappich Systemberatung
 */
@PropertyClass
public class ConfigObjectProperties implements Comparable<ConfigObjectProperties> {

    final SystemObject _systemObject;

    /*
     * Die Klasse <code>ConfigObjectProperties</code> ist eine <code>PropertyClass</code>,
     * die bestimmt wie die Properties eines nicht weiter spezifizierten <code>SystemObjects</code>
     * aussehen.
     *
     * @param systemObject ein <code>SystemObject</code>
     */
    protected ConfigObjectProperties(@NotNull final SystemObject systemObject) {
        _systemObject = systemObject;
    }

    /**
     * Diese Methode liefert falls möglich ein spezifisches, andernfalls ein unspezifisches Objekt zurück
     *
     * @param systemObject ein SystemObject
     *
     * @return ein ConfigObjectProperties-Objekt
     */
    public static ConfigObjectProperties createConfigObjectProperties(@NotNull final SystemObject systemObject) {
        if (systemObject.getType().getPid().equals("typ.inneresStraßenSegment")) {
            return new InnerSegmentProperties(systemObject);
        } else if (systemObject.getType().getPid().equals("typ.äußeresStraßenSegment")) {
            return new OuterSegmentProperties(systemObject);
        } else if (systemObject.getType().getPid().equals("typ.netz")) {
            return new NetProperties(systemObject);
        } else if (systemObject.getType().getPid().equals("typ.straßenTeilSegment")) {
            return new PartialSegmentProperties(systemObject);
        } else if (systemObject.getType().getPid().equals("typ.straße")) {
            return new StreetProperties(systemObject);
        } else {
            return new ConfigObjectProperties(systemObject);
        }
    }

    @Nullable
    private static Object getPropertiesData(final Data data) {
        if (data.isPlain()) {
            Data.ReferenceValue referenceValue = data.asReferenceValue();
            SystemObject systemObject = referenceValue.getSystemObject();
            if (null != systemObject) {
                return createConfigObjectProperties(systemObject);
            } else {
                return null;
            }
        } else if (data.isArray()) {
            List<Object> list = new ArrayList<>();
            for (Data subData : data) {
                if (subData.isPlain()) {
                    if (subData.getAttributeType() instanceof ReferenceAttributeType) {
                        Data.ReferenceValue referenceValue = subData.asReferenceValue();
                        SystemObject systemObject = referenceValue.getSystemObject();
                        if (null != systemObject) {
                            list.add(createConfigObjectProperties(systemObject));
                        } else {
                            list.add(null);
                        }
                    } else {
                        list.add(subData.valueToString());
                    }
                } else if (subData.isArray() || subData.isList()) {
                    list.add(getPropertiesData(subData));
                }
            }
            return list;
        } else if (data.isList()) {
            Map<String, Object> map = new LinkedHashMap<>();
            for (Data subData : data) {
                if (subData.isPlain()) {
                    if (subData.getAttributeType() instanceof ReferenceAttributeType) {
                        Data.ReferenceValue referenceValue = subData.asReferenceValue();
                        SystemObject systemObject = referenceValue.getSystemObject();
                        if (null != systemObject) {
                            map.put(subData.getName(), createConfigObjectProperties(systemObject));
                        } else {
                            map.put(subData.getName(), null);
                        }
                    } else {
                        map.put(subData.getName(), subData.valueToString());
                    }
                } else if (subData.isArray() || subData.isList()) {
                    map.put(subData.getName(), getPropertiesData(subData));
                }
            }
            return map;
        } else {
            return null;
        }
    }

    /**
     * Der Getter des {@code SystemObjects}.
     *
     * @return das {@code SystemObject}
     */
    public SystemObject getSystemObject() {
        return _systemObject;
    }

    /**
     * Diese Methode bestimmt, wie der Typ dargestellt wird.
     *
     * @return ein Objekt
     */
    @NotNull
    @PropertyName(name = "Typ", sortKey = 0)
    public Object getConfigurationObjectType() {
        if (_systemObject.getType().getName().equals("Typ")) {
            return _systemObject.getType();
        } else {
            return createConfigObjectProperties(_systemObject.getType());
        }
    }

    /**
     * Diese Methode bestimmt, wie der Name dargestellt wird.
     *
     * @return ein String
     */
    @NotNull
    @PropertyName(name = "Name", sortKey = 1)
    public String getName() {
        return _systemObject.getName();
    }

    /**
     * Diese Methode bestimmt, wie die Pid dargestellt wird.
     *
     * @return ein String
     */
    @NotNull
    @PropertyName(name = "Pid", sortKey = 2)
    public String getPid() {
        return _systemObject.getPid();
    }

    /**
     * Diese Methode bestimmt, welche Konfigurationsdaten dargestellt werden.
     *
     * @return eine LinkedHashMap
     */
    @PropertyName(name = "Konfigurationsdaten", sortKey = 3)
    public LinkedHashMap<String, Object> getElementProperties() {
        LinkedHashMap<String, Object> res = new LinkedHashMap<>();
        Collection<AttributeGroupUsage> atgUsages = _systemObject.getUsedAttributeGroupUsages();
        for (AttributeGroupUsage atgUsage : atgUsages) {
            Data data = _systemObject.getConfigurationData(atgUsage);
            Object propertiesData = getPropertiesData(data);
            AttributeGroup atg = atgUsage.getAttributeGroup();
            Aspect asp = atgUsage.getAspect();
            if (asp.getName().equals("Eigenschaften")) {
                res.put(atg.getName(), propertiesData);
            } else {
                res.put(atg.getName() + "/" + asp.getName(), propertiesData);
            }
        }
        return res;
    }

    /**
     * Diese Methode bestimmt, wie die Mengen dargestellt werden.
     *
     * @return eine HashBagMap
     */
    @PropertyName(name = "Mengen", sortKey = 4, getFormatter = BrowserTreeFormatter.class)
    public HashBagMap<String, ConfigObjectProperties> getObjectSets() {
        HashBagMap<String, ConfigObjectProperties> res = new HashBagMap<>();
	    if (_systemObject instanceof ConfigurationObject configurationObject) {
            List<ObjectSet> objectSets = configurationObject.getObjectSets();
            for (ObjectSet objectSet : objectSets) {
                List<ConfigObjectProperties> configObjProps = new ArrayList<>();
                for (SystemObject systemObject : objectSet.getElements()) {
                    configObjProps.add(createConfigObjectProperties(systemObject));
                }
                res.addAll(objectSet.getName(), configObjProps);
            }
        }
        return res;
    }

    @Override
    public final String toString() {
        return _systemObject.getName() + " (" + _systemObject.getPidOrId() + ")";
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
	    if (!(o instanceof ConfigObjectProperties that)) {
            return false;
        }

        return _systemObject.equals(that._systemObject);
    }

    @Override
    public int hashCode() {
        return _systemObject.hashCode();
    }

    @Override
    public int compareTo(final ConfigObjectProperties o) {
        return _systemObject.getName().compareTo(o._systemObject.getName());
    }
}
